/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.http;

import io.vertx.reactivex.RxHelper;
import io.vertx.reactivex.ObservableHelper;
import io.vertx.reactivex.FlowableHelper;
import io.vertx.reactivex.impl.AsyncResultMaybe;
import io.vertx.reactivex.impl.AsyncResultSingle;
import io.vertx.reactivex.impl.AsyncResultCompletable;
import io.vertx.reactivex.WriteStreamObserver;
import io.vertx.reactivex.WriteStreamSubscriber;
import java.util.Map;
import java.util.Set;
import java.util.List;
import java.util.Iterator;
import java.util.function.Function;
import java.util.stream.Collectors;
import io.vertx.core.Handler;
import io.vertx.core.AsyncResult;
import io.vertx.core.json.JsonObject;
import io.vertx.core.json.JsonArray;
import io.vertx.lang.rx.RxGen;
import io.vertx.lang.rx.TypeArg;
import io.vertx.lang.rx.MappingIterator;

/**
 * Represents a server-side HTTP response.
 * <p>
 * An instance of this is created and associated to every instance of
 * {@link io.vertx.reactivex.core.http.HttpServerRequest} that.
 * <p>
 * It allows the developer to control the HTTP response that is sent back to the
 * client for a particular HTTP request.
 * <p>
 * It contains methods that allow HTTP headers and trailers to be set, and for a body to be written out to the response.
 * <p>
 * It also allows files to be streamed by the kernel directly from disk to the
 * outgoing HTTP connection, bypassing user space altogether (where supported by
 * the underlying operating system). This is a very efficient way of
 * serving files from the server since buffers do not have to be read one by one
 * from the file and written to the outgoing socket.
 * <p>
 * It implements {@link io.vertx.reactivex.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.reactivex.core.streams.Pipe} to pipe data with flow control.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpServerResponse original} non RX-ified interface using Vert.x codegen.
 */

@RxGen(io.vertx.core.http.HttpServerResponse.class)
public class HttpServerResponse implements io.vertx.reactivex.core.streams.WriteStream<io.vertx.reactivex.core.buffer.Buffer> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpServerResponse that = (HttpServerResponse) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final TypeArg<HttpServerResponse> __TYPE_ARG = new TypeArg<>(    obj -> new HttpServerResponse((io.vertx.core.http.HttpServerResponse) obj),
    HttpServerResponse::getDelegate
  );

  private final io.vertx.core.http.HttpServerResponse delegate;
  
  public HttpServerResponse(io.vertx.core.http.HttpServerResponse delegate) {
    this.delegate = delegate;
  }

  public HttpServerResponse(Object delegate) {
    this.delegate = (io.vertx.core.http.HttpServerResponse)delegate;
  }

  public io.vertx.core.http.HttpServerResponse getDelegate() {
    return delegate;
  }

  private WriteStreamObserver<io.vertx.reactivex.core.buffer.Buffer> observer;
  private WriteStreamSubscriber<io.vertx.reactivex.core.buffer.Buffer> subscriber;

  public synchronized WriteStreamObserver<io.vertx.reactivex.core.buffer.Buffer> toObserver() {
    if (observer == null) {
      Function<io.vertx.reactivex.core.buffer.Buffer, io.vertx.core.buffer.Buffer> conv = io.vertx.reactivex.core.buffer.Buffer::getDelegate;
      observer = RxHelper.toObserver(getDelegate(), conv);
    }
    return observer;
  }

  public synchronized WriteStreamSubscriber<io.vertx.reactivex.core.buffer.Buffer> toSubscriber() {
    if (subscriber == null) {
      Function<io.vertx.reactivex.core.buffer.Buffer, io.vertx.core.buffer.Buffer> conv = io.vertx.reactivex.core.buffer.Buffer::getDelegate;
      subscriber = RxHelper.toSubscriber(getDelegate(), conv);
    }
    return subscriber;
  }

  private static final TypeArg<io.vertx.reactivex.core.http.Cookie> TYPE_ARG_0 = new TypeArg<io.vertx.reactivex.core.http.Cookie>(o1 -> io.vertx.reactivex.core.http.Cookie.newInstance((io.vertx.core.http.Cookie)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.Cookie> TYPE_ARG_1 = new TypeArg<io.vertx.reactivex.core.http.Cookie>(o1 -> io.vertx.reactivex.core.http.Cookie.newInstance((io.vertx.core.http.Cookie)o1), o1 -> o1.getDelegate());

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   * @param handler 
   */
  public void write(io.vertx.reactivex.core.buffer.Buffer data, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.write(data.getDelegate(), handler);
  }

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   */
  public void write(io.vertx.reactivex.core.buffer.Buffer data) {
    write(data, ar -> { });
  }

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   * @return 
   */
  public io.reactivex.Completable rxWrite(io.vertx.reactivex.core.buffer.Buffer data) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      write(data, $handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param handler 
   */
  public void end(io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.end(handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   */
  public void end() {
    end(ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @return 
   */
  public io.reactivex.Completable rxEnd() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      end($handler);
    });
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.reactivex.core.http.HttpServerResponse#setWriteQueueMaxSize}
   * @return <code>true</code> if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  public io.vertx.reactivex.core.http.HttpServerResponse exceptionHandler(io.vertx.core.Handler<java.lang.Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpServerResponse setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpServerResponse drainHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  /**
   * @return the HTTP status code of the response. The default is <code>200</code> representing <code>OK</code>.
   */
  public int getStatusCode() { 
    int ret = delegate.getStatusCode();
    return ret;
  }

  /**
   * Set the status code. If the status message hasn't been explicitly set, a default status message corresponding
   * to the code will be looked-up and used.
   * @param statusCode 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse setStatusCode(int statusCode) { 
    delegate.setStatusCode(statusCode);
    return this;
  }

  /**
   * @return the HTTP status message of the response. If this is not specified a default value will be used depending on what {@link io.vertx.reactivex.core.http.HttpServerResponse#setStatusCode} has been set to.
   */
  public java.lang.String getStatusMessage() { 
    java.lang.String ret = delegate.getStatusMessage();
    return ret;
  }

  /**
   * Set the status message
   * @param statusMessage 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse setStatusMessage(java.lang.String statusMessage) { 
    delegate.setStatusMessage(statusMessage);
    return this;
  }

  /**
   * If <code>chunked</code> is <code>true</code>, this response will use HTTP chunked encoding, and each call to write to the body
   * will correspond to a new HTTP chunk sent on the wire.
   * <p>
   * If chunked encoding is used the HTTP header <code>Transfer-Encoding</code> with a value of <code>Chunked</code> will be
   * automatically inserted in the response.
   * <p>
   * If <code>chunked</code> is <code>false</code>, this response will not use HTTP chunked encoding, and therefore the total size
   * of any data that is written in the respone body must be set in the <code>Content-Length</code> header <b>before</b> any
   * data is written out.
   * <p>
   * An HTTP chunked response is typically used when you do not know the total size of the request body up front.
   * @param chunked 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return is the response chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public io.vertx.reactivex.core.MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    io.vertx.reactivex.core.MultiMap ret = io.vertx.reactivex.core.MultiMap.newInstance((io.vertx.core.MultiMap)delegate.headers());
    cached_0 = ret;
    return ret;
  }

  /**
   * Put an HTTP header
   * @param name the header name
   * @param value the header value.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putHeader(java.lang.String name, java.lang.String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * @return The HTTP trailers
   */
  public io.vertx.reactivex.core.MultiMap trailers() { 
    if (cached_1 != null) {
      return cached_1;
    }
    io.vertx.reactivex.core.MultiMap ret = io.vertx.reactivex.core.MultiMap.newInstance((io.vertx.core.MultiMap)delegate.trailers());
    cached_1 = ret;
    return ret;
  }

  /**
   * Put an HTTP trailer
   * @param name the trailer name
   * @param value the trailer value
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putTrailer(java.lang.String name, java.lang.String value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  /**
   * Set a close handler for the response, this is called when the underlying connection is closed and the response
   * was still using the connection.
   * <p>
   * For HTTP/1.x it is called when the connection is closed before <code>end()</code> is called, therefore it is not
   * guaranteed to be called.
   * <p>
   * For HTTP/2 it is called when the related stream is closed, and therefore it will be always be called.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse closeHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.closeHandler(handler);
    return this;
  }

  /**
   * Set an end handler for the response. This will be called when the response is disposed to allow consistent cleanup
   * of the response.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse endHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.endHandler(handler);
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @param handler 
   */
  public void write(java.lang.String chunk, java.lang.String enc, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.write(chunk, enc, handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   */
  public void write(java.lang.String chunk, java.lang.String enc) {
    write(chunk, enc, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @return 
   */
  public io.reactivex.Completable rxWrite(java.lang.String chunk, java.lang.String enc) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      write(chunk, enc, $handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void write(java.lang.String chunk, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.write(chunk, handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void write(java.lang.String chunk) {
    write(chunk, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public io.reactivex.Completable rxWrite(java.lang.String chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      write(chunk, $handler);
    });
  }

  /**
   * Used to write an interim 100 Continue response to signify that the client should send the rest of the request.
   * Must only be used if the request contains an "Expect:100-Continue" header
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse writeContinue() { 
    delegate.writeContinue();
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void end(java.lang.String chunk, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.end(chunk, handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void end(java.lang.String chunk) {
    end(chunk, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public io.reactivex.Completable rxEnd(java.lang.String chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      end(chunk, $handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @param handler 
   */
  public void end(java.lang.String chunk, java.lang.String enc, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.end(chunk, enc, handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   */
  public void end(java.lang.String chunk, java.lang.String enc) {
    end(chunk, enc, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @return 
   */
  public io.reactivex.Completable rxEnd(java.lang.String chunk, java.lang.String enc) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      end(chunk, enc, $handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void end(io.vertx.reactivex.core.buffer.Buffer chunk, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.end(chunk.getDelegate(), handler);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void end(io.vertx.reactivex.core.buffer.Buffer chunk) {
    end(chunk, ar -> { });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public io.reactivex.Completable rxEnd(io.vertx.reactivex.core.buffer.Buffer chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      end(chunk, $handler);
    });
  }

  /**
   * Send the request with an empty body.
   * @param handler the completion handler
   */
  public void send(io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.send(handler);
  }

  /**
   * Send the request with an empty body.
   */
  public void send() {
    send(ar -> { });
  }

  /**
   * Send the request with an empty body.
   * @return 
   */
  public io.reactivex.Completable rxSend() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      send($handler);
    });
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   * @param handler the completion handler
   */
  public void send(java.lang.String body, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.send(body, handler);
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   */
  public void send(java.lang.String body) {
    send(body, ar -> { });
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   * @return 
   */
  public io.reactivex.Completable rxSend(java.lang.String body) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      send(body, $handler);
    });
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   * @param handler the completion handler
   */
  public void send(io.vertx.reactivex.core.buffer.Buffer body, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.send(body.getDelegate(), handler);
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   */
  public void send(io.vertx.reactivex.core.buffer.Buffer body) {
    send(body, ar -> { });
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   * @return 
   */
  public io.reactivex.Completable rxSend(io.vertx.reactivex.core.buffer.Buffer body) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      send(body, $handler);
    });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @param handler the completion handler
   */
  public void send(io.vertx.reactivex.core.streams.ReadStream<io.vertx.reactivex.core.buffer.Buffer> body, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.send(body.getDelegate(), handler);
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   */
  public void send(io.vertx.reactivex.core.streams.ReadStream<io.vertx.reactivex.core.buffer.Buffer> body) {
    send(body, ar -> { });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return 
   */
  public io.reactivex.Completable rxSend(io.vertx.reactivex.core.streams.ReadStream<io.vertx.reactivex.core.buffer.Buffer> body) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      send(body, $handler);
    });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @param handler the completion handler
   */
  public void send(io.reactivex.Flowable<io.vertx.reactivex.core.buffer.Buffer> body, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.send(io.vertx.reactivex.impl.ReadStreamSubscriber.asReadStream(body, obj -> obj.getDelegate()).resume(), handler);
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   */
  public void send(io.reactivex.Flowable<io.vertx.reactivex.core.buffer.Buffer> body) {
    send(body, ar -> { });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return 
   */
  public io.reactivex.Completable rxSend(io.reactivex.Flowable<io.vertx.reactivex.core.buffer.Buffer> body) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      send(body, $handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse sendFile(java.lang.String filename, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> resultHandler) { 
    delegate.sendFile(filename, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse sendFile(java.lang.String filename) {
    return 
sendFile(filename, ar -> { });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @return a reference to this, so the API can be used fluently
   */
  public io.reactivex.Completable rxSendFile(java.lang.String filename) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      sendFile(filename, $handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse sendFile(java.lang.String filename, long offset, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> resultHandler) { 
    delegate.sendFile(filename, offset, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse sendFile(java.lang.String filename, long offset) {
    return 
sendFile(filename, offset, ar -> { });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been completely
   * written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @return a reference to this, so the API can be used fluently
   */
  public io.reactivex.Completable rxSendFile(java.lang.String filename, long offset) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      sendFile(filename, offset, $handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been
   * completely written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param length length the number of bytes to send
   * @param resultHandler handler that will be called on completion
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse sendFile(java.lang.String filename, long offset, long length, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> resultHandler) { 
    delegate.sendFile(filename, offset, length, resultHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been
   * completely written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param length length the number of bytes to send
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse sendFile(java.lang.String filename, long offset, long length) {
    return 
sendFile(filename, offset, length, ar -> { });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} but providing a handler which will be notified once the file has been
   * completely written to the wire.
   * @param filename path to the file to serve
   * @param offset the offset to serve from
   * @param length length the number of bytes to send
   * @return a reference to this, so the API can be used fluently
   */
  public io.reactivex.Completable rxSendFile(java.lang.String filename, long offset, long length) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      sendFile(filename, offset, length, $handler);
    });
  }

  /**
   * Close the underlying TCP connection corresponding to the request.
   */
  public void close() { 
    delegate.close();
  }

  /**
   * @return has the response already ended?
   */
  public boolean ended() { 
    boolean ret = delegate.ended();
    return ret;
  }

  /**
   * @return has the underlying TCP connection corresponding to the request already been closed?
   */
  public boolean closed() { 
    boolean ret = delegate.closed();
    return ret;
  }

  /**
   * @return have the headers for the response already been written?
   */
  public boolean headWritten() { 
    boolean ret = delegate.headWritten();
    return ret;
  }

  /**
   * Provide a handler that will be called just before the headers are written to the wire.<p>
   * This provides a hook allowing you to add any more headers or do any more operations before this occurs.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse headersEndHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.headersEndHandler(handler);
    return this;
  }

  /**
   * Provides a handler that will be called after the last part of the body is written to the wire.
   * The handler is called asynchronously of when the response has been received by the client.
   * This provides a hook allowing you to do more operations once the request has been sent over the wire
   * such as resource cleanup.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse bodyEndHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.bodyEndHandler(handler);
    return this;
  }

  /**
   * @return the total number of bytes written for the body of the response.
   */
  public long bytesWritten() { 
    long ret = delegate.bytesWritten();
    return ret;
  }

  /**
   * @return the id of the stream of this response,  for HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with no headers.
   * @param method 
   * @param host 
   * @param path 
   * @param handler 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse push(io.vertx.core.http.HttpMethod method, java.lang.String host, java.lang.String path, io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.reactivex.core.http.HttpServerResponse>> handler) { 
    delegate.push(method, host, path, new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with no headers.
   * @param method 
   * @param host 
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse push(io.vertx.core.http.HttpMethod method, java.lang.String host, java.lang.String path) {
    return 
push(method, host, path, ar -> { });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with no headers.
   * @param method 
   * @param host 
   * @param path 
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpServerResponse> rxPush(io.vertx.core.http.HttpMethod method, java.lang.String host, java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      push(method, host, path, $handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param headers 
   * @param handler 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse push(io.vertx.core.http.HttpMethod method, java.lang.String path, io.vertx.reactivex.core.MultiMap headers, io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.reactivex.core.http.HttpServerResponse>> handler) { 
    delegate.push(method, path, headers.getDelegate(), new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param headers 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse push(io.vertx.core.http.HttpMethod method, java.lang.String path, io.vertx.reactivex.core.MultiMap headers) {
    return 
push(method, path, headers, ar -> { });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param headers 
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpServerResponse> rxPush(io.vertx.core.http.HttpMethod method, java.lang.String path, io.vertx.reactivex.core.MultiMap headers) { 
    return AsyncResultSingle.toSingle($handler -> {
      push(method, path, headers, $handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param handler 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse push(io.vertx.core.http.HttpMethod method, java.lang.String path, io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.reactivex.core.http.HttpServerResponse>> handler) { 
    delegate.push(method, path, new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse push(io.vertx.core.http.HttpMethod method, java.lang.String path) {
    return 
push(method, path, ar -> { });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpServerResponse> rxPush(io.vertx.core.http.HttpMethod method, java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      push(method, path, $handler);
    });
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param host the host of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @param handler the handler notified when the response can be written
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse push(io.vertx.core.http.HttpMethod method, java.lang.String host, java.lang.String path, io.vertx.reactivex.core.MultiMap headers, io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.reactivex.core.http.HttpServerResponse>> handler) { 
    delegate.push(method, host, path, headers.getDelegate(), new Handler<AsyncResult<io.vertx.core.http.HttpServerResponse>>() {
      public void handle(AsyncResult<io.vertx.core.http.HttpServerResponse> ar) {
        if (ar.succeeded()) {
          handler.handle(io.vertx.core.Future.succeededFuture(io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)ar.result())));
        } else {
          handler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param host the host of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse push(io.vertx.core.http.HttpMethod method, java.lang.String host, java.lang.String path, io.vertx.reactivex.core.MultiMap headers) {
    return 
push(method, host, path, headers, ar -> { });
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param host the host of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @return a reference to this, so the API can be used fluently
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpServerResponse> rxPush(io.vertx.core.http.HttpMethod method, java.lang.String host, java.lang.String path, io.vertx.reactivex.core.MultiMap headers) { 
    return AsyncResultSingle.toSingle($handler -> {
      push(method, host, path, headers, $handler);
    });
  }

  /**
   * Reset this HTTP/2 stream with the error code <code>0</code>.
   * @return 
   */
  public boolean reset() { 
    boolean ret = delegate.reset();
    return ret;
  }

  /**
   * Reset this response:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, close the connection when the current response has not yet been sent</li>
   * </ul>
   * <p/>
   * When the response has already been sent nothing happens and <code>false</code> is returned as indicator.
   * @param code the error code
   * @return <code>true</code> when reset has been performed
   */
  public boolean reset(long code) { 
    boolean ret = delegate.reset(code);
    return ret;
  }

  /**
   * Write an HTTP/2 frame to the response, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse writeCustomFrame(int type, int flags, io.vertx.reactivex.core.buffer.Buffer payload) { 
    delegate.writeCustomFrame(type, flags, payload.getDelegate());
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#writeCustomFrame} but with an {@link io.vertx.reactivex.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse writeCustomFrame(io.vertx.reactivex.core.http.HttpFrame frame) { 
    delegate.writeCustomFrame(frame.getDelegate());
    return this;
  }

  /**
   * Sets the priority of the associated stream
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param streamPriority the priority for this request's stream
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse setStreamPriority(io.vertx.core.http.StreamPriority streamPriority) { 
    delegate.setStreamPriority(streamPriority);
    return this;
  }

  /**
   * Add a cookie. This will be sent back to the client in the response.
   * @param cookie the cookie
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse addCookie(io.vertx.reactivex.core.http.Cookie cookie) { 
    delegate.addCookie(cookie.getDelegate());
    return this;
  }

  /**
   * Expire a cookie, notifying a User Agent to remove it from its cookie jar.
   *
   * NOTE: This method will only remove the first occurrence of the given name. Users probably may want to use:
   * {@link io.vertx.reactivex.core.http.HttpServerResponse#removeCookies}
   * @param name the name of the cookie
   * @return the cookie, if it existed, or null
   */
  public io.vertx.reactivex.core.http.Cookie removeCookie(java.lang.String name) { 
    io.vertx.reactivex.core.http.Cookie ret = io.vertx.reactivex.core.http.Cookie.newInstance((io.vertx.core.http.Cookie)delegate.removeCookie(name));
    return ret;
  }

  /**
   * Remove a cookie from the cookie set. If invalidate is <code>true</code> then it will expire a cookie, notifying a User
   * Agent to remove it from its cookie jar.
   *
   * NOTE: This method will only expire the first occurrence of the given name. Users probably may want to use:
   * {@link io.vertx.reactivex.core.http.HttpServerResponse#removeCookies}
   * @param name the name of the cookie
   * @param invalidate 
   * @return the cookie, if it existed, or <code>null</code>
   */
  public io.vertx.reactivex.core.http.Cookie removeCookie(java.lang.String name, boolean invalidate) { 
    io.vertx.reactivex.core.http.Cookie ret = io.vertx.reactivex.core.http.Cookie.newInstance((io.vertx.core.http.Cookie)delegate.removeCookie(name, invalidate));
    return ret;
  }

  /**
   * Expire all cookies, notifying a User Agent to remove it from its cookie jar.
   *
   * NOTE: the returned  is read-only. This means any attempt to modify (add or remove to the set), will
   * throw {@link java.lang.UnsupportedOperationException}.
   * @param name the name of the cookie
   * @return a read only set of affected cookies, if they existed, or an empty set.
   */
  public java.util.Set<io.vertx.reactivex.core.http.Cookie> removeCookies(java.lang.String name) { 
    java.util.Set<io.vertx.reactivex.core.http.Cookie> ret = delegate.removeCookies(name).stream().map(elt -> io.vertx.reactivex.core.http.Cookie.newInstance((io.vertx.core.http.Cookie)elt)).collect(Collectors.toSet());
    return ret;
  }

  /**
   * Remove all cookies from the cookie set. If invalidate is <code>true</code> then it will expire a cookie, notifying a
   * User Agent to remove it from its cookie jar.
   *
   * NOTE: the returned  is read-only. This means any attempt to modify (add or remove to the set), will
   * throw {@link java.lang.UnsupportedOperationException}.
   * @param name the name of the cookie
   * @param invalidate invalidate from the user agent
   * @return a read only set of affected cookies, if they existed, or an empty set.
   */
  public java.util.Set<io.vertx.reactivex.core.http.Cookie> removeCookies(java.lang.String name, boolean invalidate) { 
    java.util.Set<io.vertx.reactivex.core.http.Cookie> ret = delegate.removeCookies(name, invalidate).stream().map(elt -> io.vertx.reactivex.core.http.Cookie.newInstance((io.vertx.core.http.Cookie)elt)).collect(Collectors.toSet());
    return ret;
  }

  /**
   * Expires a cookie from the cookie set. This will notify a User Agent to remove it from its cookie jar.
   * @param name the name of the cookie
   * @param domain the domain of the cookie
   * @param path the path of the cookie
   * @return the cookie, if it existed, or <code>null</code>
   */
  public io.vertx.reactivex.core.http.Cookie removeCookie(java.lang.String name, java.lang.String domain, java.lang.String path) { 
    io.vertx.reactivex.core.http.Cookie ret = io.vertx.reactivex.core.http.Cookie.newInstance((io.vertx.core.http.Cookie)delegate.removeCookie(name, domain, path));
    return ret;
  }

  /**
   * Remove a cookie from the cookie set. If invalidate is <code>true</code> then it will expire a cookie, notifying a User
   * Agent to remove it from its cookie jar.
   * @param name the name of the cookie
   * @param domain the domain of the cookie
   * @param path the path of the cookie
   * @param invalidate 
   * @return the cookie, if it existed, or <code>null</code>
   */
  public io.vertx.reactivex.core.http.Cookie removeCookie(java.lang.String name, java.lang.String domain, java.lang.String path, boolean invalidate) { 
    io.vertx.reactivex.core.http.Cookie ret = io.vertx.reactivex.core.http.Cookie.newInstance((io.vertx.core.http.Cookie)delegate.removeCookie(name, domain, path, invalidate));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putHeader} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putHeader(java.lang.CharSequence name, java.lang.CharSequence value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putHeader} but providing multiple values via a String Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putHeader(java.lang.String name, java.lang.Iterable<java.lang.String> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putHeader} but with CharSequence Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putHeader(java.lang.CharSequence name, java.lang.Iterable<java.lang.CharSequence> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putTrailer} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putTrailer(java.lang.CharSequence name, java.lang.CharSequence value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putTrailer} but providing multiple values via a String Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putTrailer(java.lang.String name, java.lang.Iterable<java.lang.String> values) { 
    delegate.putTrailer(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putTrailer} but with CharSequence Iterable
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putTrailer(java.lang.CharSequence name, java.lang.Iterable<java.lang.CharSequence> value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  private io.vertx.reactivex.core.MultiMap cached_0;
  private io.vertx.reactivex.core.MultiMap cached_1;
  public static HttpServerResponse newInstance(io.vertx.core.http.HttpServerResponse arg) {
    return arg != null ? new HttpServerResponse(arg) : null;
  }

}
