/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.impl.launcher.commands;

import io.vertx.core.impl.launcher.commands.CommandTestBase;
import io.vertx.core.impl.launcher.commands.ExecUtils;
import io.vertx.core.impl.launcher.commands.Watcher;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import org.assertj.core.api.Assertions;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

public class WatcherTest
extends CommandTestBase {
    protected Watcher watcher;
    protected AtomicInteger deploy;
    protected AtomicInteger undeploy;
    protected File root;

    @Before
    public void prepare() {
        this.root = new File("target/junk/watcher");
        WatcherTest.deleteRecursive(this.root);
        this.root.mkdirs();
        this.deploy = new AtomicInteger();
        this.undeploy = new AtomicInteger();
        this.watcher = new Watcher(this.root, Collections.unmodifiableList(Arrays.asList("**" + File.separator + "*.txt", "windows\\*.win", "unix/*.nix", "FOO.bar")), next -> {
            this.deploy.incrementAndGet();
            if (next != null) {
                next.handle(null);
            }
        }, next -> {
            this.undeploy.incrementAndGet();
            if (next != null) {
                next.handle(null);
            }
        }, null, 10L, 10L);
    }

    @After
    public void close() {
        this.watcher.close();
    }

    @Test
    public void testFileAddition() throws IOException {
        this.watcher.watch();
        this.assertWaitUntil(() -> this.deploy.get() == 1);
        File file = new File(this.root, "foo.txt");
        file.createNewFile();
        this.assertWaitUntil(() -> this.undeploy.get() == 1 && this.deploy.get() == 2);
    }

    @Test
    public void testWithANonMatchingFile() throws IOException, InterruptedException {
        this.watcher.watch();
        this.assertWaitUntil(() -> this.deploy.get() == 1);
        File file = new File(this.root, "foo.nope");
        file.createNewFile();
        Thread.sleep(500L);
        Assertions.assertThat((int)this.undeploy.get()).isEqualTo(0);
        Assertions.assertThat((int)this.deploy.get()).isEqualTo(1);
    }

    @Test
    public void testFileModification() throws IOException, InterruptedException {
        File file = new File(this.root, "foo.txt");
        this.watcher.watch();
        this.assertWaitUntil(() -> this.deploy.get() == 1);
        file.createNewFile();
        this.assertWaitUntil(() -> this.deploy.get() == 2);
        Thread.sleep(1500L);
        file.setLastModified(System.currentTimeMillis());
        this.assertWaitUntil(() -> this.undeploy.get() == 2 && this.deploy.get() == 3);
    }

    @Test
    public void testFileDeletion() throws IOException, InterruptedException {
        File file = new File(this.root, "foo.txt");
        file.createNewFile();
        this.watcher.watch();
        this.assertWaitUntil(() -> this.deploy.get() == 1);
        Thread.sleep(500L);
        file.delete();
        this.assertWaitUntil(() -> this.undeploy.get() == 1 && this.deploy.get() == 2);
    }

    @Test
    public void testFileAdditionAndModificationInDirectory() throws IOException, InterruptedException {
        this.watcher.watch();
        Thread.sleep(500L);
        this.assertWaitUntil(() -> this.deploy.get() == 1);
        File newDir = new File(this.root, "directory");
        newDir.mkdir();
        Thread.sleep(500L);
        File file = new File(newDir, "foo.txt");
        file.createNewFile();
        this.assertWaitUntil(() -> this.undeploy.get() == 1 && this.deploy.get() == 2);
        Thread.sleep(1000L);
        file.setLastModified(System.currentTimeMillis());
        this.assertWaitUntil(() -> this.undeploy.get() == 2 && this.deploy.get() == 3);
        Thread.sleep(1000L);
        WatcherTest.deleteRecursive(newDir);
        this.assertWaitUntil(() -> this.undeploy.get() == 3 && this.deploy.get() == 4);
    }

    @Test
    public void testOSSpecificIncludePaths() throws IOException, InterruptedException {
        this.watcher.watch();
        this.assertWaitUntil(() -> this.deploy.get() == 1);
        File winDir = new File(this.root, "windows");
        winDir.mkdir();
        Thread.sleep(500L);
        File winFile = new File(winDir, "foo.win");
        winFile.createNewFile();
        this.assertWaitUntil(() -> this.undeploy.get() == 1 && this.deploy.get() == 2);
        Thread.sleep(1000L);
        File nixDir = new File(this.root, "unix");
        nixDir.mkdir();
        Thread.sleep(500L);
        File nixFile = new File(nixDir, "foo.nix");
        nixFile.createNewFile();
        this.assertWaitUntil(() -> this.undeploy.get() == 2 && this.deploy.get() == 3);
    }

    @Test
    public void testCaseSensitivity() throws IOException, InterruptedException {
        this.watcher.watch();
        this.assertWaitUntil(() -> this.deploy.get() == 1);
        File file = new File(this.root, "fOo.BAr");
        file.createNewFile();
        if (ExecUtils.isWindows()) {
            this.assertWaitUntil(() -> this.undeploy.get() == 1 && this.deploy.get() == 2, "expected undeploy " + this.undeploy.get() + " == 1 and deploy " + this.deploy.get() + " == 2");
        }
    }

    public static boolean deleteRecursive(File path) {
        File[] files;
        if (!path.exists()) {
            return false;
        }
        boolean ret = true;
        if (path.isDirectory() && (files = path.listFiles()) != null) {
            for (File f : files) {
                ret = ret && WatcherTest.deleteRecursive(f);
            }
        }
        return ret && path.delete();
    }

    @Test
    public void testWatcherPerformances() throws IOException, InterruptedException {
        ArrayList<File> files = new ArrayList<File>();
        for (int i = 0; i < 3; ++i) {
            File dir = new File(this.root, Integer.toString(i));
            dir.mkdirs();
            for (int j = 0; j < 10; ++j) {
                File sub = new File(this.root, Integer.toString(j));
                sub.mkdirs();
                File subsub = new File(sub, "sub");
                subsub.mkdirs();
                for (int k = 0; k < 1000; ++k) {
                    File txt = new File(subsub, k + ".txt");
                    files.add(txt);
                    Files.write(txt.toPath(), Integer.toString(k).getBytes(), new OpenOption[0]);
                    File java = new File(subsub, k + ".java");
                    Files.write(java.toPath(), Integer.toString(k).getBytes(), new OpenOption[0]);
                }
            }
        }
        System.out.println("Environment setup");
        this.watcher.watch();
        this.assertWaitUntil(() -> this.deploy.get() == 1);
        long begin = System.currentTimeMillis();
        Path path = ((File)files.get(0)).toPath();
        Thread.sleep(1000L);
        Files.write(path, "booooo".getBytes(), new OpenOption[0]);
        this.assertWaitUntil(() -> this.undeploy.get() == 1);
        this.assertWaitUntil(() -> this.deploy.get() == 2);
        long end = System.currentTimeMillis();
        System.out.println("Update change applied in " + (end - begin) + " ms");
        begin = System.currentTimeMillis();
        ((File)files.get(1)).delete();
        this.assertWaitUntil(() -> this.undeploy.get() == 2);
        this.assertWaitUntil(() -> this.deploy.get() == 3);
        end = System.currentTimeMillis();
        System.out.println("Deletion change applied in " + (end - begin) + " ms");
        begin = System.currentTimeMillis();
        ((File)files.get(1)).delete();
        File newFile = new File(this.root, "test.txt");
        Files.write(newFile.toPath(), "I'm a new file".getBytes(), new OpenOption[0]);
        this.assertWaitUntil(() -> this.undeploy.get() == 3);
        this.assertWaitUntil(() -> this.deploy.get() == 4);
        end = System.currentTimeMillis();
        System.out.println("Creation change applied in " + (end - begin) + " ms");
    }

    @Test
    public void testRootExtraction() {
        ArrayList<String> patterns = new ArrayList<String>();
        patterns.add("src/main/java/**/*.java");
        List results = Watcher.extractRoots((File)this.root, patterns);
        Assertions.assertThat((List)results).hasSize(1);
        Assertions.assertThat((String)((File)results.get(0)).getAbsolutePath()).contains(new CharSequence[]{this.root.getAbsolutePath()});
        patterns.clear();
        patterns.add(this.root.getParentFile().getAbsolutePath());
        results = Watcher.extractRoots((File)this.root, patterns);
        Assertions.assertThat((List)results).hasSize(1);
        Assertions.assertThat((String)((File)results.get(0)).getAbsolutePath()).contains(new CharSequence[]{this.root.getParentFile().getAbsolutePath()});
        patterns.clear();
        patterns.add("**/*.java");
        results = Watcher.extractRoots((File)this.root, patterns);
        Assertions.assertThat((List)results).hasSize(1);
        Assertions.assertThat((String)((File)results.get(0)).getAbsolutePath()).contains(new CharSequence[]{this.root.getAbsolutePath()});
        patterns.clear();
        patterns.add(this.root.getParentFile().getAbsolutePath() + "/**/*.java");
        results = Watcher.extractRoots((File)this.root, patterns);
        Assertions.assertThat((List)results).hasSize(1);
        Assertions.assertThat((String)((File)results.get(0)).getAbsolutePath()).contains(new CharSequence[]{this.root.getParentFile().getAbsolutePath()});
        patterns.clear();
        patterns.add(this.root.getParentFile().getAbsolutePath() + "/foo.txt");
        results = Watcher.extractRoots((File)this.root, patterns);
        Assertions.assertThat((List)results).hasSize(1);
        Assertions.assertThat((String)((File)results.get(0)).getAbsolutePath()).contains(new CharSequence[]{this.root.getParentFile().getAbsolutePath()});
        patterns.clear();
        patterns.add("foo.txt");
        results = Watcher.extractRoots((File)this.root, patterns);
        Assertions.assertThat((List)results).hasSize(1);
        Assertions.assertThat((String)((File)results.get(0)).getAbsolutePath()).contains(new CharSequence[]{this.root.getParentFile().getAbsolutePath()});
    }
}

