/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.core.http;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.ThreadingModel;
import io.vertx.core.Verticle;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.http.HttpClientRequest;
import io.vertx.core.http.HttpClientResponse;
import io.vertx.core.http.HttpServerResponse;
import io.vertx.core.http.HttpTestBase;
import io.vertx.core.http.HttpVersion;
import io.vertx.core.http.PoolOptions;
import io.vertx.core.http.RequestOptions;
import io.vertx.core.http.StreamResetException;
import io.vertx.core.impl.Utils;
import io.vertx.core.net.NetServer;
import io.vertx.test.core.TestUtils;
import java.util.ArrayList;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.junit.Assume;
import org.junit.Test;

public abstract class HttpClientTimeoutTest
extends HttpTestBase {
    @Test
    public void testConnectTimeoutDoesFire() throws Exception {
        int timeout = 3000;
        this.server.requestHandler(req -> req.response().end());
        this.startServer(this.testAddress);
        ArrayList<HttpClientRequest> requests = new ArrayList<HttpClientRequest>();
        for (int i = 0; i < 5; ++i) {
            HttpClientRequest request = (HttpClientRequest)this.client.request(new RequestOptions(this.requestOptions)).toCompletionStage().toCompletableFuture().get();
            requests.add(request);
        }
        long now = System.currentTimeMillis();
        this.client.request(new RequestOptions(this.requestOptions).setConnectTimeout((long)timeout).setURI("/slow")).onComplete(this.onFailure(err -> {
            this.assertTrue(System.currentTimeMillis() - now >= (long)timeout);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testConnectTimeoutDoesNotFire() throws Exception {
        int timeout = 3000;
        int ratio = 50;
        this.server.requestHandler(req -> req.response().end());
        this.startServer(this.testAddress);
        ArrayList<HttpClientRequest> requests = new ArrayList<HttpClientRequest>();
        for (int i = 0; i < 5; ++i) {
            HttpClientRequest request = (HttpClientRequest)this.client.request(new RequestOptions(this.requestOptions)).toCompletionStage().toCompletableFuture().get();
            requests.add(request);
        }
        this.vertx.setTimer((long)(timeout * ratio / 100), id -> requests.forEach(req -> req.send().compose(HttpClientResponse::body)));
        long now = System.currentTimeMillis();
        this.client.request(new RequestOptions(this.requestOptions).setConnectTimeout((long)timeout).setURI("/slow")).onComplete(this.onSuccess(req -> {
            long elapsed = System.currentTimeMillis() - now;
            this.assertTrue(elapsed >= (long)(timeout * ratio / 100));
            this.assertTrue(elapsed <= (long)timeout);
            this.testComplete();
        }));
        this.await();
    }

    @Test
    public void testTimedOutWaiterDoesNotConnect() throws Exception {
        Assume.assumeTrue((String)"Domain socket don't pass this test", (boolean)this.testAddress.isInetSocket());
        Assume.assumeTrue((String)"HTTP/2 don't pass this test", (this.createBaseClientOptions().getProtocolVersion() == HttpVersion.HTTP_1_1 ? 1 : 0) != 0);
        long responseDelay = 300L;
        int requests = 6;
        CountDownLatch firstCloseLatch = new CountDownLatch(1);
        this.server.close().onComplete(this.onSuccess(v -> firstCloseLatch.countDown()));
        this.awaitLatch(firstCloseLatch);
        this.client.close();
        this.client = this.vertx.createHttpClient(this.createBaseClientOptions().setKeepAlive(false), new PoolOptions().setHttp1MaxSize(1));
        AtomicInteger requestCount = new AtomicInteger(0);
        NetServer server = this.vertx.createNetServer();
        server.connectHandler(socket -> {
            Buffer content = Buffer.buffer();
            AtomicBoolean closed = new AtomicBoolean();
            socket.closeHandler(v -> closed.set(true));
            socket.handler(buff -> {
                content.appendBuffer(buff);
                if (buff.toString().endsWith("\r\n\r\n")) {
                    this.vertx.setTimer(responseDelay, time -> {
                        if (!closed.get()) {
                            requestCount.incrementAndGet();
                            socket.write("HTTP/1.1 200 OK\r\nContent-Length: 2\r\n\r\nOK");
                        }
                    });
                }
            });
        });
        CountDownLatch latch = new CountDownLatch(requests);
        server.listen(this.testAddress).toCompletionStage().toCompletableFuture().get(20L, TimeUnit.SECONDS);
        for (int count = 0; count < requests; ++count) {
            if (count % 2 == 0) {
                this.client.request(this.requestOptions).compose(req -> req.send().andThen(this.onSuccess(resp -> this.assertEquals(200L, resp.statusCode()))).compose(HttpClientResponse::body)).onComplete(this.onSuccess(buff -> {
                    this.assertEquals("OK", buff.toString());
                    latch.countDown();
                }));
                continue;
            }
            this.client.request(new RequestOptions(this.requestOptions).setConnectTimeout(responseDelay / 2L)).onComplete(this.onFailure(err -> latch.countDown()));
        }
        this.awaitLatch(latch);
        this.assertEquals("Incorrect number of connect attempts.", (requests + 1) / 2, requestCount.get());
        server.close();
    }

    @Test
    public void testRequestTimeoutIsNotDelayedAfterResponseIsReceived() throws Exception {
        final int n = 6;
        this.waitFor(n);
        this.server.requestHandler(req -> req.response().end());
        this.startServer(this.testAddress);
        this.vertx.deployVerticle((Verticle)new AbstractVerticle(){

            public void start() throws Exception {
                HttpClientTimeoutTest.this.client.close();
                HttpClientTimeoutTest.this.client = this.vertx.createHttpClient(HttpClientTimeoutTest.this.createBaseClientOptions(), new PoolOptions().setHttp1MaxSize(1));
                for (int i = 0; i < n; ++i) {
                    AtomicBoolean responseReceived = new AtomicBoolean();
                    HttpClientTimeoutTest.this.client.request(HttpClientTimeoutTest.this.requestOptions).onComplete(HttpClientTimeoutTest.this.onSuccess(req -> {
                        req.idleTimeout(500L);
                        req.send().onComplete(HttpClientTimeoutTest.this.onSuccess(resp -> {
                            try {
                                Thread.sleep(150L);
                            }
                            catch (InterruptedException e) {
                                HttpClientTimeoutTest.this.fail(e);
                            }
                            responseReceived.set(true);
                            this.vertx.runOnContext(v -> HttpClientTimeoutTest.this.complete());
                        }));
                    }));
                }
            }
        }, new DeploymentOptions().setThreadingModel(ThreadingModel.WORKER));
        this.await();
    }

    @Test
    public void testRequestTimeoutCanceledWhenRequestEndsNormally() throws Exception {
        this.server.requestHandler(req -> req.response().end());
        this.startServer(this.testAddress);
        AtomicReference exception = new AtomicReference();
        this.client.request(this.requestOptions).onComplete(this.onSuccess(req -> {
            req.exceptionHandler(exception::set).idleTimeout(500L).end();
            this.vertx.setTimer(1000L, id -> {
                this.assertNull("Did not expect any exception", exception.get());
                this.testComplete();
            });
        }));
        this.await();
    }

    @Test
    public void testRequestTimeoutCanceledWhenRequestHasAnOtherError() {
        Assume.assumeFalse((boolean)Utils.isWindows());
        AtomicReference exception = new AtomicReference();
        this.client.request(new RequestOptions().setPort(Integer.valueOf(5000)).setIdleTimeout(800L)).onComplete(this.onFailure(exception::set));
        this.vertx.setTimer(1500L, id -> {
            this.assertNotNull("Expected an exception to be set", exception.get());
            this.assertFalse("Expected to not end with timeout exception, but did: " + exception.get(), exception.get() instanceof TimeoutException);
            this.testComplete();
        });
        this.await();
    }

    @Test
    public void testHttpClientRequestTimeoutResetsTheConnection() throws Exception {
        this.waitFor(3);
        this.server.requestHandler(req -> {
            AtomicBoolean errored = new AtomicBoolean();
            req.exceptionHandler(err -> {
                if (errored.compareAndSet(false, true)) {
                    if (req.version() == HttpVersion.HTTP_2) {
                        StreamResetException reset = (StreamResetException)err;
                        this.assertEquals(8L, reset.getCode());
                    }
                    this.complete();
                }
            });
        });
        this.startServer(this.testAddress);
        this.client.request(this.requestOptions).onComplete(this.onSuccess(req -> {
            req.response().onComplete(this.onFailure(err -> this.complete()));
            req.setChunked(true).sendHead().onComplete(this.onSuccess(version -> req.idleTimeout(500L)));
            AtomicBoolean errored = new AtomicBoolean();
            req.exceptionHandler(err -> {
                if (errored.compareAndSet(false, true)) {
                    this.complete();
                }
            });
        }));
        this.await();
    }

    @Test
    public void testResponseDataTimeout() throws Exception {
        this.waitFor(2);
        Buffer expected = TestUtils.randomBuffer(1000);
        this.server.requestHandler(req -> req.response().setChunked(true).write((Object)expected));
        this.startServer(this.testAddress);
        Buffer received = Buffer.buffer();
        this.client.request(this.requestOptions).onComplete(this.onSuccess(req -> {
            req.response().onComplete(this.onSuccess(resp -> {
                AtomicInteger count = new AtomicInteger();
                resp.exceptionHandler(t -> {
                    if (count.getAndIncrement() == 0) {
                        this.assertTrue(t instanceof TimeoutException);
                        this.assertEquals(expected, received);
                        this.complete();
                    }
                });
                resp.request().idleTimeout(500L);
                resp.handler(buff -> {
                    received.appendBuffer(buff);
                    try {
                        Thread.sleep(100L);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                });
            }));
            AtomicInteger count = new AtomicInteger();
            req.exceptionHandler(t -> {
                if (count.getAndIncrement() == 0) {
                    this.assertTrue(t instanceof TimeoutException);
                    this.assertEquals(expected, received);
                    this.complete();
                }
            });
            req.sendHead();
        }));
        this.await();
    }

    @Test
    public void testRequestTimesOutWhenIndicatedPeriodExpiresWithoutAResponseFromRemoteServer() throws Exception {
        this.server.requestHandler(this.noOpHandler());
        AtomicBoolean failed = new AtomicBoolean();
        this.startServer(this.testAddress);
        this.client.request(new RequestOptions(this.requestOptions).setIdleTimeout(1000L)).compose(HttpClientRequest::send).onComplete(this.onFailure(t -> {
            if (failed.compareAndSet(false, true)) {
                this.testComplete();
            }
        }));
        this.await();
    }

    @Test
    public void testRequestTimeoutExtendedWhenResponseChunksReceived() throws Exception {
        long timeout = 2000L;
        int numChunks = 100;
        AtomicInteger count = new AtomicInteger(0);
        long interval = timeout * 2L / (long)numChunks;
        this.server.requestHandler(req -> {
            req.response().setChunked(true);
            this.vertx.setPeriodic(interval, timerID -> {
                req.response().write("foo");
                if (count.incrementAndGet() == numChunks) {
                    req.response().end();
                    this.vertx.cancelTimer(timerID.longValue());
                }
            });
        });
        this.startServer(this.testAddress);
        this.client.request(new RequestOptions(this.requestOptions).setIdleTimeout(timeout)).compose(req -> req.send().andThen(this.onSuccess(resp -> this.assertEquals(200L, resp.statusCode()))).compose(HttpClientResponse::end)).onComplete(this.onSuccess(v -> this.testComplete()));
        this.await();
    }

    @Test
    public void testRequestsTimeoutInQueue() throws Exception {
        this.server.requestHandler(req -> this.vertx.setTimer(1000L, id -> {
            HttpServerResponse resp = req.response();
            if (!resp.closed()) {
                resp.end();
            }
        }));
        this.client.close();
        this.client = this.vertx.createHttpClient(this.createBaseClientOptions().setKeepAlive(false), new PoolOptions().setHttp1MaxSize(1));
        this.startServer(this.testAddress);
        for (int i = 0; i < 5; ++i) {
            this.client.request(new RequestOptions(this.requestOptions).setIdleTimeout(500L)).compose(HttpClientRequest::send).onComplete(this.onFailure(t -> this.assertTrue(t instanceof TimeoutException)));
        }
        this.client.request(new RequestOptions(this.requestOptions).setIdleTimeout(3000L)).compose(HttpClientRequest::send).onComplete(this.onSuccess(resp -> {
            this.assertEquals(200L, resp.statusCode());
            this.testComplete();
        }));
        this.await();
    }
}

