/*
 * Decompiled with CFR 0.152.
 */
package io.trino.server;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.io.ByteStreams;
import com.google.common.util.concurrent.MoreExecutors;
import io.airlift.log.Logger;
import io.trino.metadata.InternalFunctionBundle;
import io.trino.server.PluginClassLoader;
import io.trino.server.PluginManager;
import io.trino.server.ServerPluginsProvider;
import io.trino.server.ServerPluginsProviderConfig;
import io.trino.spi.Plugin;
import io.trino.spi.classloader.ThreadContextClassLoader;
import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.UncheckedIOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.concurrent.Callable;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;
import picocli.CommandLine;

@CommandLine.Command(name="modulesToConnectors", mixinStandardHelpOptions=true, description={"Maps Trino plugin modules to connectors they provide and filters them using an impacted modules list."})
public class PluginReader
implements Callable<Integer> {
    private static final Logger log = Logger.get(PluginReader.class);
    @CommandLine.Option(names={"-i", "--impacted-modules"}, description={"Impacted modules file generated by the gitflow-incremental-builder (GIB) Maven plugin"})
    private Optional<File> impactedModulesFile;
    @CommandLine.Option(names={"-p", "--plugin-dir"}, description={"Trino plugin directory"})
    private File pluginDir = new File("plugin");
    @CommandLine.Option(names={"-r", "--root-pom"}, description={"Trino root module pom.xml"})
    private File rootPom = new File("pom.xml");

    public static void main(String ... args) {
        int exitCode = new CommandLine((Object)new PluginReader()).execute(args);
        System.exit(exitCode);
    }

    @Override
    public Integer call() {
        Optional<Object> impactedModules = Optional.empty();
        if (this.impactedModulesFile.isPresent() && (impactedModules = PluginReader.readImpactedModules(this.impactedModulesFile.get())).isEmpty()) {
            return 1;
        }
        Map<String, String> modulesToPlugins = PluginReader.mapModulesToPlugins(this.rootPom);
        Map plugins = PluginReader.loadPlugins(this.pluginDir).stream().collect(Collectors.toMap(plugin -> plugin.getClass().getName(), Function.identity()));
        Stream<Object> modulesStream = Objects.requireNonNull(modulesToPlugins).entrySet().stream();
        if (impactedModules.isPresent()) {
            List nonPluginModules = ((List)impactedModules.get()).stream().filter(module -> !modulesToPlugins.containsKey(module)).collect(Collectors.toList());
            if (nonPluginModules.size() != 0) {
                log.warn("Impacted modules list includes non-plugin modules, ignoring it: %s", new Object[]{nonPluginModules});
            } else {
                List finalImpactedModules = (List)impactedModules.get();
                modulesStream = modulesStream.filter(entry -> finalImpactedModules.contains(entry.getKey()));
            }
        }
        modulesStream.forEach(entry -> {
            if (!plugins.containsKey(entry.getValue())) {
                log.warn("Plugin without any connectors: %s", new Object[]{entry.getValue()});
                return;
            }
            PluginReader.printPluginFeatures((Plugin)plugins.get(entry.getValue()));
        });
        return 0;
    }

    private static void printPluginFeatures(Plugin plugin) {
        plugin.getConnectorFactories().forEach(factory -> System.out.println("connector:" + factory.getName()));
        plugin.getBlockEncodings().forEach(encoding -> System.out.println("blockEncoding:" + encoding.getName()));
        plugin.getTypes().forEach(type -> System.out.println(type.getTypeId()));
        plugin.getParametricTypes().forEach(type -> System.out.println("parametricType:" + type.getName()));
        plugin.getFunctions().forEach(functionClass -> InternalFunctionBundle.extractFunctions((Class)functionClass).getFunctions().forEach(function -> System.out.println("function:" + function.getSignature())));
        plugin.getSystemAccessControlFactories().forEach(factory -> System.out.println("systemAccessControl:" + factory.getName()));
        plugin.getGroupProviderFactories().forEach(factory -> System.out.println("groupProvider:" + factory.getName()));
        plugin.getPasswordAuthenticatorFactories().forEach(factory -> System.out.println("passwordAuthenticator:" + factory.getName()));
        plugin.getHeaderAuthenticatorFactories().forEach(factory -> System.out.println("headerAuthenticator:" + factory.getName()));
        plugin.getCertificateAuthenticatorFactories().forEach(factory -> System.out.println("certificateAuthenticator:" + factory.getName()));
        plugin.getEventListenerFactories().forEach(factory -> System.out.println("eventListener:" + factory.getName()));
        plugin.getResourceGroupConfigurationManagerFactories().forEach(factory -> System.out.println("resourceGroupConfigurationManager:" + factory.getName()));
        plugin.getSessionPropertyConfigurationManagerFactories().forEach(factory -> System.out.println("sessionPropertyConfigurationManager:" + factory.getName()));
        plugin.getExchangeManagerFactories().forEach(factory -> System.out.println("exchangeManager:" + factory.getName()));
    }

    private static Map<String, String> mapModulesToPlugins(File rootPom) {
        List<String> modules = PluginReader.readTrinoPlugins(rootPom);
        return modules.stream().collect(Collectors.toMap(Function.identity(), module -> PluginReader.readPluginClassName(rootPom, module)));
    }

    private static List<String> readTrinoPlugins(File rootPom) {
        List list;
        FileReader fileReader = new FileReader(rootPom);
        try {
            MavenXpp3Reader reader = new MavenXpp3Reader();
            Model model = reader.read((Reader)fileReader);
            list = (List)model.getModules().stream().filter(module -> PluginReader.isTrinoPlugin(Objects.requireNonNullElse(rootPom.getParent(), ".") + "/" + module)).collect(ImmutableList.toImmutableList());
        }
        catch (Throwable throwable) {
            try {
                try {
                    fileReader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException(String.format("Couldn't read file %s", rootPom), e);
            }
            catch (XmlPullParserException e) {
                throw new RuntimeException(String.format("Couldn't parse file %s", rootPom), e);
            }
        }
        fileReader.close();
        return list;
    }

    private static boolean isTrinoPlugin(String module) {
        boolean bl;
        String modulePom = module + "/pom.xml";
        FileReader fileReader = new FileReader(modulePom);
        try {
            MavenXpp3Reader reader = new MavenXpp3Reader();
            Model model = reader.read((Reader)fileReader);
            bl = model.getPackaging().equals("trino-plugin");
        }
        catch (Throwable throwable) {
            try {
                try {
                    fileReader.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                throw new UncheckedIOException(String.format("Couldn't read file %s", modulePom), e);
            }
            catch (XmlPullParserException e) {
                throw new RuntimeException(String.format("Couldn't parse file %s", modulePom), e);
            }
        }
        fileReader.close();
        return bl;
    }

    private static String readPluginClassName(File rootPom, String module) {
        String string;
        block8: {
            Path target = Path.of(Objects.requireNonNullElse(rootPom.getParent(), "."), module, "target");
            BiPredicate<Path, BasicFileAttributes> matcher = (path, attributes) -> path.toFile().getName().matches(".*-services\\.jar");
            Stream<Path> files = Files.find(target, 1, matcher, new FileVisitOption[0]);
            try {
                string = files.findFirst().map(jarFile -> PluginReader.readPluginClassName(jarFile.toFile())).orElseThrow(() -> new MissingResourceException(String.format("Couldn't find plugin name in services jar for module %s", module), Plugin.class.getName(), module));
                if (files == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (files != null) {
                        try {
                            files.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(String.format("Couldn't read services jar for module %s", module), e);
                }
            }
            files.close();
        }
        return string;
    }

    private static String readPluginClassName(File serviceJar) {
        try {
            ZipFile zipFile = new ZipFile(serviceJar);
            return zipFile.stream().filter(entry -> !entry.isDirectory() && entry.getName().equals("META-INF/services/io.trino.spi.Plugin")).findFirst().map(entry -> {
                String string;
                BufferedInputStream bis = new BufferedInputStream(zipFile.getInputStream((ZipEntry)entry));
                try {
                    string = new String(ByteStreams.toByteArray((InputStream)bis), StandardCharsets.UTF_8).trim();
                }
                catch (Throwable throwable) {
                    try {
                        try {
                            bis.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        throw new UncheckedIOException(String.format("Couldn't read plugin's service descriptor in %s", serviceJar), e);
                    }
                }
                bis.close();
                return string;
            }).orElseThrow(() -> new MissingResourceException(String.format("Couldn't find 'META-INF/services/io.trino.spi.Plugin' file in the service JAR %s", serviceJar.getPath()), Plugin.class.getName(), serviceJar.getPath()));
        }
        catch (IOException e) {
            throw new UncheckedIOException(String.format("Couldn't process service JAR %s", serviceJar), e);
        }
    }

    private static Optional<List<String>> readImpactedModules(File gibImpactedModules) {
        try {
            return Optional.of(com.google.common.io.Files.asCharSource((File)gibImpactedModules, (Charset)StandardCharsets.UTF_8).readLines());
        }
        catch (IOException e) {
            log.warn((Throwable)e, "Couldn't read file %s", new Object[]{gibImpactedModules});
            return Optional.empty();
        }
    }

    private static List<Plugin> loadPlugins(File path) {
        ServerPluginsProviderConfig config = new ServerPluginsProviderConfig();
        config.setInstalledPluginsDir(path);
        ServerPluginsProvider pluginsProvider = new ServerPluginsProvider(config, MoreExecutors.directExecutor());
        ImmutableList.Builder plugins = ImmutableList.builder();
        pluginsProvider.loadPlugins((plugin, createClassLoader) -> PluginReader.loadPlugin(createClassLoader, (ImmutableList.Builder<Plugin>)plugins), PluginManager::createClassLoader);
        return plugins.build();
    }

    private static void loadPlugin(Supplier<PluginClassLoader> createClassLoader, ImmutableList.Builder<Plugin> plugins) {
        PluginClassLoader pluginClassLoader = createClassLoader.get();
        try (ThreadContextClassLoader ignored = new ThreadContextClassLoader((ClassLoader)pluginClassLoader);){
            PluginReader.loadServicePlugin(pluginClassLoader, plugins);
        }
    }

    private static void loadServicePlugin(PluginClassLoader pluginClassLoader, ImmutableList.Builder<Plugin> plugins) {
        ServiceLoader<Plugin> serviceLoader = ServiceLoader.load(Plugin.class, (ClassLoader)pluginClassLoader);
        ImmutableList loadedPlugins = ImmutableList.copyOf(serviceLoader);
        Preconditions.checkState((!loadedPlugins.isEmpty() ? 1 : 0) != 0, (String)"No service providers of type %s in the classpath: %s", (Object)Plugin.class.getName(), Arrays.asList(pluginClassLoader.getURLs()));
        plugins.addAll((Iterable)loadedPlugins);
    }
}

