/*
 * Decompiled with CFR 0.152.
 */
package io.sentry.android.core;

import io.sentry.JsonObjectReader;
import io.sentry.SentryEnvelope;
import io.sentry.SentryEnvelopeItem;
import io.sentry.SentryEvent;
import io.sentry.SentryItemType;
import io.sentry.SentryLevel;
import io.sentry.android.core.SentryAndroidOptions;
import io.sentry.vendor.gson.stream.JsonToken;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

@ApiStatus.Internal
public final class NativeEventCollector {
    private static final String NATIVE_PLATFORM = "native";
    private static final long TIMESTAMP_TOLERANCE_MS = 5000L;
    @NotNull
    private final SentryAndroidOptions options;
    @NotNull
    private final List<NativeEnvelopeMetadata> nativeEnvelopes = new ArrayList<NativeEnvelopeMetadata>();
    private boolean collected = false;

    public NativeEventCollector(@NotNull SentryAndroidOptions options) {
        this.options = options;
    }

    public void collect() {
        if (this.collected) {
            return;
        }
        this.collected = true;
        @Nullable String outboxPath = this.options.getOutboxPath();
        if (outboxPath == null) {
            this.options.getLogger().log(SentryLevel.DEBUG, "Outbox path is null, skipping native event collection.", new Object[0]);
            return;
        }
        File outboxDir = new File(outboxPath);
        File[] files = outboxDir.listFiles();
        if (files == null) {
            this.options.getLogger().log(SentryLevel.DEBUG, "Outbox path is not a directory or an I/O error occurred: %s", new Object[]{outboxPath});
            return;
        }
        if (files.length == 0) {
            this.options.getLogger().log(SentryLevel.DEBUG, "No envelope files found in outbox.", new Object[0]);
            return;
        }
        this.options.getLogger().log(SentryLevel.DEBUG, "Scanning %d files in outbox for native events.", new Object[]{files.length});
        for (File file : files) {
            NativeEnvelopeMetadata metadata;
            if (!file.isFile() || !this.isRelevantFileName(file.getName()) || (metadata = this.extractNativeEnvelopeMetadata(file)) == null) continue;
            this.nativeEnvelopes.add(metadata);
            this.options.getLogger().log(SentryLevel.DEBUG, "Found native event in outbox: %s (timestamp: %d)", new Object[]{file.getName(), metadata.getTimestampMs()});
        }
        this.options.getLogger().log(SentryLevel.DEBUG, "Collected %d native events from outbox.", new Object[]{this.nativeEnvelopes.size()});
    }

    @Nullable
    public NativeEventData findAndRemoveMatchingNativeEvent(long tombstoneTimestampMs) {
        this.collect();
        for (NativeEnvelopeMetadata metadata : this.nativeEnvelopes) {
            long timeDiff = Math.abs(tombstoneTimestampMs - metadata.getTimestampMs());
            if (timeDiff > 5000L) continue;
            this.options.getLogger().log(SentryLevel.DEBUG, "Matched native event by timestamp (diff: %d ms)", new Object[]{timeDiff});
            this.nativeEnvelopes.remove(metadata);
            return this.loadFullNativeEventData(metadata.getFile());
        }
        return null;
    }

    public boolean deleteNativeEventFile(@NotNull NativeEventData nativeEventData) {
        File file = nativeEventData.getFile();
        try {
            if (file.delete()) {
                this.options.getLogger().log(SentryLevel.DEBUG, "Deleted native event file from outbox: %s", new Object[]{file.getName()});
                return true;
            }
            this.options.getLogger().log(SentryLevel.WARNING, "Failed to delete native event file: %s", new Object[]{file.getAbsolutePath()});
            return false;
        }
        catch (Throwable e) {
            this.options.getLogger().log(SentryLevel.ERROR, e, "Error deleting native event file: %s", new Object[]{file.getAbsolutePath()});
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    private NativeEnvelopeMetadata extractNativeEnvelopeMetadata(@NotNull File file) {
        long maxEnvelopeSize = 0xC800000L;
        long bytesProcessed = 0L;
        try (BufferedInputStream stream = new BufferedInputStream(new FileInputStream(file));){
            int next;
            int headerBytes = this.skipLine(stream);
            if (headerBytes < 0) {
                NativeEnvelopeMetadata nativeEnvelopeMetadata = null;
                return nativeEnvelopeMetadata;
            }
            bytesProcessed += (long)headerBytes;
            do {
                if (bytesProcessed >= 0xC800000L) return null;
                @Nullable String itemHeaderLine = this.readLine(stream);
                if (itemHeaderLine == null) return null;
                if (itemHeaderLine.isEmpty()) {
                    return null;
                }
                bytesProcessed += (long)(itemHeaderLine.length() + 1);
                @Nullable ItemHeaderInfo headerInfo = this.parseItemHeader(itemHeaderLine);
                if (headerInfo == null) {
                    return null;
                }
                if ("event".equals(headerInfo.type)) {
                    @Nullable NativeEnvelopeMetadata metadata = this.extractMetadataFromEventPayload(stream, headerInfo.length, file);
                    if (metadata != null) {
                        NativeEnvelopeMetadata nativeEnvelopeMetadata = metadata;
                        return nativeEnvelopeMetadata;
                    }
                } else {
                    NativeEventCollector.skipBytes(stream, headerInfo.length);
                }
                bytesProcessed += (long)headerInfo.length;
                next = ((InputStream)stream).read();
                if (next == -1) {
                    return null;
                }
                ++bytesProcessed;
            } while (next == 10);
            return null;
        }
        catch (Throwable e) {
            this.options.getLogger().log(SentryLevel.DEBUG, e, "Error extracting metadata from envelope file: %s", new Object[]{file.getAbsolutePath()});
        }
        return null;
    }

    @Nullable
    private NativeEnvelopeMetadata extractMetadataFromEventPayload(@NotNull InputStream stream, int payloadLength, @NotNull File file) {
        NativeEnvelopeMetadata result = null;
        try (BoundedInputStream boundedStream = new BoundedInputStream(stream, payloadLength);
             InputStreamReader reader = new InputStreamReader((InputStream)boundedStream, StandardCharsets.UTF_8);){
            JsonObjectReader jsonReader = new JsonObjectReader((Reader)reader);
            String platform = null;
            Date timestamp = null;
            jsonReader.beginObject();
            while (jsonReader.peek() == JsonToken.NAME) {
                String name;
                switch (name = jsonReader.nextName()) {
                    case "platform": {
                        platform = jsonReader.nextStringOrNull();
                        break;
                    }
                    case "timestamp": {
                        timestamp = jsonReader.nextDateOrNull(this.options.getLogger());
                        break;
                    }
                    default: {
                        jsonReader.skipValue();
                    }
                }
                if (platform == null || timestamp == null) continue;
                break;
            }
            if (NATIVE_PLATFORM.equals(platform) && timestamp != null) {
                result = new NativeEnvelopeMetadata(file, timestamp.getTime());
            }
        }
        catch (Throwable e) {
            this.options.getLogger().log(SentryLevel.DEBUG, e, "Error parsing event JSON from: %s", new Object[]{file.getName()});
        }
        return result;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    private NativeEventData loadFullNativeEventData(@NotNull File file) {
        try (BufferedInputStream stream = new BufferedInputStream(new FileInputStream(file));){
            SentryEnvelope envelope = this.options.getEnvelopeReader().read((InputStream)stream);
            if (envelope == null) {
                NativeEventData nativeEventData = null;
                return nativeEventData;
            }
            Iterator iterator = envelope.getItems().iterator();
            while (iterator.hasNext()) {
                SentryEnvelopeItem item = (SentryEnvelopeItem)iterator.next();
                if (!SentryItemType.Event.equals((Object)item.getHeader().getType())) continue;
                try (BufferedReader eventReader = new BufferedReader(new InputStreamReader((InputStream)new ByteArrayInputStream(item.getData()), StandardCharsets.UTF_8));){
                    SentryEvent event = (SentryEvent)this.options.getSerializer().deserialize((Reader)eventReader, SentryEvent.class);
                    if (event == null || !NATIVE_PLATFORM.equals(event.getPlatform())) continue;
                    NativeEventData nativeEventData = new NativeEventData(event, file, envelope);
                    return nativeEventData;
                }
            }
            return null;
        }
        catch (Throwable e) {
            this.options.getLogger().log(SentryLevel.DEBUG, e, "Error loading envelope file: %s", new Object[]{file.getAbsolutePath()});
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Nullable
    private ItemHeaderInfo parseItemHeader(@NotNull String headerLine) {
        try (InputStreamReader reader = new InputStreamReader((InputStream)new ByteArrayInputStream(headerLine.getBytes(StandardCharsets.UTF_8)), StandardCharsets.UTF_8);){
            JsonObjectReader jsonReader = new JsonObjectReader((Reader)reader);
            String type = null;
            int length = -1;
            jsonReader.beginObject();
            while (jsonReader.peek() == JsonToken.NAME) {
                String name;
                switch (name = jsonReader.nextName()) {
                    case "type": {
                        type = jsonReader.nextStringOrNull();
                        break;
                    }
                    case "length": {
                        length = jsonReader.nextInt();
                        break;
                    }
                    default: {
                        jsonReader.skipValue();
                    }
                }
                if (type == null || length < 0) continue;
            }
            if (length < 0) return null;
            ItemHeaderInfo itemHeaderInfo = new ItemHeaderInfo(type, length);
            return itemHeaderInfo;
        }
        catch (Throwable e) {
            this.options.getLogger().log(SentryLevel.DEBUG, e, "Error parsing item header", new Object[0]);
        }
        return null;
    }

    @Nullable
    private String readLine(@NotNull InputStream stream) throws IOException {
        int b;
        StringBuilder sb = new StringBuilder();
        while ((b = stream.read()) != -1) {
            if (b == 10) {
                return sb.toString();
            }
            sb.append((char)b);
        }
        return sb.length() > 0 ? sb.toString() : null;
    }

    private int skipLine(@NotNull InputStream stream) throws IOException {
        int b;
        int count = 0;
        while ((b = stream.read()) != -1) {
            ++count;
            if (b != 10) continue;
            return count;
        }
        return count > 0 ? count : -1;
    }

    private static void skipBytes(@NotNull InputStream stream, long count) throws IOException {
        long remaining = count;
        while (remaining > 0L) {
            long skipped = stream.skip(remaining);
            if (skipped == 0L) {
                if (stream.read() == -1) {
                    throw new EOFException("Unexpected end of stream while skipping bytes");
                }
                --remaining;
                continue;
            }
            remaining -= skipped;
        }
    }

    private boolean isRelevantFileName(@Nullable String fileName) {
        return fileName != null && !fileName.startsWith("session") && !fileName.startsWith("previous_session") && !fileName.startsWith("startup_crash");
    }

    static final class NativeEnvelopeMetadata {
        @NotNull
        private final File file;
        private final long timestampMs;

        NativeEnvelopeMetadata(@NotNull File file, long timestampMs) {
            this.file = file;
            this.timestampMs = timestampMs;
        }

        @NotNull
        File getFile() {
            return this.file;
        }

        long getTimestampMs() {
            return this.timestampMs;
        }
    }

    public static final class NativeEventData {
        @NotNull
        private final SentryEvent event;
        @NotNull
        private final File file;
        @NotNull
        private final SentryEnvelope envelope;

        NativeEventData(@NotNull SentryEvent event, @NotNull File file, @NotNull SentryEnvelope envelope) {
            this.event = event;
            this.file = file;
            this.envelope = envelope;
        }

        @NotNull
        public SentryEvent getEvent() {
            return this.event;
        }

        @NotNull
        public File getFile() {
            return this.file;
        }

        @NotNull
        public SentryEnvelope getEnvelope() {
            return this.envelope;
        }
    }

    private static final class ItemHeaderInfo {
        @Nullable
        final String type;
        final int length;

        ItemHeaderInfo(@Nullable String type, int length) {
            this.type = type;
            this.length = length;
        }
    }

    private static final class BoundedInputStream
    extends InputStream {
        @NotNull
        private final InputStream inner;
        private long remaining;

        BoundedInputStream(@NotNull InputStream inner, int limit) {
            this.inner = inner;
            this.remaining = limit;
        }

        @Override
        public int read() throws IOException {
            if (this.remaining <= 0L) {
                return -1;
            }
            int result = this.inner.read();
            if (result != -1) {
                --this.remaining;
            }
            return result;
        }

        @Override
        public int read(byte[] b, int off, int len) throws IOException {
            if (this.remaining <= 0L) {
                return -1;
            }
            int toRead = Math.min(len, (int)this.remaining);
            int result = this.inner.read(b, off, toRead);
            if (result > 0) {
                this.remaining -= (long)result;
            }
            return result;
        }

        @Override
        public long skip(long n) throws IOException {
            long toSkip = Math.min(n, this.remaining);
            long skipped = this.inner.skip(toSkip);
            this.remaining -= skipped;
            return skipped;
        }

        @Override
        public int available() throws IOException {
            return Math.min(this.inner.available(), (int)this.remaining);
        }

        @Override
        public void close() throws IOException {
            NativeEventCollector.skipBytes(this.inner, this.remaining);
            this.remaining = 0L;
        }
    }
}

