/*
 * Decompiled with CFR 0.152.
 */
package io.rsocket;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufAllocator;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.collection.IntObjectHashMap;
import io.rsocket.DuplexConnection;
import io.rsocket.KeepAliveHandler;
import io.rsocket.Payload;
import io.rsocket.RSocket;
import io.rsocket.StreamIdSupplier;
import io.rsocket.exceptions.ConnectionErrorException;
import io.rsocket.exceptions.Exceptions;
import io.rsocket.frame.CancelFrameFlyweight;
import io.rsocket.frame.ErrorFrameFlyweight;
import io.rsocket.frame.FrameHeaderFlyweight;
import io.rsocket.frame.FrameType;
import io.rsocket.frame.MetadataPushFrameFlyweight;
import io.rsocket.frame.PayloadFrameFlyweight;
import io.rsocket.frame.RequestChannelFrameFlyweight;
import io.rsocket.frame.RequestFireAndForgetFrameFlyweight;
import io.rsocket.frame.RequestNFrameFlyweight;
import io.rsocket.frame.RequestResponseFrameFlyweight;
import io.rsocket.frame.RequestStreamFrameFlyweight;
import io.rsocket.frame.decoder.PayloadDecoder;
import io.rsocket.internal.LimitableRequestPublisher;
import io.rsocket.internal.UnboundedProcessor;
import io.rsocket.internal.UnicastMonoProcessor;
import java.nio.channels.ClosedChannelException;
import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;
import java.util.function.Consumer;
import org.reactivestreams.Processor;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.SignalType;
import reactor.core.publisher.UnicastProcessor;

class RSocketClient
implements RSocket {
    private final DuplexConnection connection;
    private final PayloadDecoder payloadDecoder;
    private final Consumer<Throwable> errorConsumer;
    private final StreamIdSupplier streamIdSupplier;
    private final Map<Integer, LimitableRequestPublisher> senders;
    private final Map<Integer, Processor<Payload, Payload>> receivers;
    private final UnboundedProcessor<ByteBuf> sendProcessor;
    private final Lifecycle lifecycle = new Lifecycle();
    private final ByteBufAllocator allocator;
    private KeepAliveHandler keepAliveHandler;

    RSocketClient(ByteBufAllocator allocator, DuplexConnection connection, PayloadDecoder payloadDecoder, Consumer<Throwable> errorConsumer, StreamIdSupplier streamIdSupplier) {
        this(allocator, connection, payloadDecoder, errorConsumer, streamIdSupplier, Duration.ZERO, Duration.ZERO, 0);
    }

    RSocketClient(ByteBufAllocator allocator, DuplexConnection connection, PayloadDecoder payloadDecoder, Consumer<Throwable> errorConsumer, StreamIdSupplier streamIdSupplier, Duration tickPeriod, Duration ackTimeout, int missedAcks) {
        this.allocator = allocator;
        this.connection = connection;
        this.payloadDecoder = payloadDecoder;
        this.errorConsumer = errorConsumer;
        this.streamIdSupplier = streamIdSupplier;
        this.senders = Collections.synchronizedMap(new IntObjectHashMap());
        this.receivers = Collections.synchronizedMap(new IntObjectHashMap());
        this.sendProcessor = new UnboundedProcessor();
        connection.onClose().doFinally(signalType -> this.terminate()).subscribe(null, errorConsumer);
        connection.send((Publisher<ByteBuf>)this.sendProcessor).doFinally(this::handleSendProcessorCancel).subscribe(null, this::handleSendProcessorError);
        connection.receive().subscribe(this::handleIncomingFrames, errorConsumer);
        if (!Duration.ZERO.equals(tickPeriod)) {
            this.keepAliveHandler = KeepAliveHandler.ofClient(new KeepAliveHandler.KeepAlive(tickPeriod, ackTimeout, missedAcks));
            this.keepAliveHandler.timeout().subscribe(keepAlive -> {
                String message = String.format("No keep-alive acks for %d ms", keepAlive.getTimeoutMillis());
                ConnectionErrorException err = new ConnectionErrorException(message);
                this.lifecycle.setTerminationError(err);
                errorConsumer.accept(err);
                connection.dispose();
            });
            this.keepAliveHandler.send().subscribe(this.sendProcessor::onNext);
        } else {
            this.keepAliveHandler = null;
        }
    }

    private void handleSendProcessorError(Throwable t) {
        Throwable terminationError = this.lifecycle.getTerminationError();
        Throwable err = terminationError != null ? terminationError : t;
        this.receivers.values().forEach(subscriber -> {
            try {
                subscriber.onError(err);
            }
            catch (Throwable e) {
                this.errorConsumer.accept(e);
            }
        });
        this.senders.values().forEach(LimitableRequestPublisher::cancel);
    }

    private void handleSendProcessorCancel(SignalType t) {
        if (SignalType.ON_ERROR == t) {
            return;
        }
        this.receivers.values().forEach(subscriber -> {
            try {
                subscriber.onError(new Throwable("closed connection"));
            }
            catch (Throwable e) {
                this.errorConsumer.accept(e);
            }
        });
        this.senders.values().forEach(LimitableRequestPublisher::cancel);
    }

    @Override
    public Mono<Void> fireAndForget(Payload payload) {
        return this.handleFireAndForget(payload);
    }

    @Override
    public Mono<Payload> requestResponse(Payload payload) {
        return this.handleRequestResponse(payload);
    }

    @Override
    public Flux<Payload> requestStream(Payload payload) {
        return this.handleRequestStream(payload);
    }

    @Override
    public Flux<Payload> requestChannel(Publisher<Payload> payloads) {
        return this.handleChannel((Flux<Payload>)Flux.from(payloads));
    }

    @Override
    public Mono<Void> metadataPush(Payload payload) {
        return this.handleMetadataPush(payload);
    }

    @Override
    public double availability() {
        return this.connection.availability();
    }

    public void dispose() {
        this.connection.dispose();
    }

    public boolean isDisposed() {
        return this.connection.isDisposed();
    }

    @Override
    public Mono<Void> onClose() {
        return this.connection.onClose();
    }

    private Mono<Void> handleFireAndForget(Payload payload) {
        return this.lifecycle.active().then(Mono.fromRunnable(() -> {
            int streamId = this.streamIdSupplier.nextStreamId();
            ByteBuf requestFrame = RequestFireAndForgetFrameFlyweight.encode(this.allocator, streamId, false, payload.hasMetadata() ? payload.sliceMetadata().retain() : null, payload.sliceData().retain());
            payload.release();
            this.sendProcessor.onNext(requestFrame);
        }));
    }

    private Flux<Payload> handleRequestStream(Payload payload) {
        return this.lifecycle.active().thenMany((Publisher)Flux.defer(() -> {
            int streamId = this.streamIdSupplier.nextStreamId();
            UnicastProcessor receiver = UnicastProcessor.create();
            this.receivers.put(streamId, (Processor<Payload, Payload>)receiver);
            AtomicBoolean first = new AtomicBoolean(false);
            return receiver.doOnRequest(n -> {
                if (first.compareAndSet(false, true) && !receiver.isDisposed()) {
                    this.sendProcessor.onNext(RequestStreamFrameFlyweight.encode(this.allocator, streamId, false, n, payload.sliceMetadata().retain(), payload.sliceData().retain()));
                } else if (this.contains(streamId) && !receiver.isDisposed()) {
                    this.sendProcessor.onNext(RequestNFrameFlyweight.encode(this.allocator, streamId, n));
                }
                this.sendProcessor.drain();
            }).doOnError(t -> {
                if (this.contains(streamId) && !receiver.isDisposed()) {
                    this.sendProcessor.onNext(ErrorFrameFlyweight.encode(this.allocator, streamId, t));
                }
            }).doOnCancel(() -> {
                if (this.contains(streamId) && !receiver.isDisposed()) {
                    this.sendProcessor.onNext(CancelFrameFlyweight.encode(this.allocator, streamId));
                }
            }).doFinally(s -> this.receivers.remove(streamId));
        }));
    }

    private Mono<Payload> handleRequestResponse(Payload payload) {
        return this.lifecycle.active().then(Mono.defer(() -> {
            int streamId = this.streamIdSupplier.nextStreamId();
            ByteBuf requestFrame = RequestResponseFrameFlyweight.encode(this.allocator, streamId, false, payload.sliceMetadata().retain(), payload.sliceData().retain());
            payload.release();
            UnicastMonoProcessor receiver = UnicastMonoProcessor.create();
            this.receivers.put(streamId, receiver);
            this.sendProcessor.onNext(requestFrame);
            return receiver.doOnError(t -> this.sendProcessor.onNext(ErrorFrameFlyweight.encode(this.allocator, streamId, t))).doFinally(s -> {
                if (s == SignalType.CANCEL) {
                    this.sendProcessor.onNext(CancelFrameFlyweight.encode(this.allocator, streamId));
                }
                this.receivers.remove(streamId);
            });
        }));
    }

    private Flux<Payload> handleChannel(Flux<Payload> request) {
        return this.lifecycle.active().thenMany((Publisher)Flux.defer(() -> {
            UnicastProcessor receiver = UnicastProcessor.create();
            int streamId = this.streamIdSupplier.nextStreamId();
            AtomicBoolean firstRequest = new AtomicBoolean(true);
            return receiver.doOnRequest(n -> {
                if (firstRequest.compareAndSet(true, false)) {
                    AtomicBoolean firstPayload = new AtomicBoolean(true);
                    Flux requestFrames = request.transform(f -> {
                        LimitableRequestPublisher wrapped = LimitableRequestPublisher.wrap(f);
                        wrapped.increaseRequestLimit(1L);
                        this.senders.put(streamId, wrapped);
                        this.receivers.put(streamId, (Processor<Payload, Payload>)receiver);
                        return wrapped;
                    }).map(payload -> {
                        ByteBuf requestFrame = firstPayload.compareAndSet(true, false) ? RequestChannelFrameFlyweight.encode(this.allocator, streamId, false, false, n, payload.sliceMetadata().retain(), payload.sliceData().retain()) : PayloadFrameFlyweight.encode(this.allocator, streamId, false, false, true, payload);
                        return requestFrame;
                    }).doOnComplete(() -> {
                        if (this.contains(streamId) && !receiver.isDisposed()) {
                            this.sendProcessor.onNext(PayloadFrameFlyweight.encodeComplete(this.allocator, streamId));
                        }
                        if (firstPayload.get()) {
                            receiver.onComplete();
                        }
                    });
                    requestFrames.subscribe(this.sendProcessor::onNext, t -> {
                        this.errorConsumer.accept((Throwable)t);
                        receiver.dispose();
                    });
                } else if (this.contains(streamId) && !receiver.isDisposed()) {
                    this.sendProcessor.onNext(RequestNFrameFlyweight.encode(this.allocator, streamId, n));
                }
            }).doOnError(t -> {
                if (this.contains(streamId) && !receiver.isDisposed()) {
                    this.sendProcessor.onNext(ErrorFrameFlyweight.encode(this.allocator, streamId, t));
                }
            }).doOnCancel(() -> {
                if (this.contains(streamId) && !receiver.isDisposed()) {
                    this.sendProcessor.onNext(CancelFrameFlyweight.encode(this.allocator, streamId));
                }
            }).doFinally(s -> {
                this.receivers.remove(streamId);
                LimitableRequestPublisher sender = this.senders.remove(streamId);
                if (sender != null) {
                    sender.cancel();
                }
            });
        }));
    }

    private Mono<Void> handleMetadataPush(Payload payload) {
        return this.lifecycle.active().then(Mono.fromRunnable(() -> this.sendProcessor.onNext(MetadataPushFrameFlyweight.encode(this.allocator, payload.sliceMetadata().retain()))));
    }

    private boolean contains(int streamId) {
        return this.receivers.containsKey(streamId);
    }

    protected void terminate() {
        this.lifecycle.setTerminationError(new ClosedChannelException());
        if (this.keepAliveHandler != null) {
            this.keepAliveHandler.dispose();
        }
        try {
            this.receivers.values().forEach(this::cleanUpSubscriber);
            this.senders.values().forEach(this::cleanUpLimitableRequestPublisher);
        }
        finally {
            this.senders.clear();
            this.receivers.clear();
            this.sendProcessor.dispose();
        }
    }

    private synchronized void cleanUpLimitableRequestPublisher(LimitableRequestPublisher<?> limitableRequestPublisher) {
        try {
            limitableRequestPublisher.cancel();
        }
        catch (Throwable t) {
            this.errorConsumer.accept(t);
        }
    }

    private synchronized void cleanUpSubscriber(Processor subscriber) {
        try {
            subscriber.onError(this.lifecycle.getTerminationError());
        }
        catch (Throwable t) {
            this.errorConsumer.accept(t);
        }
    }

    private void handleIncomingFrames(ByteBuf frame) {
        try {
            int streamId = FrameHeaderFlyweight.streamId(frame);
            FrameType type = FrameHeaderFlyweight.frameType(frame);
            if (streamId == 0) {
                this.handleStreamZero(type, frame);
            } else {
                this.handleFrame(streamId, type, frame);
            }
            frame.release();
        }
        catch (Throwable t) {
            ReferenceCountUtil.safeRelease((Object)frame);
            throw reactor.core.Exceptions.propagate((Throwable)t);
        }
    }

    private void handleStreamZero(FrameType type, ByteBuf frame) {
        switch (type) {
            case ERROR: {
                RuntimeException error = Exceptions.from(frame);
                this.lifecycle.setTerminationError(error);
                this.errorConsumer.accept(error);
                this.connection.dispose();
                break;
            }
            case LEASE: {
                break;
            }
            case KEEPALIVE: {
                if (this.keepAliveHandler == null) break;
                this.keepAliveHandler.receive(frame);
                break;
            }
            default: {
                this.errorConsumer.accept(new IllegalStateException("Client received supported frame on stream 0: " + frame.toString()));
            }
        }
    }

    private void handleFrame(int streamId, FrameType type, ByteBuf frame) {
        Subscriber receiver = (Subscriber)this.receivers.get(streamId);
        if (receiver == null) {
            this.handleMissingResponseProcessor(streamId, type, frame);
        } else {
            switch (type) {
                case ERROR: {
                    receiver.onError((Throwable)Exceptions.from(frame));
                    this.receivers.remove(streamId);
                    break;
                }
                case NEXT_COMPLETE: {
                    receiver.onNext(this.payloadDecoder.apply(frame));
                    receiver.onComplete();
                    break;
                }
                case CANCEL: {
                    LimitableRequestPublisher sender = this.senders.remove(streamId);
                    this.receivers.remove(streamId);
                    if (sender == null) break;
                    sender.cancel();
                    break;
                }
                case NEXT: {
                    receiver.onNext(this.payloadDecoder.apply(frame));
                    break;
                }
                case REQUEST_N: {
                    LimitableRequestPublisher sender = this.senders.get(streamId);
                    if (sender == null) break;
                    int n = RequestNFrameFlyweight.requestN(frame);
                    sender.increaseRequestLimit(n);
                    this.sendProcessor.drain();
                    break;
                }
                case COMPLETE: {
                    receiver.onComplete();
                    this.receivers.remove(streamId);
                    break;
                }
                default: {
                    throw new IllegalStateException("Client received supported frame on stream " + streamId + ": " + frame.toString());
                }
            }
        }
    }

    private void handleMissingResponseProcessor(int streamId, FrameType type, ByteBuf frame) {
        if (!this.streamIdSupplier.isBeforeOrCurrent(streamId)) {
            if (type == FrameType.ERROR) {
                String errorMessage = ErrorFrameFlyweight.dataUtf8(frame);
                throw new IllegalStateException("Client received error for non-existent stream: " + streamId + " Message: " + errorMessage);
            }
            throw new IllegalStateException("Client received message for non-existent stream: " + streamId + ", frame type: " + (Object)((Object)type));
        }
    }

    private static class Lifecycle {
        private static final AtomicReferenceFieldUpdater<Lifecycle, Throwable> TERMINATION_ERROR = AtomicReferenceFieldUpdater.newUpdater(Lifecycle.class, Throwable.class, "terminationError");
        private volatile Throwable terminationError;

        private Lifecycle() {
        }

        public Mono<Void> active() {
            return Mono.create(sink -> {
                if (this.terminationError == null) {
                    sink.success();
                } else {
                    sink.error(this.terminationError);
                }
            });
        }

        public Throwable getTerminationError() {
            return this.terminationError;
        }

        public void setTerminationError(Throwable err) {
            TERMINATION_ERROR.compareAndSet(this, null, err);
        }
    }
}

