/*
 * Decompiled with CFR 0.152.
 */
package io.netty5.handler.codec.compression;

import io.netty5.buffer.Buffer;
import io.netty5.buffer.BufferAllocator;
import io.netty5.buffer.BufferComponent;
import io.netty5.buffer.ComponentIterator;
import io.netty5.handler.codec.compression.BufferChecksum;
import io.netty5.handler.codec.compression.DecompressionException;
import io.netty5.handler.codec.compression.Decompressor;
import io.netty5.handler.codec.compression.ZlibWrapper;
import java.nio.ByteBuffer;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.zip.CRC32;
import java.util.zip.Inflater;

public final class ZlibDecompressor
implements Decompressor {
    private static final int FHCRC = 2;
    private static final int FEXTRA = 4;
    private static final int FNAME = 8;
    private static final int FCOMMENT = 16;
    private static final int FRESERVED = 224;
    private Inflater inflater;
    private final byte[] dictionary;
    private final BufferChecksum crc;
    private final boolean decompressConcatenated;
    private final int maxAllocation;
    private GzipState gzipState = GzipState.HEADER_START;
    private int flags = -1;
    private int xlen = -1;
    private boolean finished;
    private boolean closed;
    private boolean decideZlibOrNone;

    private ZlibDecompressor(ZlibWrapper wrapper, byte[] dictionary, boolean decompressConcatenated, int maxAllocation) {
        this.maxAllocation = maxAllocation;
        this.decompressConcatenated = decompressConcatenated;
        switch (wrapper) {
            case GZIP: {
                this.inflater = new Inflater(true);
                this.crc = new BufferChecksum(new CRC32());
                break;
            }
            case NONE: {
                this.inflater = new Inflater(true);
                this.crc = null;
                break;
            }
            case ZLIB: {
                this.inflater = new Inflater();
                this.crc = null;
                break;
            }
            case ZLIB_OR_NONE: {
                this.decideZlibOrNone = true;
                this.crc = null;
                break;
            }
            default: {
                throw new IllegalArgumentException("Only GZIP or ZLIB is supported, but you used " + wrapper);
            }
        }
        this.dictionary = dictionary;
    }

    public static Supplier<ZlibDecompressor> newFactory() {
        return ZlibDecompressor.newFactory(ZlibWrapper.ZLIB, null, false, 0);
    }

    public static Supplier<ZlibDecompressor> newFactory(int maxAllocation) {
        return ZlibDecompressor.newFactory(ZlibWrapper.ZLIB, null, false, maxAllocation);
    }

    public static Supplier<ZlibDecompressor> newFactory(byte[] dictionary) {
        return ZlibDecompressor.newFactory(ZlibWrapper.ZLIB, dictionary, false, 0);
    }

    public static Supplier<ZlibDecompressor> newFactory(byte[] dictionary, int maxAllocation) {
        return ZlibDecompressor.newFactory(ZlibWrapper.ZLIB, dictionary, false, maxAllocation);
    }

    public static Supplier<ZlibDecompressor> newFactory(ZlibWrapper wrapper) {
        return ZlibDecompressor.newFactory(wrapper, null, false, 0);
    }

    public static Supplier<ZlibDecompressor> newFactory(ZlibWrapper wrapper, int maxAllocation) {
        return ZlibDecompressor.newFactory(wrapper, null, false, maxAllocation);
    }

    public static Supplier<ZlibDecompressor> newFactory(ZlibWrapper wrapper, boolean decompressConcatenated) {
        return ZlibDecompressor.newFactory(wrapper, null, decompressConcatenated, 0);
    }

    public static Supplier<ZlibDecompressor> newFactory(ZlibWrapper wrapper, boolean decompressConcatenated, int maxAllocation) {
        return ZlibDecompressor.newFactory(wrapper, null, decompressConcatenated, maxAllocation);
    }

    public static Supplier<ZlibDecompressor> newFactory(boolean decompressConcatenated) {
        return ZlibDecompressor.newFactory(ZlibWrapper.GZIP, null, decompressConcatenated, 0);
    }

    public static Supplier<ZlibDecompressor> newFactory(boolean decompressConcatenated, int maxAllocation) {
        return ZlibDecompressor.newFactory(ZlibWrapper.GZIP, null, decompressConcatenated, maxAllocation);
    }

    private static Supplier<ZlibDecompressor> newFactory(ZlibWrapper wrapper, byte[] dictionary, boolean decompressConcatenated, int maxAllocation) {
        Objects.requireNonNull(wrapper, "wrapper");
        return () -> new ZlibDecompressor(wrapper, dictionary, decompressConcatenated, maxAllocation);
    }

    @Override
    public Buffer decompress(Buffer in, BufferAllocator allocator) throws DecompressionException {
        if (this.closed) {
            throw new DecompressionException("Decompressor closed");
        }
        if (this.finished) {
            return allocator.allocate(0);
        }
        int readableBytes = in.readableBytes();
        if (readableBytes == 0) {
            return null;
        }
        if (this.decideZlibOrNone) {
            if (readableBytes < 2) {
                return null;
            }
            boolean nowrap = !ZlibDecompressor.looksLikeZlib(in.getShort(in.readerOffset()));
            this.inflater = new Inflater(nowrap);
            this.decideZlibOrNone = false;
        }
        if (this.crc != null && this.gzipState != GzipState.HEADER_END) {
            if (this.gzipState == GzipState.FOOTER_START) {
                if (!this.handleGzipFooter(in)) {
                    return null;
                }
                assert (this.gzipState == GzipState.HEADER_START);
            }
            if (!this.readGZIPHeader(in)) {
                return null;
            }
            readableBytes = in.readableBytes();
            if (readableBytes == 0) {
                return null;
            }
        }
        if (this.inflater.needsInput()) {
            try (ComponentIterator readableIteration = in.forEachComponent();){
                BufferComponent readableComponent = (BufferComponent)readableIteration.firstReadable();
                if (readableComponent.hasReadableArray()) {
                    this.inflater.setInput(readableComponent.readableArray(), readableComponent.readableArrayOffset(), readableComponent.readableBytes());
                } else {
                    this.inflater.setInput(readableComponent.readableBuffer());
                }
            }
        }
        try (Buffer decompressed = this.prepareDecompressBuffer(allocator, null, this.inflater.getRemaining() << 1);){
            int writableComponents;
            boolean readFooter = false;
            while (!this.inflater.needsInput() && (writableComponents = decompressed.countWritableComponents()) != 0) {
                if (writableComponents > 1) {
                    throw new IllegalStateException("Decompress buffer must have array or exactly 1 NIO buffer: " + decompressed);
                }
                try (ComponentIterator writableIteration = decompressed.forEachComponent();){
                    int outputLength;
                    BufferComponent writableComponent = (BufferComponent)writableIteration.firstWritable();
                    int writerIndex = decompressed.writerOffset();
                    int writable = decompressed.writableBytes();
                    if (writableComponent.hasWritableArray()) {
                        byte[] outArray = writableComponent.writableArray();
                        int outIndex = writableComponent.writableArrayOffset();
                        outputLength = this.inflater.inflate(outArray, outIndex, writable);
                    } else {
                        ByteBuffer buffer = writableComponent.writableBuffer();
                        outputLength = this.inflater.inflate(buffer);
                    }
                    if (outputLength > 0) {
                        writableComponent.skipWritableBytes(outputLength);
                        if (this.crc != null) {
                            this.crc.update(decompressed, writerIndex, outputLength);
                        }
                    } else if (this.inflater.needsDictionary()) {
                        if (this.dictionary == null) {
                            throw new DecompressionException("decompression failure, unable to set dictionary as non was specified");
                        }
                        this.inflater.setDictionary(this.dictionary);
                    }
                    if (this.inflater.finished()) {
                        if (this.crc == null) {
                            this.finished = true;
                            break;
                        }
                        readFooter = true;
                        break;
                    }
                }
                decompressed = this.prepareDecompressBuffer(allocator, decompressed, this.inflater.getRemaining() << 1);
            }
            int remaining = this.inflater.getRemaining();
            in.skipReadableBytes(readableBytes - remaining);
            if (readFooter) {
                this.gzipState = GzipState.FOOTER_START;
                this.handleGzipFooter(in);
            }
            if (decompressed.readableBytes() > 0) {
                return decompressed;
            }
            decompressed.close();
            return null;
        }
    }

    private boolean handleGzipFooter(Buffer in) {
        if (this.readGZIPFooter(in)) {
            boolean bl = this.finished = !this.decompressConcatenated;
            if (!this.finished) {
                this.inflater.reset();
                this.crc.reset();
                this.gzipState = GzipState.HEADER_START;
                return true;
            }
        }
        return false;
    }

    private boolean readGZIPHeader(Buffer in) {
        switch (this.gzipState) {
            case HEADER_START: {
                if (in.readableBytes() < 10) {
                    return false;
                }
                byte magic0 = in.readByte();
                byte magic1 = in.readByte();
                if (magic0 != 31) {
                    throw new DecompressionException("Input is not in the GZIP format");
                }
                this.crc.update(magic0);
                this.crc.update(magic1);
                int method = in.readUnsignedByte();
                if (method != 8) {
                    throw new DecompressionException("Unsupported compression method " + method + " in the GZIP header");
                }
                this.crc.update(method);
                this.flags = in.readUnsignedByte();
                this.crc.update(this.flags);
                if ((this.flags & 0xE0) != 0) {
                    throw new DecompressionException("Reserved flags are set in the GZIP header");
                }
                this.crc.update(in, in.readerOffset(), 4);
                in.skipReadableBytes(4);
                this.crc.update(in.readUnsignedByte());
                this.crc.update(in.readUnsignedByte());
                this.gzipState = GzipState.FLG_READ;
            }
            case FLG_READ: {
                if ((this.flags & 4) != 0) {
                    if (in.readableBytes() < 2) {
                        return false;
                    }
                    int xlen1 = in.readUnsignedByte();
                    int xlen2 = in.readUnsignedByte();
                    this.crc.update(xlen1);
                    this.crc.update(xlen2);
                    this.xlen |= xlen1 << 8 | xlen2;
                }
                this.gzipState = GzipState.XLEN_READ;
            }
            case XLEN_READ: {
                if (this.xlen != -1) {
                    if (in.readableBytes() < this.xlen) {
                        return false;
                    }
                    this.crc.update(in, in.readerOffset(), this.xlen);
                    in.skipReadableBytes(this.xlen);
                }
                this.gzipState = GzipState.SKIP_FNAME;
            }
            case SKIP_FNAME: {
                if (!this.skipIfNeeded(in, 8)) {
                    return false;
                }
                this.gzipState = GzipState.SKIP_COMMENT;
            }
            case SKIP_COMMENT: {
                if (!this.skipIfNeeded(in, 16)) {
                    return false;
                }
                this.gzipState = GzipState.PROCESS_FHCRC;
            }
            case PROCESS_FHCRC: {
                if ((this.flags & 2) != 0 && !this.verifyCrc16(in)) {
                    return false;
                }
                this.crc.reset();
                this.gzipState = GzipState.HEADER_END;
            }
            case HEADER_END: {
                return true;
            }
        }
        throw new IllegalStateException();
    }

    private boolean skipIfNeeded(Buffer in, int flagMask) {
        block2: {
            int b;
            if ((this.flags & flagMask) == 0) break block2;
            do {
                if (in.readableBytes() == 0) {
                    return false;
                }
                b = in.readUnsignedByte();
                this.crc.update(b);
            } while (b != 0);
        }
        return true;
    }

    private boolean readGZIPFooter(Buffer in) {
        if (in.readableBytes() < 8) {
            return false;
        }
        boolean enoughData = this.verifyCrc(in);
        assert (enoughData);
        int dataLength = 0;
        for (int i = 0; i < 4; ++i) {
            dataLength |= in.readUnsignedByte() << i * 8;
        }
        int readLength = this.inflater.getTotalOut();
        if (dataLength != readLength) {
            throw new DecompressionException("Number of bytes mismatch. Expected: " + dataLength + ", Got: " + readLength);
        }
        return true;
    }

    private boolean verifyCrc(Buffer in) {
        if (in.readableBytes() < 4) {
            return false;
        }
        long crcValue = 0L;
        for (int i = 0; i < 4; ++i) {
            crcValue |= (long)in.readUnsignedByte() << i * 8;
        }
        long readCrc = this.crc.getValue();
        if (crcValue != readCrc) {
            throw new DecompressionException("CRC value mismatch. Expected: " + crcValue + ", Got: " + readCrc);
        }
        return true;
    }

    private boolean verifyCrc16(Buffer in) {
        if (in.readableBytes() < 2) {
            return false;
        }
        long readCrc32 = this.crc.getValue();
        long crc16Value = 0L;
        long readCrc16 = 0L;
        for (int i = 0; i < 2; ++i) {
            crc16Value |= (long)in.readUnsignedByte() << i * 8;
            readCrc16 |= (readCrc32 >> i * 8 & 0xFFL) << i * 8;
        }
        if (crc16Value != readCrc16) {
            throw new DecompressionException("CRC16 value mismatch. Expected: " + crc16Value + ", Got: " + readCrc16);
        }
        return true;
    }

    private static boolean looksLikeZlib(short cmf_flg) {
        return (cmf_flg & 0x7800) == 30720 && cmf_flg % 31 == 0;
    }

    protected Buffer prepareDecompressBuffer(BufferAllocator allocator, Buffer buffer, int preferredSize) {
        if (buffer == null) {
            if (this.maxAllocation == 0) {
                return allocator.allocate(preferredSize);
            }
            Buffer buf = allocator.allocate(Math.min(preferredSize, this.maxAllocation));
            buf.implicitCapacityLimit(this.maxAllocation);
            return buf;
        }
        if (buffer.implicitCapacityLimit() < preferredSize) {
            this.decompressionBufferExhausted(buffer);
            buffer.skipReadableBytes(buffer.readableBytes());
            throw new DecompressionException("Decompression buffer has reached maximum size: " + buffer.implicitCapacityLimit());
        }
        buffer.ensureWritable(preferredSize);
        return buffer;
    }

    protected void decompressionBufferExhausted(Buffer buffer) {
        this.finished = true;
    }

    @Override
    public boolean isFinished() {
        return this.finished;
    }

    @Override
    public void close() {
        this.closed = true;
        this.finished = true;
        if (this.inflater != null) {
            this.inflater.end();
        }
    }

    @Override
    public boolean isClosed() {
        return this.closed;
    }

    private static enum GzipState {
        HEADER_START,
        HEADER_END,
        FLG_READ,
        XLEN_READ,
        SKIP_FNAME,
        SKIP_COMMENT,
        PROCESS_FHCRC,
        FOOTER_START;

    }
}

