package io.ethers.abi.error

import io.ethers.abi.error.CustomErrorRegistry.appendResolver
import io.ethers.abi.error.CustomErrorRegistry.prependResolver
import io.ethers.core.types.Bytes

/**
 * Registry of custom contract errors. This registry is used to decode errors returned by contract calls. By default,
 * it contains [CustomErrorFactoryResolver] which can decode errors generated by the ABI generator.
 *
 * Custom resolvers can be added by calling [prependResolver] or [appendResolver]. The resolvers are called in order,
 * and the first resolver that can decode the error is used.
 * */
object CustomErrorRegistry {
    private val resolvers = ArrayList<CustomErrorResolver>().apply {
        add(CustomErrorFactoryResolver)
    }

    /**
     * Add [CustomErrorResolver] to the beginning of the list of resolvers.
     * */
    @JvmStatic
    fun prependResolver(resolver: CustomErrorResolver) {
        synchronized(resolvers) {
            resolvers.add(0, resolver)
        }
    }

    /**
     * Add [CustomErrorResolver] to the end of the list of resolvers.
     * */
    @JvmStatic
    fun appendResolver(resolver: CustomErrorResolver) {
        synchronized(resolvers) {
            resolvers.add(resolver)
        }
    }

    /**
     * Return decoded [CustomContractError] from the first [CustomErrorResolver] that can decode it. If no resolver can decode
     * the error, return null.
     *
     * @return the decoded [CustomContractError], or null if no resolver can decode it.
     * */
    @JvmStatic
    fun getOrNull(error: Bytes): CustomContractError? {
        if (error.size < 4) return null

        for (i in resolvers.indices) {
            val customError = resolvers[i].resolve(error)
            if (customError != null) {
                return customError
            }
        }

        return null
    }
}

/**
 * Resolver for decoding [CustomContractError] from [Bytes].
 * */
interface CustomErrorResolver {
    fun resolve(error: Bytes): CustomContractError?
}

/**
 * [CustomErrorResolver] that uses a list of [CustomErrorFactory] to decode [CustomContractError]s. ABI generated contract
 * errors are automatically added to this resolver.
 * */
object CustomErrorFactoryResolver : CustomErrorResolver {
    private val factories = ArrayList<CustomErrorFactory<*>>()

    @JvmStatic
    fun <T : CustomContractError> addFactories(newFactories: Array<CustomErrorFactory<out T>>) {
        synchronized(factories) {
            factories.addAll(newFactories)
        }
    }

    override fun resolve(error: Bytes): CustomContractError? {
        // the values are only appended to the end of the list, so we can safely iterate by index. Worst case,
        // we miss newly added errors in the current iteration.
        for (i in factories.indices) {
            return factories[i].decode(error) ?: continue
        }

        return null
    }
}
