package io.intercom.android.sdk.push;

import android.app.Application;
import android.content.Context;
import android.os.Bundle;

import com.intercom.twig.Twig;

import java.util.Map;

import io.intercom.android.sdk.Injector;
import io.intercom.android.sdk.identity.UserIdentity;
import io.intercom.android.sdk.logger.LumberMill;
import io.intercom.android.sdk.metrics.MetricFactory;
import io.intercom.android.sdk.utilities.ContextLocaliser;

/**
 * <p>The IntercomPushClient is responsible for communication between the host app push integration and Intercom.
 * This class is automatically called unless your app has it's own GCM/FCM integration or
 * uses other third party GCM/FCM integrations.</p>
 * <p>If you have a class that extends com.google.android.gms.gcm.GcmListenerService,
 * com.google.firebase.iid.FirebaseInstanceIdService or
 * com.google.firebase.messaging.FirebaseMessagingService
 * you will need to create an instance of IntercomPushClient:</p>
 * <pre>private final IntercomPushClient intercomPushClient = new IntercomPushClient();</pre>
 */
public class IntercomPushClient {
    private final Twig twig = LumberMill.getLogger();
    private final PushHandler pushHandler = new PushHandler();

    /**
     * <p>Handle the Intercom push message</p>
     *
     * <p>This takes a push registration token to send to Intercom to enable this device to receive push.</p>
     * <p>In a FCM integration in the class that extends com.google.firebase.iid.FirebaseInstanceIdService</p>
     * <p>You will need to pass the token to the intercomPushClient</p>
     *
     * <pre>
     * public void onTokenRefresh() {
     *     String refreshedToken = FirebaseInstanceId.getInstance().getToken();
     *     pushClient.sendTokenToIntercom(getApplication(), refreshedToken);
     * }
     * </pre>
     *
     * <p>In a GCM integration in the class where you generate your token.</p>
     * <p>You will need to pass the token to the intercomPushClient. For example in an intent service:</p>
     *
     * <pre>
     * public void onHandleIntent(Intent intent) {
     *     InstanceID instanceId = InstanceID.getInstance(this);
     *     String senderId = "YOUR_SENDER_ID";
     *     String token = instanceId.getToken(senderId, GoogleCloudMessaging.INSTANCE_ID_SCOPE, null);
     *     pushClient.sendTokenToIntercom(getApplication(), token);
     * }
     * </pre>
     *
     * @param application A reference to the Application
     * @param token A device push registration token.
     * @since 3.0.7
     */
    public void sendTokenToIntercom(Application application, String token) {
        Injector.init(application);

        if (pushHandler.shouldSendDeviceToken(application, token)) {
            pushHandler.sendTokenToIntercom(application, token, Injector.get().getApi());
        }
    }

    /**
     * <p>Handle the Intercom push message</p>
     *
     * <p>This will take the cloud message bundle delivered to your GcmListenerService and create a notification.</p>
     * <p>In the method onMessageReceived you will need to pass the message onto our push client:</p>
     *
     * <pre>
     * public void onMessageReceived(String from, Bundle message) {
     *     if (intercomPushClient.isIntercomPush(message)) {
     *         intercomPushClient.handlePush(getApplication(), message);
     *     } else {
     *         //DO HOST LOGIC HERE
     *     }
     * }
     * </pre>
     *
     * @param application A reference to the Application
     * @param message The bundle message which was sent to onMessageReceived in your GcmListenerService
     * @since 3.0.5
     */
    public void handlePush(Application application, Bundle message) {
        if (!isIntercomPush(message)) {
            twig.i("The message passed to handlePush was not an Intercom push message.");
            return;
        }

        Injector.init(application);

        Context localisedContext = new ContextLocaliser(Injector.get().getAppIdentity())
                .createLocalisedContext(application);
        boolean appBackgrounded = Injector.get().getStore().state().hostAppState().isBackgrounded();
        UserIdentity userIdentity = Injector.get().getUserIdentity();
        MetricFactory metricFactory = new MetricFactory(userIdentity);
        SystemNotificationManager systemNotificationManager = Injector.get().getSystemNotificationManager();
        pushHandler.handlePush(message, userIdentity, systemNotificationManager,
                Injector.get().getMetricsStore(), appBackgrounded, localisedContext, metricFactory);
    }

    /**
     * <p>Handle the Intercom push message</p>
     *
     * <p>This will take the cloud message bundle delivered to your FirebaseMessagingService
     * and create a notification.</p>
     *
     * <pre>
     * public void onMessageReceived(RemoteMessage remoteMessage) {
     *     Map&lt;String, String&gt; message = remoteMessage.getData();
     *     if (intercomPushClient.isIntercomPush(message)) {
     *         intercomPushClient.handlePush(getApplication(), message);
     *     } else {
     *         //DO HOST LOGIC HERE
     *     }
     * }
     * </pre>
     *
     * @param application A reference to the Application
     * @param message Pass in the remoteMessage.getData() from your onMessageReceived method.
     * @since 3.0.5
     */
    public void handlePush(Application application, Map<String, String> message) {
        handlePush(application, convertMessageMapToBundle(message));
    }

    /**
     * <p>Check if the cloud message is from Intercom</p>
     *
     * <p>This will take the cloud message bundle delivered to your GcmListenerService
     * and determine if it is from Intercom.</p>
     *
     * @param message The bundle message which was sent to onMessageReceived in your GcmListenerService
     * @return a boolean allowing you to check if this is an Intercom cloud message or not.
     * @since 3.0.5
     */
    public boolean isIntercomPush(Bundle message) {
        return pushHandler.isIntercomPush(message);
    }

    /**
     * <p>Check if the cloud message is from Intercom</p>
     *
     * <p>This will take the cloud message bundle delivered to your FirebaseMessagingService
     * and determine if it is from Intercom.</p>
     *
     * @param message Pass in the remoteMessage.getData() from your onMessageReceived method.
     * @return a boolean allowing you to check if this is an Intercom cloud message or not.
     * @since 3.0.5
     */
    public boolean isIntercomPush(Map<String, String> message) {
        return isIntercomPush(convertMessageMapToBundle(message));
    }

    //helper method to convert FCM RemoteMessage data map to GCM style bundle
    private Bundle convertMessageMapToBundle(Map<String, String> message) {
        Bundle bundle = new Bundle();
        for (Map.Entry<String, String> entry : message.entrySet()) {
            bundle.putString(entry.getKey(), entry.getValue());
        }
        return bundle;
    }

}
