package io.intercom.android.sdk.push;

import android.app.Application;
import android.app.NotificationManager;
import android.content.Context;
import android.os.Bundle;

import java.util.Map;

import io.intercom.android.sdk.Injector;
import io.intercom.android.sdk.utilities.ContextLocaliser;

/**
 * <p>The IntercomPushClient is responsible for taking an Intercom push message
 * and creating a notification on the device. This class is automatically called
 * unless your app has it's own GCM/FCM integration or uses other third party GCM/FCM integrations.</p>
 * <p>If you have a class that extends com.google.android.gms.gcm.GcmListenerService or
 * com.google.firebase.messaging.FirebaseMessagingService you will need to
 * create an instance of IntercomPushClient:</p>
 * <pre>private final IntercomPushClient intercomPushClient = new IntercomPushClient();</pre>
 *
 * <p>In the method onMessageReceived you will need to pass the message onto our push client:</p>
 * <p>For a GCM integration:</p>
 * <pre>
 * public void onMessageReceived(String from, Bundle message) {
 *     if (intercomPushClient.isIntercomPush(message)) {
 *         intercomPushClient.handlePush(getApplication(), message);
 *     } else {
 *         //DO HOST LOGIC HERE
 *     }
 * }
 * </pre>
 * <p>For an FCM integration:</p>
 * <pre>
 * public void onMessageReceived(RemoteMessage remoteMessage) {
 *     Map&lt;String, String&gt; message = remoteMessage.getData();
 *     if (intercomPushClient.isIntercomPush(message)) {
 *         intercomPushClient.handlePush(getApplication(), message);
 *     } else {
 *         //DO HOST LOGIC HERE
 *     }
 * }
 * </pre>
 */
public class IntercomPushClient {
    private final IntercomPushHandler intercomPushHandler = new IntercomPushHandler();
    private IntercomNotificationManager intercomNotificationManager;

    /**
     * <p>Handle the Intercom push message</p>
     *
     * <p>This will take the cloud message bundle delivered to your GcmListenerService and create a notification.</p>
     *
     * @param application A reference to the Application
     * @param message The bundle message which was sent to onMessageReceived in your GcmListenerService
     * @since 3.0.5
     */
    public void handlePush(Application application, Bundle message) {
        Injector.init(application);

        if (intercomNotificationManager == null) {
            NotificationManager notificationManager = (NotificationManager) application
                    .getSystemService(Context.NOTIFICATION_SERVICE);
            intercomNotificationManager = new IntercomNotificationManager(notificationManager);
        }
        Context localisedContext = new ContextLocaliser(Injector.get().getAppIdentity())
                .createLocalisedContext(application);
        boolean appBackgrounded = Injector.get().getLifecycleTracker().isAppBackgrounded();
        intercomPushHandler.handlePush(message, Injector.get().getUserIdentity(), intercomNotificationManager,
                Injector.get().getMetricsStore(), appBackgrounded, localisedContext);
    }

    /**
     * <p>Handle the Intercom push message</p>
     *
     * <p>This will take the cloud message bundle delivered to your FirebaseMessagingService
     * and create a notification.</p>
     *
     * @param application A reference to the Application
     * @param message Pass in the remoteMessage.getData() from your onMessageReceived method.
     * @since 3.0.5
     */
    public void handlePush(Application application, Map<String, String> message) {
        handlePush(application, convertMessageMapToBundle(message));
    }

    /**
     * <p>Check if the cloud message is from Intercom</p>
     *
     * <p>This will take the cloud message bundle delivered to your GcmListenerService
     * and determine if it is from Intercom.</p>
     *
     * @param message The bundle message which was sent to onMessageReceived in your GcmListenerService
     * @return a boolean allowing you to check if this is an Intercom cloud message or not.
     * @since 3.0.5
     */
    public boolean isIntercomPush(Bundle message) {
        return intercomPushHandler.isIntercomPush(message);
    }

    /**
     * <p>Check if the cloud message is from Intercom</p>
     *
     * <p>This will take the cloud message bundle delivered to your FirebaseMessagingService
     * and determine if it is from Intercom.</p>
     *
     * @param message Pass in the remoteMessage.getData() from your onMessageReceived method.
     * @return a boolean allowing you to check if this is an Intercom cloud message or not.
     * @since 3.0.5
     */
    public boolean isIntercomPush(Map<String, String> message) {
        return isIntercomPush(convertMessageMapToBundle(message));
    }

    //helper method to convert FCM RemoteMessage data map to GCM style bundle
    private Bundle convertMessageMapToBundle(Map<String, String> message) {
        Bundle bundle = new Bundle();
        for (Map.Entry<String, String> entry : message.entrySet()) {
            bundle.putString(entry.getKey(), entry.getValue());
        }
        return bundle;
    }
}
