/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2022-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.LinkedList;
import java.util.Queue;
import java.util.concurrent.locks.ReentrantLock;

/**
 * A rate meter to limit the number of calls over a rolling time window.
 */
public class RateMeter {
    /**
     * Reference to the logger.
     */
    private static final Logger LOG = LoggerFactory.getLogger(RateMeter.class);

    /**
     * The maximum number of operations allowed in the rolling window.
     */
    private final int quantity;

    /**
     * The duration of the rolling window in milliseconds.
     */
    private final long durationMilliseconds;

    /**
     * Queue to track timestamps of allowed operations.
     */
    private final Queue<Long> ticketTimestamps = new LinkedList<>();

    /**
     * Lock to ensure thread safety.
     */
    private final ReentrantLock lock = new ReentrantLock();

    /**
     * Constructs a new rate meter with the specified quantity and duration.
     *
     * @param quantity the maximum number of operations allowed in the rolling window
     * @param durationMilliseconds the duration of the rolling window in milliseconds
     */
    public RateMeter(int quantity, long durationMilliseconds) {
        this.quantity = quantity;
        this.durationMilliseconds = durationMilliseconds;
    }

    /**
     * Checks if the rate limit has been reached and waits if necessary.
     *
     * @throws InterruptedException if the thread is interrupted while waiting
     */
    public void checkRateLimit() throws InterruptedException {
        lock.lock();
        try {
            long waitTime = calculateWaitTime();

            if (waitTime > 0) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Rate limit reached. Waiting for {}ms.", waitTime);
                }
                Thread.sleep(waitTime);
            }

            recordRequest();
        }
        finally {
            lock.unlock();
        }
    }

    /**
     * Calculates the wait time needed before the next request can be made.
     *
     * @return the wait time in milliseconds, or 0 if no wait is needed
     */
    private long calculateWaitTime() {
        long now = System.currentTimeMillis();
        // Remove expired timestamps from the queue
        while (!ticketTimestamps.isEmpty() && ticketTimestamps.peek() < now - durationMilliseconds) {
            ticketTimestamps.poll();
        }

        // If we've reached the limit, calculate wait time
        if (ticketTimestamps.size() >= quantity) {
            long oldestTimestamp = ticketTimestamps.peek();
            long waitTime = oldestTimestamp + durationMilliseconds - now;
            return Math.max(waitTime, 0);
        }
        return 0;
    }

    /**
     * Records a new request in the rate meter.
     */
    private void recordRequest() {
        long now = System.currentTimeMillis();
        // Clean up expired timestamps again (in case time passed during waiting)
        while (!ticketTimestamps.isEmpty() && ticketTimestamps.peek() < now - durationMilliseconds) {
            ticketTimestamps.poll();
        }

        // Add the current timestamp to the queue
        ticketTimestamps.add(now);

        if (LOG.isDebugEnabled()) {
            LOG.debug("Rate limit usage: {}/{} in the last {}ms", ticketTimestamps.size(), quantity,
                    durationMilliseconds);
        }
    }

    /**
     * Returns the quantity of operations allowed in the rolling window.
     *
     * @return the quantity of operations allowed in the rolling window.
     */
    public int getQuantity() {
        return this.quantity;
    }

    /**
     * Returns the duration of the rolling window in milliseconds.
     * 
     * @return the duration of the rolling window in milliseconds.
     */
    public long getDuration() {
        return this.durationMilliseconds;
    }
}
