/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonAlias;
import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonSetter;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.annotation.Nulls;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * JSON Schema for Common Vulnerability Scoring System version 4.0
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"version", "vectorString", "attackVector", "attackComplexity", "attackRequirements",
        "privilegesRequired", "userInteraction", "vulnConfidentialityImpact", "vulnIntegrityImpact",
        "vulnAvailabilityImpact", "subConfidentialityImpact", "subIntegrityImpact", "subAvailabilityImpact",
        "exploitMaturity", "confidentialityRequirement", "integrityRequirement", "availabilityRequirement",
        "modifiedAttackVector", "modifiedAttackComplexity", "modifiedAttackRequirements", "modifiedPrivilegesRequired",
        "modifiedUserInteraction", "modifiedVulnConfidentialityImpact", "modifiedVulnIntegrityImpact",
        "modifiedVulnAvailabilityImpact", "modifiedSubConfidentialityImpact", "modifiedSubIntegrityImpact",
        "modifiedSubAvailabilityImpact", "Safety", "Automatable", "Recovery", "valueDensity",
        "vulnerabilityResponseEffort", "providerUrgency", "baseScore", "baseSeverity", "threatScore", "threatSeverity",
        "environmentalScore", "environmentalSeverity"})
@NullMarked
public class CvssV4Data implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 8537782209754450697L;
    // "version", "vectorString", "baseScore", "baseSeverity"

    /**
     * Constructor taking only the required set of JSON properties as the minimal constructor for a CvssV4Data.
     *
     * @param version The version
     * @param vectorString The CVSSv4 vector string
     * @param baseScore The CVSSv4 Base score
     * @param baseSeverity The CVSSv4 Base Severity
     */
    @JsonCreator
    public CvssV4Data(@JsonProperty("version") Version version, @JsonProperty("vectorString") String vectorString,
            @JsonProperty("baseScore") Double baseScore, @JsonProperty("baseSeverity") SeverityType baseSeverity) {
        this.version = version;
        this.vectorString = vectorString;
        this.baseScore = baseScore;
        this.baseSeverity = baseSeverity;
    }

    /**
     * Full constructor for CvssV4Data.
     *
     * @param version the CVSS version
     * @param vectorString the CVSS vector string
     * @param attackVector the attack vector
     * @param attackComplexity the attack complexity
     * @param attackRequirements the attack requirements
     * @param privilegesRequired the privileges required
     * @param userInteraction the user interaction
     * @param vulnConfidentialityImpact the vulnerable system confidentiality impact
     * @param vulnIntegrityImpact the vulnerable system integrity impact
     * @param vulnAvailabilityImpact the vulnerable system availability impact
     * @param subConfidentialityImpact the subsequent system confidentiality impact
     * @param subIntegrityImpact the subsequent system integrity impact
     * @param subAvailabilityImpact the subsequent system availability impact
     * @param exploitMaturity the exploit maturity
     * @param confidentialityRequirement the confidentiality requirement
     * @param integrityRequirement the integrity requirement
     * @param availabilityRequirement the availability requirement
     * @param modifiedAttackVector the modified attack vector
     * @param modifiedAttackComplexity the modified attack complexity
     * @param modifiedAttackRequirements the modified attack requirements
     * @param modifiedPrivilegesRequired the modified privileges required
     * @param modifiedUserInteraction the modified user interaction
     * @param modifiedVulnConfidentialityImpact the modified vulnerable system confidentiality impact
     * @param modifiedVulnIntegrityImpact the modified vulnerable system integrity impact
     * @param modifiedVulnAvailabilityImpact the modified vulnerable system availability impact
     * @param modifiedSubConfidentialityImpact the modified subsequent system confidentiality impact
     * @param modifiedSubIntegrityImpact the modified subsequent system integrity impact
     * @param modifiedSubAvailabilityImpact the modified subsequent system availability impact
     * @param safety the safety metric
     * @param automatable the automatable metric
     * @param recovery the recovery metric
     * @param valueDensity the value density
     * @param vulnerabilityResponseEffort the vulnerability response effort
     * @param providerUrgency the provider urgency
     * @param baseScore the base score
     * @param baseSeverity the base severity
     * @param threatScore the threat score
     * @param threatSeverity the threat severity
     * @param environmentalScore the environmental score
     * @param environmentalSeverity the environmental severity
     */
    public CvssV4Data(Version version, String vectorString, @Nullable AttackVectorType attackVector,
            @Nullable AttackComplexityType attackComplexity, @Nullable AttackRequirementsType attackRequirements,
            @Nullable PrivilegesRequiredType privilegesRequired, @Nullable UserInteractionType userInteraction,
            @Nullable CiaType vulnConfidentialityImpact, @Nullable CiaType vulnIntegrityImpact,
            @Nullable CiaType vulnAvailabilityImpact, @Nullable CiaType subConfidentialityImpact,
            @Nullable CiaType subIntegrityImpact, @Nullable CiaType subAvailabilityImpact,
            @Nullable ExploitMaturityType exploitMaturity, @Nullable CiaRequirementType confidentialityRequirement,
            @Nullable CiaRequirementType integrityRequirement, @Nullable CiaRequirementType availabilityRequirement,
            @Nullable ModifiedAttackVectorType modifiedAttackVector,
            @Nullable ModifiedAttackComplexityType modifiedAttackComplexity,
            @Nullable ModifiedAttackRequirementsType modifiedAttackRequirements,
            @Nullable ModifiedPrivilegesRequiredType modifiedPrivilegesRequired,
            @Nullable ModifiedUserInteractionType modifiedUserInteraction,
            @Nullable ModifiedCiaType modifiedVulnConfidentialityImpact,
            @Nullable ModifiedCiaType modifiedVulnIntegrityImpact,
            @Nullable ModifiedCiaType modifiedVulnAvailabilityImpact,
            @Nullable ModifiedSubCType modifiedSubConfidentialityImpact,
            @Nullable ModifiedSubIaType modifiedSubIntegrityImpact,
            @Nullable ModifiedSubIaType modifiedSubAvailabilityImpact, @Nullable SafetyType safety,
            @Nullable AutomatableType automatable, @Nullable RecoveryType recovery,
            @Nullable ValueDensityType valueDensity,
            @Nullable VulnerabilityResponseEffortType vulnerabilityResponseEffort,
            @Nullable ProviderUrgencyType providerUrgency, Double baseScore, SeverityType baseSeverity,
            @Nullable Double threatScore, @Nullable SeverityType threatSeverity, @Nullable Double environmentalScore,
            @Nullable SeverityType environmentalSeverity) {
        this(version, vectorString, baseScore, baseSeverity);
        this.attackVector = attackVector;
        this.attackComplexity = attackComplexity;
        this.attackRequirements = attackRequirements;
        this.privilegesRequired = privilegesRequired;
        this.userInteraction = userInteraction;
        this.vulnConfidentialityImpact = vulnConfidentialityImpact;
        this.vulnIntegrityImpact = vulnIntegrityImpact;
        this.vulnAvailabilityImpact = vulnAvailabilityImpact;
        this.subConfidentialityImpact = subConfidentialityImpact;
        this.subIntegrityImpact = subIntegrityImpact;
        this.subAvailabilityImpact = subAvailabilityImpact;
        this.exploitMaturity = exploitMaturity;
        this.confidentialityRequirement = confidentialityRequirement;
        this.integrityRequirement = integrityRequirement;
        this.availabilityRequirement = availabilityRequirement;
        this.modifiedAttackVector = modifiedAttackVector;
        this.modifiedAttackComplexity = modifiedAttackComplexity;
        this.modifiedAttackRequirements = modifiedAttackRequirements;
        this.modifiedPrivilegesRequired = modifiedPrivilegesRequired;
        this.modifiedUserInteraction = modifiedUserInteraction;
        this.modifiedVulnConfidentialityImpact = modifiedVulnConfidentialityImpact;
        this.modifiedVulnIntegrityImpact = modifiedVulnIntegrityImpact;
        this.modifiedVulnAvailabilityImpact = modifiedVulnAvailabilityImpact;
        this.modifiedSubConfidentialityImpact = modifiedSubConfidentialityImpact;
        this.modifiedSubIntegrityImpact = modifiedSubIntegrityImpact;
        this.modifiedSubAvailabilityImpact = modifiedSubAvailabilityImpact;
        this.safety = safety;
        this.automatable = automatable;
        this.recovery = recovery;
        this.valueDensity = valueDensity;
        this.vulnerabilityResponseEffort = vulnerabilityResponseEffort;
        this.providerUrgency = providerUrgency;
        this.threatScore = threatScore;
        this.threatSeverity = threatSeverity;
        this.environmentalScore = environmentalScore;
        this.environmentalSeverity = environmentalSeverity;
    }

    /**
     * CVSS Version (Required)
     */
    @JsonProperty("version")
    @JsonPropertyDescription("CVSS Version")
    private Version version;
    /**
     * (Required)
     */
    @JsonProperty("vectorString")
    private String vectorString;
    @JsonProperty("attackVector")
    private @Nullable AttackVectorType attackVector;
    @JsonProperty("attackComplexity")
    private @Nullable AttackComplexityType attackComplexity;
    @JsonProperty("attackRequirements")
    private @Nullable AttackRequirementsType attackRequirements;
    @JsonProperty("privilegesRequired")
    private @Nullable PrivilegesRequiredType privilegesRequired;
    @JsonProperty("userInteraction")
    private @Nullable UserInteractionType userInteraction;
    @JsonAlias("vulnerableSystemConfidentiality")
    @JsonProperty("vulnConfidentialityImpact")
    private @Nullable CiaType vulnConfidentialityImpact;
    @JsonAlias("vulnerableSystemIntegrity")
    @JsonProperty("vulnIntegrityImpact")
    private @Nullable CiaType vulnIntegrityImpact;
    @JsonAlias("vulnerableSystemAvailability")
    @JsonProperty("vulnAvailabilityImpact")
    private @Nullable CiaType vulnAvailabilityImpact;
    @JsonAlias("subsequentSystemConfidentiality")
    @JsonProperty("subConfidentialityImpact")
    private @Nullable CiaType subConfidentialityImpact;
    @JsonAlias("subsequentSystemIntegrity")
    @JsonProperty("subIntegrityImpact")
    private @Nullable CiaType subIntegrityImpact;
    @JsonAlias("subsequentSystemAvailability")
    @JsonProperty("subAvailabilityImpact")
    private @Nullable CiaType subAvailabilityImpact;
    @JsonProperty("exploitMaturity")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ExploitMaturityType exploitMaturity = ExploitMaturityType.NOT_DEFINED;
    @JsonAlias("confidentialityRequirements")
    @JsonProperty("confidentialityRequirement")
    private @Nullable CiaRequirementType confidentialityRequirement;
    @JsonAlias("integrityRequirements")
    @JsonProperty("integrityRequirement")
    private @Nullable CiaRequirementType integrityRequirement;
    @JsonAlias("availabilityRequirements")
    @JsonProperty("availabilityRequirement")
    private @Nullable CiaRequirementType availabilityRequirement;
    @JsonProperty("modifiedAttackVector")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedAttackVectorType modifiedAttackVector = ModifiedAttackVectorType.NOT_DEFINED;
    @JsonProperty("modifiedAttackComplexity")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedAttackComplexityType modifiedAttackComplexity = ModifiedAttackComplexityType.NOT_DEFINED;
    @JsonProperty("modifiedAttackRequirements")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedAttackRequirementsType modifiedAttackRequirements = ModifiedAttackRequirementsType.NOT_DEFINED;
    @JsonProperty("modifiedPrivilegesRequired")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedPrivilegesRequiredType modifiedPrivilegesRequired = ModifiedPrivilegesRequiredType.NOT_DEFINED;
    @JsonProperty("modifiedUserInteraction")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedUserInteractionType modifiedUserInteraction = ModifiedUserInteractionType.NOT_DEFINED;
    @JsonAlias("modifiedVulnerableSystemConfidentiality")
    @JsonProperty("modifiedVulnConfidentialityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedCiaType modifiedVulnConfidentialityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonAlias("modifiedVulnerableSystemIntegrity")
    @JsonProperty("modifiedVulnIntegrityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedCiaType modifiedVulnIntegrityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonAlias("modifiedVulnerableSystemAvailability")
    @JsonProperty("modifiedVulnAvailabilityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedCiaType modifiedVulnAvailabilityImpact = ModifiedCiaType.NOT_DEFINED;
    @JsonAlias("modifiedSubsequentSystemConfidentiality")
    @JsonProperty("modifiedSubConfidentialityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedSubCType modifiedSubConfidentialityImpact = ModifiedSubCType.NOT_DEFINED;
    @JsonAlias("modifiedSubsequentSystemIntegrity")
    @JsonProperty("modifiedSubIntegrityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedSubIaType modifiedSubIntegrityImpact = ModifiedSubIaType.NOT_DEFINED;
    @JsonAlias("modifiedSubsequentSystemAvailability")
    @JsonProperty("modifiedSubAvailabilityImpact")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ModifiedSubIaType modifiedSubAvailabilityImpact = ModifiedSubIaType.NOT_DEFINED;
    @JsonAlias("safety")
    @JsonProperty("Safety")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable SafetyType safety = SafetyType.NOT_DEFINED;
    @JsonAlias("automatable")
    @JsonProperty("Automatable")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable AutomatableType automatable = AutomatableType.NOT_DEFINED;
    @JsonAlias("recovery")
    @JsonProperty("Recovery")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable RecoveryType recovery = RecoveryType.NOT_DEFINED;
    @JsonProperty("valueDensity")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ValueDensityType valueDensity = ValueDensityType.NOT_DEFINED;
    @JsonProperty("vulnerabilityResponseEffort")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable VulnerabilityResponseEffortType vulnerabilityResponseEffort = VulnerabilityResponseEffortType.NOT_DEFINED;
    @JsonProperty("providerUrgency")
    @JsonSetter(nulls = Nulls.SKIP)
    private @Nullable ProviderUrgencyType providerUrgency = ProviderUrgencyType.NOT_DEFINED;
    /**
     * (Required)
     */
    @JsonProperty("baseScore")
    private Double baseScore;
    /**
     * (Required)
     */
    @JsonProperty("baseSeverity")
    private SeverityType baseSeverity;
    @JsonProperty("threatScore")
    private @Nullable Double threatScore;
    @JsonProperty("threatSeverity")
    private @Nullable SeverityType threatSeverity;
    @JsonProperty("environmentalScore")
    private @Nullable Double environmentalScore;
    @JsonProperty("environmentalSeverity")
    private @Nullable SeverityType environmentalSeverity;

    /**
     * CVSS Version (Required)
     *
     * @return version
     */
    @JsonProperty("version")
    public Version getVersion() {
        return version;
    }

    /**
     * (Required)
     *
     * @return vectorString
     */
    @JsonProperty("vectorString")
    public String getVectorString() {
        return vectorString;
    }

    /**
     * @return attackVector
     */
    @JsonProperty("attackVector")
    public @Nullable AttackVectorType getAttackVector() {
        return attackVector;
    }

    /**
     * @return attackComplexity
     */
    @JsonProperty("attackComplexity")
    public @Nullable AttackComplexityType getAttackComplexity() {
        return attackComplexity;
    }

    /**
     * @return attackRequirements
     */
    @JsonProperty("attackRequirements")
    public @Nullable AttackRequirementsType getAttackRequirements() {
        return attackRequirements;
    }

    /**
     * @return privilegesRequired
     */
    @JsonProperty("privilegesRequired")
    public @Nullable PrivilegesRequiredType getPrivilegesRequired() {
        return privilegesRequired;
    }

    /**
     * @return userInteraction
     */
    @JsonProperty("userInteraction")
    public @Nullable UserInteractionType getUserInteraction() {
        return userInteraction;
    }

    /**
     * @return vulnConfidentialityImpact
     */
    @JsonProperty("vulnConfidentialityImpact")
    public @Nullable CiaType getVulnConfidentialityImpact() {
        return vulnConfidentialityImpact;
    }

    /**
     * @return vulnIntegrityImpact
     */
    @JsonProperty("vulnIntegrityImpact")
    public @Nullable CiaType getVulnIntegrityImpact() {
        return vulnIntegrityImpact;
    }

    /**
     * @return vulnAvailabilityImpact
     */
    @JsonProperty("vulnAvailabilityImpact")
    public @Nullable CiaType getVulnAvailabilityImpact() {
        return vulnAvailabilityImpact;
    }

    /**
     * @return subConfidentialityImpact
     */
    @JsonProperty("subConfidentialityImpact")
    public @Nullable CiaType getSubConfidentialityImpact() {
        return subConfidentialityImpact;
    }

    /**
     * @return subIntegrityImpact
     */
    @JsonProperty("subIntegrityImpact")
    public @Nullable CiaType getSubIntegrityImpact() {
        return subIntegrityImpact;
    }

    /**
     * @return subAvailabilityImpact
     */
    @JsonProperty("subAvailabilityImpact")
    public @Nullable CiaType getSubAvailabilityImpact() {
        return subAvailabilityImpact;
    }

    /**
     * @return exploitMaturity
     */
    @JsonProperty("exploitMaturity")
    public @Nullable ExploitMaturityType getExploitMaturity() {
        return exploitMaturity;
    }

    /**
     * (Required)
     *
     * @return baseScore
     */
    @JsonProperty("baseScore")
    public Double getBaseScore() {
        return baseScore;
    }

    /**
     * (Required)
     *
     * @return baseSeverity
     */
    @JsonProperty("baseSeverity")
    public SeverityType getBaseSeverity() {
        return baseSeverity;
    }

    /**
     * @return confidentialityRequirement
     */
    @JsonProperty("confidentialityRequirement")
    public @Nullable CiaRequirementType getConfidentialityRequirement() {
        return confidentialityRequirement;
    }

    /**
     * @return integrityRequirement
     */
    @JsonProperty("integrityRequirement")
    public @Nullable CiaRequirementType getIntegrityRequirement() {
        return integrityRequirement;
    }

    /**
     * @return availabilityRequirement
     */
    @JsonProperty("availabilityRequirement")
    public @Nullable CiaRequirementType getAvailabilityRequirement() {
        return availabilityRequirement;
    }

    /**
     * @return modifiedAttackVector
     */
    @JsonProperty("modifiedAttackVector")
    public @Nullable ModifiedAttackVectorType getModifiedAttackVector() {
        return modifiedAttackVector;
    }

    /**
     * @return modifiedAttackComplexity
     */
    @JsonProperty("modifiedAttackComplexity")
    public @Nullable ModifiedAttackComplexityType getModifiedAttackComplexity() {
        return modifiedAttackComplexity;
    }

    /**
     * @return modifiedAttackRequirements
     */
    @JsonProperty("modifiedAttackRequirements")
    public @Nullable ModifiedAttackRequirementsType getModifiedAttackRequirements() {
        return modifiedAttackRequirements;
    }

    /**
     * @return modifiedPrivilegesRequired
     */
    @JsonProperty("modifiedPrivilegesRequired")
    public @Nullable ModifiedPrivilegesRequiredType getModifiedPrivilegesRequired() {
        return modifiedPrivilegesRequired;
    }

    /**
     * @return modifiedUserInteraction
     */
    @JsonProperty("modifiedUserInteraction")
    public @Nullable ModifiedUserInteractionType getModifiedUserInteraction() {
        return modifiedUserInteraction;
    }

    /**
     * @return modifiedVulnConfidentialityImpact
     */
    @JsonProperty("modifiedVulnConfidentialityImpact")
    public @Nullable ModifiedCiaType getModifiedVulnConfidentialityImpact() {
        return modifiedVulnConfidentialityImpact;
    }

    /**
     * @return modifiedVulnIntegrityImpact
     */
    @JsonProperty("modifiedVulnIntegrityImpact")
    public @Nullable ModifiedCiaType getModifiedVulnIntegrityImpact() {
        return modifiedVulnIntegrityImpact;
    }

    /**
     * @return modifiedVulnAvailabilityImpact
     */
    @JsonProperty("modifiedVulnAvailabilityImpact")
    public @Nullable ModifiedCiaType getModifiedVulnAvailabilityImpact() {
        return modifiedVulnAvailabilityImpact;
    }

    /**
     * @return modifiedSubConfidentialityImpact
     */
    @JsonProperty("modifiedSubConfidentialityImpact")
    public @Nullable ModifiedSubCType getModifiedSubConfidentialityImpact() {
        return modifiedSubConfidentialityImpact;
    }

    /**
     * @return modifiedSubIntegrityImpact
     */
    @JsonProperty("modifiedSubIntegrityImpact")
    public @Nullable ModifiedSubIaType getModifiedSubIntegrityImpact() {
        return modifiedSubIntegrityImpact;
    }

    /**
     * @return modifiedSubAvailabilityImpact
     */
    @JsonProperty("modifiedSubAvailabilityImpact")
    public @Nullable ModifiedSubIaType getModifiedSubAvailabilityImpact() {
        return modifiedSubAvailabilityImpact;
    }

    /**
     * @return safety
     */
    @JsonProperty("Safety")
    @JsonAlias("safety")
    public @Nullable SafetyType getSafety() {
        return safety;
    }

    /**
     * @return automatable
     */
    @JsonProperty("Automatable")
    @JsonAlias("automatable")
    public @Nullable AutomatableType getAutomatable() {
        return automatable;
    }

    /**
     * @return recovery
     */
    @JsonProperty("Recovery")
    @JsonAlias("recovery")
    public @Nullable RecoveryType getRecovery() {
        return recovery;
    }

    /**
     * @return valueDensity
     */
    @JsonProperty("valueDensity")
    public @Nullable ValueDensityType getValueDensity() {
        return valueDensity;
    }

    /**
     * @return vulnerabilityResponseEffort
     */
    @JsonProperty("vulnerabilityResponseEffort")
    public @Nullable VulnerabilityResponseEffortType getVulnerabilityResponseEffort() {
        return vulnerabilityResponseEffort;
    }

    /**
     * @return providerUrgency
     */
    @JsonProperty("providerUrgency")
    public @Nullable ProviderUrgencyType getProviderUrgency() {
        return providerUrgency;
    }

    /**
     * @return threatScore
     */
    @JsonProperty("threatScore")
    public @Nullable Double getThreatScore() {
        return threatScore;
    }

    /**
     * @return threatSeverity
     */
    @JsonProperty("threatSeverity")
    public @Nullable SeverityType getThreatSeverity() {
        return threatSeverity;
    }

    /**
     * @return environmentalScore
     */
    @JsonProperty("environmentalScore")
    public @Nullable Double getEnvironmentalScore() {
        return environmentalScore;
    }

    /**
     * @return environmentalSeverity
     */
    @JsonProperty("environmentalSeverity")
    public @Nullable SeverityType getEnvironmentalSeverity() {
        return environmentalSeverity;
    }

    @Override
    public String toString() {
        // "pattern":
        // "^CVSS:4[.]0\/AV:[NALP]\/AC:[LH]\/AT:[NP]\/PR:[NLH]\/UI:[NPA]\/VC:[HLN]\/VI:[HLN]\/VA:[HLN]\/SC:[HLN]\/SI:[HLN]\/SA:[HLN]
        // TODO fix the string format
        StringBuilder v = new StringBuilder("CVSS:").append(version).append("/AV:")
                .append(attackVector == null ? "" : attackVector.value().charAt(0)).append("/AC:")
                .append(attackComplexity == null ? "" : attackComplexity.value().charAt(0)).append("/AT:")
                .append(attackRequirements == null ? "" : attackRequirements.value().charAt(0)).append("/PR:")
                .append(privilegesRequired == null ? "" : privilegesRequired.value().charAt(0)).append("/UI:")
                .append(userInteraction == null ? "" : userInteraction.value().charAt(0)).append("/VC:")
                .append(vulnConfidentialityImpact == null ? "" : vulnConfidentialityImpact.value().charAt(0))
                .append("/VI:").append(vulnIntegrityImpact == null ? "" : vulnIntegrityImpact.value().charAt(0))
                .append("/VA:").append(vulnAvailabilityImpact == null ? "" : vulnAvailabilityImpact.value().charAt(0))
                .append("/SC:")
                .append(subConfidentialityImpact == null ? "" : subConfidentialityImpact.value().charAt(0))
                .append("/SI:").append(subIntegrityImpact == null ? "" : subIntegrityImpact.value().charAt(0))
                .append("/SA:").append(subAvailabilityImpact == null ? "" : subAvailabilityImpact.value().charAt(0));
        // (\/E:[XAPU])?(\/CR:[XHML])?(\/IR:[XHML])?(\/AR:[XHML])?
        if (exploitMaturity != null) {
            v.append("/E:").append(
                    exploitMaturity == ExploitMaturityType.NOT_DEFINED ? "X" : exploitMaturity.value().charAt(0));
        }
        if (confidentialityRequirement != null) {
            v.append("/CR:").append(confidentialityRequirement == CiaRequirementType.NOT_DEFINED ? "X"
                    : confidentialityRequirement.value().charAt(0));
        }
        if (integrityRequirement != null) {
            v.append("/IR:").append(integrityRequirement == CiaRequirementType.NOT_DEFINED ? "X"
                    : integrityRequirement.value().charAt(0));
        }
        if (availabilityRequirement != null) {
            v.append("/AR:").append(availabilityRequirement == CiaRequirementType.NOT_DEFINED ? "X"
                    : availabilityRequirement.value().charAt(0));
        }
        // (\/MAV:[XNALP])?(\/MAC:[XLH])?(\/MAT:[XNP])?(\/MPR:[XNLH])?(\/MUI:[XNPA])?(\/MVC:[XNLH])?(\/MVI:[XNLH])?(\/MVA:[XNLH])?
        if (modifiedAttackVector != null) {
            v.append("/MAV:").append(modifiedAttackVector == ModifiedAttackVectorType.NOT_DEFINED ? "X"
                    : modifiedAttackVector.value().charAt(0));
        }
        if (modifiedAttackComplexity != null) {
            v.append("/MAC:").append(modifiedAttackComplexity == ModifiedAttackComplexityType.NOT_DEFINED ? "X"
                    : modifiedAttackComplexity.value().charAt(0));
        }
        if (modifiedAttackRequirements != null) {
            v.append("/MAT:").append(modifiedAttackRequirements == ModifiedAttackRequirementsType.NOT_DEFINED ? "X"
                    : modifiedAttackRequirements.value().charAt(0));
        }
        if (modifiedPrivilegesRequired != null) {
            v.append("/MPR:").append(modifiedPrivilegesRequired == ModifiedPrivilegesRequiredType.NOT_DEFINED ? "X"
                    : modifiedPrivilegesRequired.value().charAt(0));
        }
        if (modifiedUserInteraction != null) {
            v.append("/MUI:").append(modifiedUserInteraction == ModifiedUserInteractionType.NOT_DEFINED ? "X"
                    : modifiedUserInteraction.value().charAt(0));
        }
        if (modifiedVulnConfidentialityImpact != null) {
            v.append("/MVC:").append(modifiedVulnConfidentialityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedVulnConfidentialityImpact.value().charAt(0));
        }
        if (modifiedVulnIntegrityImpact != null) {
            v.append("/MVI:").append(modifiedVulnIntegrityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedVulnIntegrityImpact.value().charAt(0));
        }
        if (modifiedVulnAvailabilityImpact != null) {
            v.append("/MVA:").append(modifiedVulnAvailabilityImpact == ModifiedCiaType.NOT_DEFINED ? "X"
                    : modifiedVulnAvailabilityImpact.value().charAt(0));
        }
        // (\/MSC:[XNLH])?(\/MSI:[XNLHS])?(\/MSA:[XNLHS])?(\/S:[XNP])?(\/AU:[XNY])?(\/R:[XAUI])?
        if (modifiedSubConfidentialityImpact != null) {
            v.append("/MSC:").append(modifiedSubConfidentialityImpact == ModifiedSubCType.NOT_DEFINED ? "X"
                    : modifiedSubConfidentialityImpact.value().charAt(0));
        }
        if (modifiedSubIntegrityImpact != null) {
            v.append("/MSI:").append(modifiedSubIntegrityImpact == ModifiedSubIaType.NOT_DEFINED ? "X"
                    : modifiedSubIntegrityImpact.value().charAt(0));
        }
        if (modifiedSubAvailabilityImpact != null) {
            v.append("/MSA:").append(modifiedSubAvailabilityImpact == ModifiedSubIaType.NOT_DEFINED ? "X"
                    : modifiedSubAvailabilityImpact.value().charAt(0));
        }
        if (safety != null) {
            v.append("/S:").append(safety == SafetyType.NOT_DEFINED ? "X" : safety.value().charAt(0));
        }
        if (automatable != null) {
            v.append("/AU:").append(automatable == AutomatableType.NOT_DEFINED ? "X" : automatable.value().charAt(0));
        }
        if (recovery != null) {
            v.append("/R:").append(recovery.value().charAt(0));
        }
        // (\/V:[XDC])?(\/RE:[XLMH])?(\/U:(X|Clear|Green|Amber|Red))?$"
        if (valueDensity != null) {
            v.append("/V:").append(valueDensity == ValueDensityType.NOT_DEFINED ? "X" : valueDensity.value().charAt(0));
        }
        if (vulnerabilityResponseEffort != null) {
            v.append("/RE:").append(vulnerabilityResponseEffort == VulnerabilityResponseEffortType.NOT_DEFINED ? "X"
                    : vulnerabilityResponseEffort.value().charAt(0));
        }
        if (providerUrgency != null) {
            v.append("/U:");
            if (providerUrgency == ProviderUrgencyType.NOT_DEFINED) {
                v.append("X");
            } else {
                v.append(providerUrgency.value().charAt(0)).append(providerUrgency.value().substring(1).toLowerCase());
            }
        }
        return v.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (!(o instanceof CvssV4Data))
            return false;
        CvssV4Data that = (CvssV4Data) o;
        return version == that.version && Objects.equals(vectorString, that.vectorString)
                && attackVector == that.attackVector && attackComplexity == that.attackComplexity
                && attackRequirements == that.attackRequirements && privilegesRequired == that.privilegesRequired
                && userInteraction == that.userInteraction
                && vulnConfidentialityImpact == that.vulnConfidentialityImpact
                && vulnIntegrityImpact == that.vulnIntegrityImpact
                && vulnAvailabilityImpact == that.vulnAvailabilityImpact
                && subConfidentialityImpact == that.subConfidentialityImpact
                && subIntegrityImpact == that.subIntegrityImpact && subAvailabilityImpact == that.subAvailabilityImpact
                && exploitMaturity == that.exploitMaturity
                && confidentialityRequirement == that.confidentialityRequirement
                && integrityRequirement == that.integrityRequirement
                && availabilityRequirement == that.availabilityRequirement
                && modifiedAttackVector == that.modifiedAttackVector
                && modifiedAttackComplexity == that.modifiedAttackComplexity
                && modifiedAttackRequirements == that.modifiedAttackRequirements
                && modifiedPrivilegesRequired == that.modifiedPrivilegesRequired
                && modifiedUserInteraction == that.modifiedUserInteraction
                && modifiedVulnConfidentialityImpact == that.modifiedVulnConfidentialityImpact
                && modifiedVulnIntegrityImpact == that.modifiedVulnIntegrityImpact
                && modifiedVulnAvailabilityImpact == that.modifiedVulnAvailabilityImpact
                && modifiedSubConfidentialityImpact == that.modifiedSubConfidentialityImpact
                && modifiedSubIntegrityImpact == that.modifiedSubIntegrityImpact
                && modifiedSubAvailabilityImpact == that.modifiedSubAvailabilityImpact && safety == that.safety
                && automatable == that.automatable && recovery == that.recovery && valueDensity == that.valueDensity
                && vulnerabilityResponseEffort == that.vulnerabilityResponseEffort
                && providerUrgency == that.providerUrgency && Objects.equals(baseScore, that.baseScore)
                && baseSeverity == that.baseSeverity && Objects.equals(threatScore, that.threatScore)
                && threatSeverity == that.threatSeverity && Objects.equals(environmentalScore, that.environmentalScore)
                && environmentalSeverity == that.environmentalSeverity;
    }

    @Override
    public int hashCode() {
        return Objects.hash(version, vectorString, attackVector, attackComplexity, attackRequirements,
                privilegesRequired, userInteraction, vulnConfidentialityImpact, vulnIntegrityImpact,
                vulnAvailabilityImpact, subConfidentialityImpact, subIntegrityImpact, subAvailabilityImpact,
                exploitMaturity, confidentialityRequirement, integrityRequirement, availabilityRequirement,
                modifiedAttackVector, modifiedAttackComplexity, modifiedAttackRequirements, modifiedPrivilegesRequired,
                modifiedUserInteraction, modifiedVulnConfidentialityImpact, modifiedVulnIntegrityImpact,
                modifiedVulnAvailabilityImpact, modifiedSubConfidentialityImpact, modifiedSubIntegrityImpact,
                modifiedSubAvailabilityImpact, safety, automatable, recovery, valueDensity, vulnerabilityResponseEffort,
                providerUrgency, baseScore, baseSeverity, threatScore, threatSeverity, environmentalScore,
                environmentalSeverity);
    }

    /**
     * CVSS v4 Attack Complexity metric values.
     */
    public enum AttackComplexityType {
        /** High attack complexity. */
        HIGH("HIGH"),
        /** Low attack complexity. */
        LOW("LOW");

        private static final Map<String, AttackComplexityType> CONSTANTS = new HashMap<>();

        static {
            for (AttackComplexityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AttackComplexityType(String value) {
            this.value = value;
        }

        /**
         * Creates an AttackComplexityType from its string value.
         *
         * @param value the string value
         * @return the AttackComplexityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AttackComplexityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (AttackComplexityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            AttackComplexityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this attack complexity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Attack Vector metric values.
     */
    public enum AttackVectorType {
        /** Network attack vector. */
        NETWORK("NETWORK"),
        /** Adjacent network attack vector. */
        ADJACENT("ADJACENT"),
        /** Local attack vector. */
        LOCAL("LOCAL"),
        /** Physical attack vector. */
        PHYSICAL("PHYSICAL");

        private static final Map<String, AttackVectorType> CONSTANTS = new HashMap<>();

        static {
            for (AttackVectorType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AttackVectorType(String value) {
            this.value = value;
        }

        /**
         * Creates an AttackVectorType from its string value.
         *
         * @param value the string value
         * @return the AttackVectorType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AttackVectorType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (AttackVectorType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            AttackVectorType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this attack vector type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Confidentiality, Integrity, and Availability requirement values.
     */
    public enum CiaRequirementType {
        /** Low requirement. */
        LOW("LOW"),
        /** Medium requirement. */
        MEDIUM("MEDIUM"),
        /** High requirement. */
        HIGH("HIGH"),
        /** Not defined requirement. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, CiaRequirementType> CONSTANTS = new HashMap<>();

        static {
            for (CiaRequirementType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CiaRequirementType(String value) {
            this.value = value;
        }

        /**
         * Creates a CiaRequirementType from its string value.
         *
         * @param value the string value
         * @return the CiaRequirementType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static CiaRequirementType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (CiaRequirementType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            CiaRequirementType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this CIA requirement type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Confidentiality, Integrity, and Availability impact values.
     */
    public enum CiaType {
        /** No impact. */
        NONE("NONE"),
        /** Low impact. */
        LOW("LOW"),
        /** High impact. */
        HIGH("HIGH");

        private static final Map<String, CiaType> CONSTANTS = new HashMap<>();

        static {
            for (CiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        CiaType(String value) {
            this.value = value;
        }

        /**
         * Creates a CiaType from its string value.
         *
         * @param value the string value
         * @return the CiaType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static CiaType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (CiaType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            CiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this CIA type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Exploit Maturity metric values for threat scoring.
     */
    public enum ExploitMaturityType {
        /** Unreported exploit maturity. */
        UNREPORTED("UNREPORTED"),
        /** Proof-of-concept exploit exists. */
        PROOF_OF_CONCEPT("PROOF_OF_CONCEPT"),
        /** Attacked exploit maturity. */
        ATTACKED("ATTACKED"),
        /** Not defined exploit maturity. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ExploitMaturityType> CONSTANTS = new HashMap<>();

        static {
            for (ExploitMaturityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ExploitMaturityType(String value) {
            this.value = value;
        }

        /**
         * Creates an ExploitMaturityType from its string value.
         *
         * @param value the string value
         * @return the ExploitMaturityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ExploitMaturityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ExploitMaturityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ExploitMaturityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this exploit maturity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Modified Attack Complexity metric values for environmental scoring.
     */
    public enum ModifiedAttackComplexityType {
        /** High modified attack complexity. */
        HIGH("HIGH"),
        /** Low modified attack complexity. */
        LOW("LOW"),
        /** Not defined modified attack complexity. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedAttackComplexityType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedAttackComplexityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedAttackComplexityType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedAttackComplexityType from its string value.
         *
         * @param value the string value
         * @return the ModifiedAttackComplexityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedAttackComplexityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedAttackComplexityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedAttackComplexityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified attack complexity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Attack Requirements metric values.
     */
    public enum AttackRequirementsType {
        /** No attack requirements. */
        NONE("NONE"),
        /** Attack requirements present. */
        PRESENT("PRESENT");

        private static final Map<String, AttackRequirementsType> CONSTANTS = new HashMap<>();

        static {
            for (AttackRequirementsType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AttackRequirementsType(String value) {
            this.value = value;
        }

        /**
         * Creates an AttackRequirementsType from its string value.
         *
         * @param value the string value
         * @return the AttackRequirementsType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AttackRequirementsType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (AttackRequirementsType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            AttackRequirementsType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this attack requirements type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Modified Attack Requirements metric values for environmental scoring.
     */
    public enum ModifiedAttackRequirementsType {
        /** No modified attack requirements. */
        NONE("NONE"),
        /** Modified attack requirements present. */
        PRESENT("PRESENT"),
        /** Not defined modified attack requirements. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedAttackRequirementsType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedAttackRequirementsType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedAttackRequirementsType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedAttackRequirementsType from its string value.
         *
         * @param value the string value
         * @return the ModifiedAttackRequirementsType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedAttackRequirementsType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedAttackRequirementsType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedAttackRequirementsType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified attack requirements type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Modified Attack Vector metric values for environmental scoring.
     */
    public enum ModifiedAttackVectorType {
        /** Network modified attack vector. */
        NETWORK("NETWORK"),
        /** Adjacent network modified attack vector. */
        ADJACENT("ADJACENT"),
        /** Local modified attack vector. */
        LOCAL("LOCAL"),
        /** Physical modified attack vector. */
        PHYSICAL("PHYSICAL"),
        /** Not defined modified attack vector. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedAttackVectorType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedAttackVectorType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedAttackVectorType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedAttackVectorType from its string value.
         *
         * @param value the string value
         * @return the ModifiedAttackVectorType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedAttackVectorType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedAttackVectorType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedAttackVectorType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified attack vector type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Modified Confidentiality, Integrity, and Availability impact values for environmental scoring.
     */
    public enum ModifiedCiaType {
        /** No modified impact. */
        NONE("NONE"),
        /** Low modified impact. */
        LOW("LOW"),
        /** High modified impact. */
        HIGH("HIGH"),
        /** Not defined modified impact. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedCiaType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedCiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedCiaType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedCiaType from its string value.
         *
         * @param value the string value
         * @return the ModifiedCiaType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedCiaType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedCiaType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedCiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified CIA type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Modified Subsequent System Confidentiality impact metric values.
     */
    public enum ModifiedSubCType {
        /** Negligible impact. */
        NEGLIGIBLE("NEGLIGIBLE"),
        /** Low impact. */
        LOW("LOW"),
        /** High impact. */
        HIGH("HIGH"),
        /** Modified subsequent system confidentiality impact not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedSubCType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedSubCType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedSubCType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedSubCType from its string value.
         *
         * @param value the string value
         * @return the ModifiedSubCType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedSubCType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedSubCType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedSubCType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified subsequent system confidentiality type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Modified Subsequent System Integrity and Availability impact metric values.
     */
    public enum ModifiedSubIaType {
        /** Negligible impact. */
        NEGLIGIBLE("NEGLIGIBLE"),
        /** Low impact. */
        LOW("LOW"),
        /** High impact. */
        HIGH("HIGH"),
        /** Safety impact. */
        SAFETY("SAFETY"),
        /** Modified subsequent system integrity/availability impact not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedSubIaType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedSubIaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedSubIaType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedSubIaType from its string value.
         *
         * @param value the string value
         * @return the ModifiedSubIaType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedSubIaType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedSubIaType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedSubIaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified subsequent system integrity/availability type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Safety metric values.
     */
    public enum SafetyType {
        /** Negligible safety impact. */
        NEGLIGIBLE("NEGLIGIBLE"),
        /** Safety impact is present. */
        PRESENT("PRESENT"),
        /** Safety impact not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, SafetyType> CONSTANTS = new HashMap<>();

        static {
            for (SafetyType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        SafetyType(String value) {
            this.value = value;
        }

        /**
         * Creates a SafetyType from its string value.
         *
         * @param value the string value
         * @return the SafetyType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static SafetyType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (SafetyType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            SafetyType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this safety type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Automatable metric values.
     */
    public enum AutomatableType {
        /** Not automatable. */
        NO("NO"),
        /** Automatable. */
        YES("YES"),
        /** Automatable not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, AutomatableType> CONSTANTS = new HashMap<>();

        static {
            for (AutomatableType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        AutomatableType(String value) {
            this.value = value;
        }

        /**
         * Creates an AutomatableType from its string value.
         *
         * @param value the string value
         * @return the AutomatableType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static AutomatableType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (AutomatableType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            AutomatableType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this automatable type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Recovery metric values.
     */
    public enum RecoveryType {
        /** Automatic recovery. */
        AUTOMATIC("AUTOMATIC"),
        /** User recovery required. */
        USER("USER"),
        /** Irrecoverable. */
        IRRECOVERABLE("IRRECOVERABLE"),
        /** Recovery not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, RecoveryType> CONSTANTS = new HashMap<>();

        static {
            for (RecoveryType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        RecoveryType(String value) {
            this.value = value;
        }

        /**
         * Creates a RecoveryType from its string value.
         *
         * @param value the string value
         * @return the RecoveryType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static RecoveryType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (RecoveryType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            RecoveryType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this recovery type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Value Density metric values.
     */
    public enum ValueDensityType {
        /** Diffuse value density. */
        DIFFUSE("DIFFUSE"),
        /** Concentrated value density. */
        CONCENTRATED("CONCENTRATED"),
        /** Value density not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ValueDensityType> CONSTANTS = new HashMap<>();

        static {
            for (ValueDensityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ValueDensityType(String value) {
            this.value = value;
        }

        /**
         * Creates a ValueDensityType from its string value.
         *
         * @param value the string value
         * @return the ValueDensityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ValueDensityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ValueDensityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ValueDensityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this value density type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Vulnerability Response Effort metric values.
     */
    public enum VulnerabilityResponseEffortType {
        /** Low response effort. */
        LOW("LOW"),
        /** Moderate response effort. */
        MODERATE("MODERATE"),
        /** High response effort. */
        HIGH("HIGH"),
        /** Response effort not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, VulnerabilityResponseEffortType> CONSTANTS = new HashMap<>();

        static {
            for (VulnerabilityResponseEffortType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        VulnerabilityResponseEffortType(String value) {
            this.value = value;
        }

        /**
         * Creates a VulnerabilityResponseEffortType from its string value.
         *
         * @param value the string value
         * @return the VulnerabilityResponseEffortType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static VulnerabilityResponseEffortType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (VulnerabilityResponseEffortType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            VulnerabilityResponseEffortType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this vulnerability response effort type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Provider Urgency metric values.
     */
    public enum ProviderUrgencyType {
        /** Clear urgency. */
        CLEAR("CLEAR"),
        /** Green urgency. */
        GREEN("GREEN"),
        /** Amber urgency. */
        AMBER("AMBER"),
        /** Red urgency. */
        RED("RED"),
        /** Provider urgency not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ProviderUrgencyType> CONSTANTS = new HashMap<>();

        static {
            for (ProviderUrgencyType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ProviderUrgencyType(String value) {
            this.value = value;
        }

        /**
         * Creates a ProviderUrgencyType from its string value.
         *
         * @param value the string value
         * @return the ProviderUrgencyType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ProviderUrgencyType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ProviderUrgencyType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ProviderUrgencyType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this provider urgency type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }

    /**
     * CVSS v4 Modified Privileges Required metric values.
     */
    public enum ModifiedPrivilegesRequiredType {
        /** High privileges required. */
        HIGH("HIGH"),
        /** Low privileges required. */
        LOW("LOW"),
        /** No privileges required. */
        NONE("NONE"),
        /** Modified privileges required not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedPrivilegesRequiredType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedPrivilegesRequiredType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedPrivilegesRequiredType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedPrivilegesRequiredType from its string value.
         *
         * @param value the string value
         * @return the ModifiedPrivilegesRequiredType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedPrivilegesRequiredType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedPrivilegesRequiredType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedPrivilegesRequiredType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified privileges required type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Modified Vulnerability Confidentiality, Integrity, and Availability impact metric values.
     */
    public enum ModifiedVulnCiaType {
        /** No impact. */
        NONE("NONE"),
        /** Low impact. */
        LOW("LOW"),
        /** High impact. */
        HIGH("HIGH"),
        /** Modified vulnerability CIA impact not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedVulnCiaType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedVulnCiaType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedVulnCiaType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedVulnCiaType from its string value.
         *
         * @param value the string value
         * @return the ModifiedVulnCiaType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedVulnCiaType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedVulnCiaType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedVulnCiaType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified vulnerability CIA type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Modified User Interaction metric values.
     */
    public enum ModifiedUserInteractionType {
        /** No user interaction required. */
        NONE("NONE"),
        /** Passive user interaction required. */
        PASSIVE("PASSIVE"),
        /** Active user interaction required. */
        ACTIVE("ACTIVE"),
        /** Modified user interaction not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, ModifiedUserInteractionType> CONSTANTS = new HashMap<>();

        static {
            for (ModifiedUserInteractionType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        ModifiedUserInteractionType(String value) {
            this.value = value;
        }

        /**
         * Creates a ModifiedUserInteractionType from its string value.
         *
         * @param value the string value
         * @return the ModifiedUserInteractionType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static ModifiedUserInteractionType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (ModifiedUserInteractionType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            ModifiedUserInteractionType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this modified user interaction type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Privileges Required metric values.
     */
    public enum PrivilegesRequiredType {
        /** High privileges required. */
        HIGH("HIGH"),
        /** Low privileges required. */
        LOW("LOW"),
        /** No privileges required. */
        NONE("NONE");

        private static final Map<String, PrivilegesRequiredType> CONSTANTS = new HashMap<>();

        static {
            for (PrivilegesRequiredType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        PrivilegesRequiredType(String value) {
            this.value = value;
        }

        /**
         * Creates a PrivilegesRequiredType from its string value.
         *
         * @param value the string value
         * @return the PrivilegesRequiredType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static PrivilegesRequiredType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (PrivilegesRequiredType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            PrivilegesRequiredType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this privileges required type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Remediation Level metric values.
     */
    public enum RemediationLevelType {
        /** Official fix available. */
        OFFICIAL_FIX("OFFICIAL_FIX"),
        /** Temporary fix available. */
        TEMPORARY_FIX("TEMPORARY_FIX"),
        /** Workaround available. */
        WORKAROUND("WORKAROUND"),
        /** No remediation available. */
        UNAVAILABLE("UNAVAILABLE"),
        /** Remediation level not defined. */
        NOT_DEFINED("NOT_DEFINED");

        private static final Map<String, RemediationLevelType> CONSTANTS = new HashMap<>();

        static {
            for (RemediationLevelType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        RemediationLevelType(String value) {
            this.value = value;
        }

        /**
         * Creates a RemediationLevelType from its string value.
         *
         * @param value the string value
         * @return the RemediationLevelType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static RemediationLevelType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                if ("x".equalsIgnoreCase(value)) {
                    return NOT_DEFINED;
                }
                for (RemediationLevelType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            RemediationLevelType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this remediation level type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 Severity rating values.
     */
    public enum SeverityType {
        /** None severity. */
        NONE("NONE"),
        /** Low severity. */
        LOW("LOW"),
        /** Medium severity. */
        MEDIUM("MEDIUM"),
        /** High severity. */
        HIGH("HIGH"),
        /** Critical severity. */
        CRITICAL("CRITICAL");

        private static final Map<String, SeverityType> CONSTANTS = new HashMap<>();

        static {
            for (SeverityType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        SeverityType(String value) {
            this.value = value;
        }

        /**
         * Creates a SeverityType from its string value.
         *
         * @param value the string value
         * @return the SeverityType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static SeverityType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (SeverityType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            SeverityType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this severity type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS v4 User Interaction metric values.
     */
    public enum UserInteractionType {
        /** No user interaction required. */
        NONE("NONE"),
        /** Passive user interaction required. */
        PASSIVE("PASSIVE"),
        /** Active user interaction required. */
        ACTIVE("ACTIVE");

        private static final Map<String, UserInteractionType> CONSTANTS = new HashMap<>();

        static {
            for (UserInteractionType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        UserInteractionType(String value) {
            this.value = value;
        }

        /**
         * Creates a UserInteractionType from its string value.
         *
         * @param value the string value
         * @return the UserInteractionType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static UserInteractionType fromValue(String value) {
            // allow conversion from vector string
            if (value.length() == 1) {
                for (UserInteractionType t : values()) {
                    if (t.value.startsWith(value.toUpperCase())) {
                        return t;
                    }
                }
            }
            UserInteractionType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this user interaction type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

    /**
     * CVSS Version
     */
    public enum Version {
        /** CVSS version 4.0. */
        _4_0("4.0");

        private static final Map<String, Version> CONSTANTS = new HashMap<>();

        static {
            for (Version c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        Version(String value) {
            this.value = value;
        }

        /**
         * Creates a Version from its string value.
         *
         * @param value the string value
         * @return the Version enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static Version fromValue(String value) {
            Version constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this version.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

}
