/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonValue;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * Represents CVSS version 2.0 scoring data from the NVD API.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({"source", "type", "cvssData", "baseSeverity", "exploitabilityScore", "impactScore", "acInsufInfo",
        "obtainAllPrivilege", "obtainUserPrivilege", "obtainOtherPrivilege", "userInteractionRequired"})
@NullMarked
public class CvssV2 implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 7595837336051753457L;

    /**
     * Constructor taking only the required set of JSON properties as the minimal constructor for a CvssV2.
     *
     * @param source The source for the CvssV2 (required JSON property)
     * @param type The type for the CvssV2 (required JSON property)
     * @param cvssData The cvssData for the CvssV2 (required JSON property)
     */
    @JsonCreator
    public CvssV2(@JsonProperty("source") String source, @JsonProperty("type") Type type,
            @JsonProperty("cvssData") CvssV2Data cvssData) {
        this.source = source;
        this.type = type;
        this.cvssData = cvssData;
    }

    /**
     * Full constructor for CvssV2.
     *
     * @param source the source of the CVSS score
     * @param type the type (Primary or Secondary)
     * @param cvssData the CVSS v2 data
     * @param baseSeverity the base severity
     * @param exploitabilityScore the exploitability subscore
     * @param impactScore the impact subscore
     * @param acInsufInfo access complexity insufficient info flag
     * @param obtainAllPrivilege obtain all privilege flag
     * @param obtainUserPrivilege obtain user privilege flag
     * @param obtainOtherPrivilege obtain other privilege flag
     * @param userInteractionRequired user interaction required flag
     */
    public CvssV2(String source, Type type, CvssV2Data cvssData, @Nullable String baseSeverity,
            @Nullable Double exploitabilityScore, @Nullable Double impactScore, @Nullable Boolean acInsufInfo,
            @Nullable Boolean obtainAllPrivilege, @Nullable Boolean obtainUserPrivilege,
            @Nullable Boolean obtainOtherPrivilege, @Nullable Boolean userInteractionRequired) {
        this(source, type, cvssData);
        this.baseSeverity = baseSeverity;
        this.exploitabilityScore = exploitabilityScore;
        this.impactScore = impactScore;
        this.acInsufInfo = acInsufInfo;
        this.obtainAllPrivilege = obtainAllPrivilege;
        this.obtainUserPrivilege = obtainUserPrivilege;
        this.obtainOtherPrivilege = obtainOtherPrivilege;
        this.userInteractionRequired = userInteractionRequired;
    }

    /**
     * (Required)
     */
    @JsonProperty("source")
    private String source;
    /**
     * (Required)
     */
    @JsonProperty("type")
    private Type type;
    /**
     * JSON Schema for Common Vulnerability Scoring System version 2.0
     * <p>
     * (Required)
     */
    @JsonProperty("cvssData")
    private CvssV2Data cvssData;
    /**
     * The base severity - appears to be a mistake in the schema as this is duplicated in the CvssV2Data data and this
     * field is never populated by the NVD API.
     */
    @JsonProperty("baseSeverity")
    private @Nullable String baseSeverity;
    /**
     * CVSS subscore.
     */
    @JsonProperty("exploitabilityScore")
    @JsonPropertyDescription("CVSS subscore.")
    private @Nullable Double exploitabilityScore;
    /**
     * CVSS subscore.
     */
    @JsonProperty("impactScore")
    @JsonPropertyDescription("CVSS subscore.")
    private @Nullable Double impactScore;
    @JsonProperty("acInsufInfo")
    private @Nullable Boolean acInsufInfo;
    @JsonProperty("obtainAllPrivilege")
    private @Nullable Boolean obtainAllPrivilege;
    @JsonProperty("obtainUserPrivilege")
    private @Nullable Boolean obtainUserPrivilege;
    @JsonProperty("obtainOtherPrivilege")
    private @Nullable Boolean obtainOtherPrivilege;
    @JsonProperty("userInteractionRequired")
    private @Nullable Boolean userInteractionRequired;

    /**
     * (Required)
     *
     * @return source
     */
    @JsonProperty("source")
    public String getSource() {
        return source;
    }

    /**
     * (Required)
     *
     * @return type
     */
    @JsonProperty("type")
    public Type getType() {
        return type;
    }

    /**
     * JSON Schema for Common Vulnerability Scoring System version 2.0
     * <p>
     * (Required)
     *
     * @return cvssData
     */
    @JsonProperty("cvssData")
    public CvssV2Data getCvssData() {
        return cvssData;
    }

    /**
     * @return baseSeverity
     */
    @JsonProperty("baseSeverity")
    public @Nullable String getBaseSeverity() {
        return baseSeverity;
    }

    /**
     * CVSS subscore.
     *
     * @return exploitabilityScore
     */
    @JsonProperty("exploitabilityScore")
    public @Nullable Double getExploitabilityScore() {
        return exploitabilityScore;
    }

    /**
     * CVSS subscore.
     *
     * @return impactScore
     */
    @JsonProperty("impactScore")
    public @Nullable Double getImpactScore() {
        return impactScore;
    }

    /**
     * @return acInsufInfo
     */
    @JsonProperty("acInsufInfo")
    public @Nullable Boolean getAcInsufInfo() {
        return acInsufInfo;
    }

    /**
     * @return obtainAllPrivilege
     */
    @JsonProperty("obtainAllPrivilege")
    public @Nullable Boolean getObtainAllPrivilege() {
        return obtainAllPrivilege;
    }

    /**
     * @return obtainUserPrivilege
     */
    @JsonProperty("obtainUserPrivilege")
    public @Nullable Boolean getObtainUserPrivilege() {
        return obtainUserPrivilege;
    }

    /**
     * @return obtainOtherPrivilege
     */
    @JsonProperty("obtainOtherPrivilege")
    public @Nullable Boolean getObtainOtherPrivilege() {
        return obtainOtherPrivilege;
    }

    /**
     * @return userInteractionRequired
     */
    @JsonProperty("userInteractionRequired")
    public @Nullable Boolean getUserInteractionRequired() {
        return userInteractionRequired;
    }

    @Override
    public String toString() {
        return cvssData.getVectorString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        CvssV2 cvssV2 = (CvssV2) o;
        return Objects.equals(source, cvssV2.source) && type == cvssV2.type && Objects.equals(cvssData, cvssV2.cvssData)
                && Objects.equals(baseSeverity, cvssV2.baseSeverity)
                && Objects.equals(exploitabilityScore, cvssV2.exploitabilityScore)
                && Objects.equals(impactScore, cvssV2.impactScore) && Objects.equals(acInsufInfo, cvssV2.acInsufInfo)
                && Objects.equals(obtainAllPrivilege, cvssV2.obtainAllPrivilege)
                && Objects.equals(obtainUserPrivilege, cvssV2.obtainUserPrivilege)
                && Objects.equals(obtainOtherPrivilege, cvssV2.obtainOtherPrivilege)
                && Objects.equals(userInteractionRequired, cvssV2.userInteractionRequired);
    }

    @Override
    public int hashCode() {
        return Objects.hash(source, type, cvssData, baseSeverity, exploitabilityScore, impactScore, acInsufInfo,
                obtainAllPrivilege, obtainUserPrivilege, obtainOtherPrivilege, userInteractionRequired);
    }

    /**
     * Type of CVSS score (Primary or Secondary).
     */
    public enum Type {
        /** Primary CVSS score. */
        PRIMARY("Primary"),
        /** Secondary CVSS score. */
        SECONDARY("Secondary");

        private static final Map<String, Type> CONSTANTS = new HashMap<>();

        static {
            for (Type c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        Type(String value) {
            this.value = value;
        }

        /**
         * Creates a Type from its string value.
         *
         * @param value the string value
         * @return the Type enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static Type fromValue(String value) {
            Type constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }

    }

}
