/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonValue;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * Represents tags associated with a CVE from the NVD API.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonPropertyOrder({"sourceIdentifier", "tags"})
@NullMarked
public class CveTag implements Serializable {
    private static final long serialVersionUID = 6119071096772721680L;

    /**
     * The email address or UUID of the source that contributed the information
     */
    @JsonProperty("sourceIdentifier")
    @Nullable
    private String sourceIdentifier;

    @JsonProperty("tags")
    @Nullable
    private List<TagType> tags;

    /**
     * Default constructor.
     */
    public CveTag() {
    }

    /**
     * Constructor for CveTag.
     *
     * @param sourceIdentifier the source identifier
     * @param tags the list of tags
     */
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP2"}, justification = "I prefer to suppress these FindBugs warnings")
    public CveTag(@Nullable String sourceIdentifier, @Nullable List<TagType> tags) {
        this.sourceIdentifier = sourceIdentifier;
        this.tags = tags;
    }

    /**
     * @return sourceIdentifier
     */
    @JsonProperty("sourceIdentifier")
    public @Nullable String getSourceIdentifier() {
        return sourceIdentifier;
    }

    /**
     * @return tags
     */
    @JsonProperty("tags")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public @Nullable List<TagType> getTags() {
        return tags;
    }

    @Override
    public String toString() {
        return "CveTag{" + "sourceIdentifier='" + sourceIdentifier + '\'' + ", tags=" + tags + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;

        CveTag cveTag = (CveTag) o;
        return Objects.equals(sourceIdentifier, cveTag.sourceIdentifier) && Objects.equals(tags, cveTag.tags);
    }

    @Override
    public int hashCode() {
        return Objects.hash(sourceIdentifier, tags);
    }

    /**
     * Types of tags that can be applied to CVEs.
     */
    public enum TagType {
        /** Indicates the product was unsupported when the CVE was assigned. */
        UNSUPPORTED_WHEN_ASSIGNED("unsupported-when-assigned"),
        /** Indicates the vulnerability affects only hosted services. */
        EXCLUSIVELY_HOSTED_SERVICE("exclusively-hosted-service"),
        /** Indicates the CVE is disputed. */
        DISPUTED("disputed");

        private static final Map<String, TagType> CONSTANTS = new HashMap<>();

        static {
            for (TagType c : values()) {
                CONSTANTS.put(c.value, c);
            }
        }

        private final String value;

        TagType(String value) {
            this.value = value;
        }

        /**
         * Creates a TagType from its string value.
         *
         * @param value the string value
         * @return the TagType enum constant
         * @throws IllegalArgumentException if the value is not valid
         */
        @JsonCreator
        public static TagType fromValue(String value) {
            TagType constant = CONSTANTS.get(value);
            if (constant == null) {
                throw new IllegalArgumentException(value);
            } else {
                return constant;
            }
        }

        @Override
        public String toString() {
            return this.value;
        }

        /**
         * Gets the string value of this tag type.
         *
         * @return the string value
         */
        @JsonValue
        public String value() {
            return this.value;
        }
    }
}
