/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.nvd;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonPropertyDescription;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import org.jspecify.annotations.NullMarked;
import org.jspecify.annotations.Nullable;

import java.io.Serializable;
import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Objects;

/**
 * Represents a CVE item from the NVD API containing vulnerability details.
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"id", "sourceIdentifier", "published", "lastModified", "vulnStatus", "evaluatorComment",
        "evaluatorSolution", "evaluatorImpact", "cisaExploitAdd", "cisaActionDue", "cisaRequiredAction",
        "cisaVulnerabilityName", "cveTags", "descriptions", "vendorComments", "metrics", "weaknesses", "configurations",
        "references"})
@NullMarked
public class CveItem implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = -3429894394769351686L;
    /**
     * (Required)
     */
    @JsonProperty("id")
    private String id;
    @JsonProperty("sourceIdentifier")
    private @Nullable String sourceIdentifier;
    @JsonProperty("vulnStatus")
    private @Nullable String vulnStatus;
    /**
     * (Required)
     */
    @JsonProperty("published")
    @JsonFormat(pattern = "uuuu-MM-dd'T'HH:mm:ss.SSS", timezone = "UTC")
    private ZonedDateTime published;
    /**
     * (Required)
     */
    @JsonProperty("lastModified")
    @JsonFormat(pattern = "uuuu-MM-dd'T'HH:mm:ss.SSS", timezone = "UTC")
    private ZonedDateTime lastModified;
    @JsonProperty("evaluatorComment")
    private @Nullable String evaluatorComment;
    @JsonProperty("evaluatorSolution")
    private @Nullable String evaluatorSolution;
    @JsonProperty("evaluatorImpact")
    private @Nullable String evaluatorImpact;
    @JsonProperty("cisaExploitAdd")
    @JsonFormat(pattern = "yyyy-MM-dd")
    private @Nullable LocalDate cisaExploitAdd;
    @JsonProperty("cisaActionDue")
    @JsonFormat(pattern = "yyyy-MM-dd")
    private @Nullable LocalDate cisaActionDue;
    @JsonProperty("cisaRequiredAction")
    private @Nullable String cisaRequiredAction;
    @JsonProperty("cisaVulnerabilityName")
    private @Nullable String cisaVulnerabilityName;
    @JsonProperty("cveTags")
    private @Nullable List<CveTag> cveTags;
    /**
     * (Required)
     */
    @JsonProperty("descriptions")
    private List<LangString> descriptions;
    /**
     * (Required)
     */
    @JsonProperty("references")
    private List<Reference> references;
    /**
     * Metric scores for a vulnerability as found on NVD.
     */
    @JsonProperty("metrics")
    @JsonPropertyDescription("Metric scores for a vulnerability as found on NVD.")
    private @Nullable Metrics metrics;
    @JsonProperty("weaknesses")
    private @Nullable List<Weakness> weaknesses;
    @JsonProperty("configurations")
    private @Nullable List<Config> configurations;
    @JsonProperty("vendorComments")
    private @Nullable List<VendorComment> vendorComments;

    /**
     * Constructor taking only the required set of JSON properties as the minimal constructor for a CveItem.
     *
     * @param id The id of the cveItem data from the NVD API (required JSON property)
     * @param published The published of the cveItem data from the NVD API (required JSON property)
     * @param lastModified The lastModified of the cveItem data from the NVD API (required JSON property)
     * @param descriptions The descriptions of the cveItem data from the NVD API (required JSON property)
     * @param references The references of the cveItem data from the NVD API (required JSON property)
     */
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    @JsonCreator
    public CveItem(@JsonProperty("id") String id, @JsonProperty("published") ZonedDateTime published,
            @JsonProperty("lastModified") ZonedDateTime lastModified,
            @JsonProperty("descriptions") List<LangString> descriptions,
            @JsonProperty("references") List<Reference> references) {
        this.id = id;
        this.published = published;
        this.lastModified = lastModified;
        this.references = references;
        this.descriptions = descriptions;
    }

    /**
     * Full constructor for CveItem.
     *
     * @param id the CVE identifier
     * @param sourceIdentifier the source identifier
     * @param vulnStatus the vulnerability status
     * @param published the publication date
     * @param lastModified the last modified date
     * @param evaluatorComment evaluator comment
     * @param evaluatorSolution evaluator solution
     * @param evaluatorImpact evaluator impact assessment
     * @param cisaExploitAdd date added to CISA KEV catalog
     * @param cisaActionDue CISA required action due date
     * @param cisaRequiredAction CISA required action
     * @param cisaVulnerabilityName CISA vulnerability name
     * @param cveTags CVE tags
     * @param descriptions vulnerability descriptions
     * @param references vulnerability references
     * @param metrics CVSS metrics
     * @param weaknesses CWE weaknesses
     * @param configurations affected configurations
     * @param vendorComments vendor comments
     */
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public CveItem(String id, @Nullable String sourceIdentifier, @Nullable String vulnStatus, ZonedDateTime published,
            ZonedDateTime lastModified, @Nullable String evaluatorComment, @Nullable String evaluatorSolution,
            @Nullable String evaluatorImpact, @Nullable LocalDate cisaExploitAdd, @Nullable LocalDate cisaActionDue,
            @Nullable String cisaRequiredAction, @Nullable String cisaVulnerabilityName, @Nullable List<CveTag> cveTags,
            List<LangString> descriptions, List<Reference> references, @Nullable Metrics metrics,
            @Nullable List<Weakness> weaknesses, @Nullable List<Config> configurations,
            @Nullable List<VendorComment> vendorComments) {
        this(id, published, lastModified, descriptions, references);
        this.sourceIdentifier = sourceIdentifier;
        this.vulnStatus = vulnStatus;
        this.evaluatorComment = evaluatorComment;
        this.evaluatorSolution = evaluatorSolution;
        this.evaluatorImpact = evaluatorImpact;
        this.cisaExploitAdd = cisaExploitAdd;
        this.cisaActionDue = cisaActionDue;
        this.cisaRequiredAction = cisaRequiredAction;
        this.cisaVulnerabilityName = cisaVulnerabilityName;
        this.cveTags = cveTags;
        this.metrics = metrics;
        this.weaknesses = weaknesses;
        this.configurations = configurations;
        this.vendorComments = vendorComments;
    }

    /**
     * @return id
     */
    @JsonProperty("id")
    public String getId() {
        return id;
    }

    /**
     * @return sourceIdentifier
     */
    @JsonProperty("sourceIdentifier")
    public @Nullable String getSourceIdentifier() {
        return sourceIdentifier;
    }

    /**
     * @return vulnStatus
     */
    @JsonProperty("vulnStatus")
    public @Nullable String getVulnStatus() {
        return vulnStatus;
    }

    /**
     * @return published
     */
    @JsonProperty("published")
    public ZonedDateTime getPublished() {
        return published;
    }

    /**
     * @return lastModified
     */
    @JsonProperty("lastModified")
    public ZonedDateTime getLastModified() {
        return lastModified;
    }

    /**
     * @return evaluatorComment
     */
    @JsonProperty("evaluatorComment")
    public @Nullable String getEvaluatorComment() {
        return evaluatorComment;
    }

    /**
     * @return evaluatorSolution
     */
    @JsonProperty("evaluatorSolution")
    public @Nullable String getEvaluatorSolution() {
        return evaluatorSolution;
    }

    /**
     * @return evaluatorImpact
     */
    @JsonProperty("evaluatorImpact")
    public @Nullable String getEvaluatorImpact() {
        return evaluatorImpact;
    }

    /**
     * @return cisaExploitAdd
     */
    @JsonProperty("cisaExploitAdd")
    public @Nullable LocalDate getCisaExploitAdd() {
        return cisaExploitAdd;
    }

    /**
     * @return cisaActionDue
     */
    @JsonProperty("cisaActionDue")
    public @Nullable LocalDate getCisaActionDue() {
        return cisaActionDue;
    }

    /**
     * @return cisaRequiredAction
     */
    @JsonProperty("cisaRequiredAction")
    public @Nullable String getCisaRequiredAction() {
        return cisaRequiredAction;
    }

    /**
     * @return cisaVulnerabilityName
     */
    @JsonProperty("cisaVulnerabilityName")
    public @Nullable String getCisaVulnerabilityName() {
        return cisaVulnerabilityName;
    }

    /**
     * @return cveTags
     */
    @JsonProperty("cveTags")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public @Nullable List<CveTag> getCveTags() {
        return cveTags;
    }

    /**
     * (Required)
     *
     * @return descriptions
     */
    @JsonProperty("descriptions")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public List<LangString> getDescriptions() {
        return descriptions;
    }

    /**
     * (Required)
     *
     * @return references
     */
    @JsonProperty("references")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public List<Reference> getReferences() {
        return references;
    }

    /**
     * Metric scores for a vulnerability as found on NVD.
     *
     * @return metrics
     */
    @JsonProperty("metrics")
    public @Nullable Metrics getMetrics() {
        return metrics;
    }

    /**
     * @return weaknesses
     */
    @JsonProperty("weaknesses")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public @Nullable List<Weakness> getWeaknesses() {
        return weaknesses;
    }

    /**
     * @return configurations
     */
    @JsonProperty("configurations")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public @Nullable List<Config> getConfigurations() {
        return configurations;
    }

    /**
     * @return vendorComments
     */
    @JsonProperty("vendorComments")
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public @Nullable List<VendorComment> getVendorComments() {
        return vendorComments;
    }

    @Override
    public String toString() {
        return "CveItem{" + "id='" + id + '\'' + ", sourceIdentifier='" + sourceIdentifier + '\'' + ", vulnStatus='"
                + vulnStatus + '\'' + ", published=" + published + ", lastModified=" + lastModified
                + ", evaluatorComment='" + evaluatorComment + '\'' + ", evaluatorSolution='" + evaluatorSolution + '\''
                + ", evaluatorImpact='" + evaluatorImpact + '\'' + ", cisaExploitAdd=" + cisaExploitAdd
                + ", cisaActionDue=" + cisaActionDue + ", cisaRequiredAction='" + cisaRequiredAction + '\''
                + ", cisaVulnerabilityName='" + cisaVulnerabilityName + '\'' + ", cveTags=" + cveTags
                + ", descriptions=" + descriptions + ", references=" + references + ", metrics=" + metrics
                + ", weaknesses=" + weaknesses + ", configurations=" + configurations + ", vendorComments="
                + vendorComments + '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        CveItem cveItem = (CveItem) o;
        return Objects.equals(id, cveItem.id) && Objects.equals(sourceIdentifier, cveItem.sourceIdentifier)
                && Objects.equals(vulnStatus, cveItem.vulnStatus) && Objects.equals(published, cveItem.published)
                && Objects.equals(lastModified, cveItem.lastModified)
                && Objects.equals(evaluatorComment, cveItem.evaluatorComment)
                && Objects.equals(evaluatorSolution, cveItem.evaluatorSolution)
                && Objects.equals(evaluatorImpact, cveItem.evaluatorImpact)
                && Objects.equals(cisaExploitAdd, cveItem.cisaExploitAdd)
                && Objects.equals(cisaActionDue, cveItem.cisaActionDue)
                && Objects.equals(cisaRequiredAction, cveItem.cisaRequiredAction)
                && Objects.equals(cisaVulnerabilityName, cveItem.cisaVulnerabilityName)
                && Objects.equals(cveTags, cveItem.cveTags) && Objects.equals(descriptions, cveItem.descriptions)
                && Objects.equals(references, cveItem.references) && Objects.equals(metrics, cveItem.metrics)
                && Objects.equals(weaknesses, cveItem.weaknesses)
                && Objects.equals(configurations, cveItem.configurations)
                && Objects.equals(vendorComments, cveItem.vendorComments);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, sourceIdentifier, vulnStatus, published, lastModified, evaluatorComment,
                evaluatorSolution, evaluatorImpact, cisaExploitAdd, cisaActionDue, cisaRequiredAction,
                cisaVulnerabilityName, cveTags, descriptions, references, metrics, weaknesses, configurations,
                vendorComments);
    }
}
