/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.kev;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;

import java.io.Serializable;
import java.time.ZonedDateTime;
import java.util.List;

/**
 * Represents the CISA Known Exploited Vulnerabilities Catalog.
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"cve", "epss", "percentile"})
public class KevCatalog implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 3682701556631237639L;
    @JsonProperty("vulnerabilities")
    List<KevItem> vulnerabilities;
    @JsonProperty("title")
    private String title;
    @JsonProperty("catalogVersion")
    private String catalogVersion;
    @JsonFormat(pattern = "uuuu-MM-dd'T'HH:mm:ss[.[SSSSSSSSS][SSSSSSSS][SSSSSSS][SSSSSS][SSSSS][SSSS][SSS][SS][S]]X", timezone = "UTC")
    @JsonProperty("dateReleased")
    private ZonedDateTime dateReleased;
    @JsonProperty("count")
    private int count;

    /**
     * Returns the list of known exploited vulnerabilities.
     *
     * @return the vulnerabilities
     */
    @SuppressFBWarnings(value = {"EI_EXPOSE_REP"}, justification = "I prefer to suppress these FindBugs warnings")
    public List<KevItem> getVulnerabilities() {
        return vulnerabilities;
    }

    /**
     * Returns the catalog title.
     *
     * @return the title
     */
    public String getTitle() {
        return title;
    }

    /**
     * Returns the catalog version.
     *
     * @return the catalog version
     */
    public String getCatalogVersion() {
        return catalogVersion;
    }

    /**
     * Returns the date the catalog was released.
     *
     * @return the release date
     */
    @JsonFormat(pattern = "uuuu-MM-dd'T'HH:mm:ssX", timezone = "UTC")
    public ZonedDateTime getDateReleased() {
        return dateReleased;
    }

    /**
     * Returns the count of vulnerabilities in the catalog.
     *
     * @return the vulnerability count
     */
    public int getCount() {
        return count;
    }
}
