/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.ghsa;

import org.apache.hc.client5.http.impl.async.CloseableHttpAsyncClient;
import org.apache.hc.client5.http.async.methods.SimpleBody;
import org.apache.hc.client5.http.async.methods.SimpleHttpRequest;
import org.apache.hc.client5.http.async.methods.SimpleHttpResponse;
import org.apache.hc.client5.http.async.methods.SimpleRequestProducer;
import org.apache.hc.client5.http.async.methods.SimpleResponseConsumer;
import org.apache.hc.core5.concurrent.FutureCallback;
import org.apache.hc.core5.http.Header;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Closeable;
import java.io.IOException;
import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.*;
import java.util.function.Supplier;

/**
 * A thin wrapper around CloseableHttpAsyncClient that:
 * <ul>
 * <li>Limits concurrency (to avoid secondary limits)</li>
 * <li>Throttles requests to a maxRequestsPerMinute</li>
 * <li>Tracks primary rate limit headers</li>
 * <li>Applies GitHub's recommended behavior on 403/429 (incl. secondary limits)</li>
 * </ul>
 * All retries are best-effort and you should still handle failures at the caller.
 */
public final class GithubRateLimitedAsyncClient implements Closeable {

    /**
     * Reference to the logger.
     */
    private static final Logger LOG = LoggerFactory.getLogger(GithubRateLimitedAsyncClient.class);

    private final CloseableHttpAsyncClient delegate;
    private final ScheduledExecutorService scheduler;
    private final Semaphore concurrency;

    private final int maxSecondaryRetries;
    private final long minDelayBetweenRequestsMs;

    // Shared rate-limit state
    private final Object lock = new Object();
    private long remaining = Long.MAX_VALUE; // x-ratelimit-remaining
    private long resetEpochSeconds = 0L; // x-ratelimit-reset
    private long nextAllowedRequestTimeMs = 0L;

    /**
     * Create a new rate-limited GitHub async client.
     *
     * @param delegate The underlying HTTP async client
     * @param maxConcurrentRequests Max concurrent requests to allow
     * @param maxRequestsPerMinute Max requests per minute to allow
     * @param maxSecondaryRetries Max retries on secondary rate limit errors (403/429)
     */
    public GithubRateLimitedAsyncClient(CloseableHttpAsyncClient delegate, int maxConcurrentRequests,
            int maxRequestsPerMinute, int maxSecondaryRetries) {

        this.delegate = Objects.requireNonNull(delegate, "delegate");
        this.concurrency = new Semaphore(Math.max(1, maxConcurrentRequests));
        this.maxSecondaryRetries = Math.max(0, maxSecondaryRetries);
        this.minDelayBetweenRequestsMs = maxRequestsPerMinute <= 0 ? 0L : (60_000L / maxRequestsPerMinute);

        this.scheduler = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "github-rate-limit-scheduler");
            t.setDaemon(true);
            return t;
        });
    }

    /**
     * Execute a GitHub request with rate limiting and retry on 403/429.
     * <p>
     * requestSupplier MUST return a fresh SimpleHttpRequest each time it is called. This avoids issues with replaying
     * consumed request bodies.
     *
     * @param requestSupplier Supplier that creates a fresh {@link SimpleHttpRequest} for each attempt
     * @return a {@link CompletableFuture} that completes with the HTTP response or exceptionally on failure
     */
    public CompletableFuture<SimpleHttpResponse> execute(Supplier<SimpleHttpRequest> requestSupplier) {
        Objects.requireNonNull(requestSupplier, "requestSupplier must not be null");
        CompletableFuture<SimpleHttpResponse> future = new CompletableFuture<>();
        scheduleAttempt(requestSupplier, future, 0);
        return future;
    }

    private void scheduleAttempt(Supplier<SimpleHttpRequest> requestSupplier,
            CompletableFuture<SimpleHttpResponse> future, int attempt) {

        long delayMs;

        synchronized (lock) {
            long nowMs = System.currentTimeMillis();
            long nowSec = nowMs / 1000L;

            // Reset the primary window when we've passed reset time
            if (resetEpochSeconds > 0 && nowSec >= resetEpochSeconds) {
                remaining = Long.MAX_VALUE;
                resetEpochSeconds = 0L;
            }

            long primaryWaitMs = 0L;
            if (remaining <= 0 && resetEpochSeconds > 0) {
                long waitSec = resetEpochSeconds - nowSec;
                if (waitSec > 0) {
                    primaryWaitMs = waitSec * 1000L;
                }
            }

            long smoothingWaitMs = Math.max(0, nextAllowedRequestTimeMs - nowMs);

            delayMs = Math.max(primaryWaitMs, smoothingWaitMs);

            // Consume one "slot" from our view of the primary rate limit
            if (remaining > 0) {
                remaining--;
            }

            // Push nextAllowedRequestTimeMs forward to enforce maxRequestsPerMinute
            if (minDelayBetweenRequestsMs > 0) {
                long sendTime = nowMs + delayMs;
                nextAllowedRequestTimeMs = sendTime + minDelayBetweenRequestsMs;
            }
        }

        if (future.isDone()) {
            return;
        }

        try {
            scheduler.schedule(() -> doExecute(requestSupplier, future, attempt), delayMs, TimeUnit.MILLISECONDS);
        } catch (RejectedExecutionException e) {
            future.completeExceptionally(new CancellationException("Client closed"));
        }
    }

    private void doExecute(Supplier<SimpleHttpRequest> requestSupplier, CompletableFuture<SimpleHttpResponse> future,
            int attempt) {

        if (future.isDone()) {
            return;
        }

        try {
            concurrency.acquire();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            future.completeExceptionally(e);
            return;
        }

        SimpleHttpRequest request;
        try {
            request = requestSupplier.get();
        } catch (Throwable t) {
            concurrency.release();
            future.completeExceptionally(t);
            return;
        }

        delegate.execute(SimpleRequestProducer.create(request), SimpleResponseConsumer.create(),
                new FutureCallback<SimpleHttpResponse>() {
                    @Override
                    public void completed(SimpleHttpResponse response) {
                        try {
                            handleCompleted(response, requestSupplier, future, attempt);
                        }
                        finally {
                            concurrency.release();
                        }
                    }

                    @Override
                    public void failed(Exception ex) {
                        try {
                            future.completeExceptionally(ex);
                        }
                        finally {
                            concurrency.release();
                        }
                    }

                    @Override
                    public void cancelled() {
                        try {
                            future.cancel(true);
                        }
                        finally {
                            concurrency.release();
                        }
                    }
                });
    }

    private void handleCompleted(SimpleHttpResponse response, Supplier<SimpleHttpRequest> requestSupplier,
            CompletableFuture<SimpleHttpResponse> future, int attempt) {

        updatePrimaryRateLimitFromHeaders(response);

        int status = response.getCode();

        boolean rateLimitedStatus = isRateLimited(response, status);
        boolean canRetry = attempt < maxSecondaryRetries;

        if (rateLimitedStatus && canRetry) {
            long backoffMs = computeSecondaryBackoffMillis(response, attempt);
            if (backoffMs > 0) {
                // Schedule another attempt with exponential backoff
                try {
                    scheduler.schedule(() -> scheduleAttempt(requestSupplier, future, attempt + 1), backoffMs,
                            TimeUnit.MILLISECONDS);
                    // log retry wait time in human readable format
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Request was rate limited (HTTP {}), retrying attempt {} in {} ms", status,
                                attempt + 1, backoffMs);
                    }
                } catch (RejectedExecutionException e) {
                    future.completeExceptionally(new CancellationException("Client closed"));
                }
                return;
            }
        }

        // Either success or we've exhausted retries / not a rate-limit error
        if (rateLimitedStatus) {
            String message = extractMessage(response);
            future.completeExceptionally(
                    new IOException("GitHub rate limited the request (HTTP " + status + "): " + message));
        } else {
            future.complete(response);
        }
    }

    /**
     * Determines if a response indicates rate limiting. 429 is always rate-limited. 403 is only considered rate-limited
     * if there's evidence (Retry-After header, primary limit exhausted, or message indicates secondary rate limit).
     *
     * @param response the HTTP response
     * @param status the HTTP status code
     * @return true if the response indicates rate limiting
     */
    private boolean isRateLimited(SimpleHttpResponse response, int status) {
        if (status == 429) {
            return true;
        }
        if (status != 403) {
            return false;
        }

        // 403 is only a rate limit if there's evidence
        Header retryAfter = response.getFirstHeader("Retry-After");
        if (retryAfter != null) {
            return true;
        }

        Header remainingHeader = response.getFirstHeader("X-RateLimit-Remaining");
        Header resetHeader = response.getFirstHeader("X-RateLimit-Reset");
        if (remainingHeader != null && "0".equals(remainingHeader.getValue()) && resetHeader != null) {
            return true;
        }

        // Check message body for secondary rate limit indicators
        String message = extractMessage(response);
        String lowerMessage = message.toLowerCase();
        if (lowerMessage.contains("secondary rate limit") || lowerMessage.contains("wait a few minutes") || lowerMessage
                .contains("https://docs.github.com/en/site-policy/github-terms/github-terms-of-service")) {
            return true;
        }

        return false;
    }

    private void updatePrimaryRateLimitFromHeaders(SimpleHttpResponse response) {
        Header remainingHeader = response.getFirstHeader("X-RateLimit-Remaining");
        Header resetHeader = response.getFirstHeader("X-RateLimit-Reset");

        if (remainingHeader == null && resetHeader == null) {
            return;
        }

        synchronized (lock) {
            if (remainingHeader != null) {
                try {
                    remaining = Long.parseLong(remainingHeader.getValue());
                    LOG.debug("Rate limit remaining: {}", remaining);
                } catch (NumberFormatException ignore) {
                    // Keep old value
                }
            }
            if (resetHeader != null) {
                try {
                    resetEpochSeconds = Long.parseLong(resetHeader.getValue());
                    if (LOG.isDebugEnabled()) {
                        Instant resetInstant = Instant.ofEpochSecond(resetEpochSeconds);
                        LOG.debug("Rate limit resets in {} seconds",
                                Math.max(0, resetEpochSeconds - Instant.now().getEpochSecond()));
                    }
                } catch (NumberFormatException ignore) {
                    // Keep old value
                }
            }
        }
    }

    /**
     * Secondary rate limit handling based on GitHub docs:
     * <ul>
     * <li>Prefer Retry-After if present.</li>
     * <li>If x-ratelimit-remaining == 0 and reset is present, wait until reset.</li>
     * <li>Else wait at least 60 seconds.</li>
     * <li>Apply simple exponential backoff per attempt.</li>
     * </ul>
     *
     * @param response the HTTP response
     * @param attempt the current retry attempt number
     * @return the backoff delay in milliseconds
     */
    private long computeSecondaryBackoffMillis(SimpleHttpResponse response, int attempt) {
        long baseBackoffSec = 60; // "wait for at least one minute"
        Header retryAfter = response.getFirstHeader("Retry-After");
        Header remainingHeader = response.getFirstHeader("X-RateLimit-Remaining");
        Header resetHeader = response.getFirstHeader("X-RateLimit-Reset");

        if (retryAfter != null) {
            try {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Using Retry-After header for secondary rate limit backoff: {} seconds",
                            retryAfter.getValue());
                }
                baseBackoffSec = Long.parseLong(retryAfter.getValue());
            } catch (NumberFormatException ignore) {
                // fall back to default
            }
        } else if (remainingHeader != null && "0".equals(remainingHeader.getValue()) && resetHeader != null) {

            try {
                long reset = Long.parseLong(resetHeader.getValue());
                long now = Instant.now().getEpochSecond();
                long diff = reset - now;
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Using X-RateLimit-Reset header for secondary rate limit backoff: {} seconds", diff);
                }
                if (diff > 0) {
                    baseBackoffSec = diff;
                }
            } catch (NumberFormatException ignore) {
                // fall back to default
            }
        }

        long factor = 1L << attempt; // 1,2,4,...
        return baseBackoffSec * factor * 1000L;
    }

    private String extractMessage(SimpleHttpResponse response) {
        SimpleBody body = response.getBody();
        if (body == null) {
            return "";
        }
        try {
            return body.getBodyText();
        } catch (UnsupportedOperationException e) {
            return "";
        }
    }

    @Override
    public void close() throws IOException {
        scheduler.shutdownNow(); // Cancel pending tasks
        try {
            if (!scheduler.awaitTermination(5, TimeUnit.SECONDS)) {
                LOG.warn("Scheduler did not terminate within timeout");
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOG.warn("Interrupted while waiting for scheduler termination", e);
        }
        delegate.close();
    }
}
