/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.ghsa;

import com.fasterxml.jackson.annotation.JsonFormat;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;

import java.io.Serializable;
import java.util.Objects;

/**
 * Github Security Advisory EPSS Score
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonPropertyOrder({"percentage", "percentile"})
public class Epss implements Serializable {

    /**
     * Serialization version UID.
     */
    private static final long serialVersionUID = 1L;

    /**
     * The percentage of vulnerabilities with a lower EPSS score (0-100).
     */
    @JsonProperty("percentage")
    @JsonFormat(shape = JsonFormat.Shape.NUMBER)
    Float percentage;

    /**
     * The EPSS score (0.0-1.0).
     */
    @JsonProperty("percentile")
    @JsonFormat(shape = JsonFormat.Shape.NUMBER)
    Float percentile;

    /**
     * The probability that the vulnerability will be exploited in the next 30 days (0.0-1.0).
     *
     * @return the EPSS score (0.0-1.0).
     */
    public Float getPercentile() {
        return percentile;
    }

    /**
     * Sets the EPSS score (0.0-1.0).
     *
     * @param percentile the EPSS score (0.0-1.0).
     */
    public void setPercentile(Float percentile) {
        this.percentile = percentile;
    }

    /**
     * Returns the percentage of vulnerabilities with a lower EPSS score (0-100).
     *
     * @return the percentage of vulnerabilities with a lower EPSS score (0-100).
     */
    public Float getPercentage() {
        return percentage;
    }

    /**
     * Sets the percentage of vulnerabilities with a lower EPSS score (0-100).
     *
     * @param percentage the percentage of vulnerabilities with a lower EPSS score (0-100).
     */
    public void setPercentage(Float percentage) {
        this.percentage = percentage;
    }

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof Epss))
            return false;
        Epss epss = (Epss) o;
        return Objects.equals(percentage, epss.percentage) && Objects.equals(percentile, epss.percentile);
    }

    @Override
    public int hashCode() {
        return Objects.hash(percentage, percentile);
    }

    @Override
    public String toString() {
        return String.format("Epss{percentage=%.5f, percentile=%.5f}", percentage != null ? percentage : 0.0f,
                percentile != null ? percentile : 0.0f);
    }
}
