/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client.epss;

import io.github.jeremylong.openvulnerability.client.DataFeed;
import io.github.jeremylong.openvulnerability.client.HttpClientSupplier;
import org.apache.hc.client5.http.classic.methods.HttpGet;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;

import java.io.IOException;
import java.util.List;

/**
 * Data Feed for the Exploit Prediction Scoring System (EPSS) scores from the Forum of Incident Response and Security
 * Teams (FIRST)
 *
 * @see <a href="https://www.first.org/epss/data_stats">https://www.first.org/epss/data_stats</a>
 */
public class EpssDataFeed implements DataFeed<List<EpssItem>> {
    private final static String DEFAULT_LOCATION = "https://epss.cyentia.com/epss_scores-current.csv.gz";

    private final HttpClientSupplier httpClientSupplier;
    private final String downloadUrl;

    /**
     * Constructs a new EpssDataFeed with the default download location.
     */
    public EpssDataFeed() {
        this(DEFAULT_LOCATION);
    }

    /**
     * Constructs a new EpssDataFeed with a custom download URL.
     *
     * @param downloadUrl the URL to download EPSS data from
     */
    public EpssDataFeed(String downloadUrl) {
        this(downloadUrl, null);
    }

    /**
     * Constructs a new EpssDataFeed with a custom download URL and HTTP client supplier.
     *
     * @param downloadUrl the URL to download EPSS data from
     * @param httpClientSupplier the HTTP client supplier to use, or null for the default
     */
    public EpssDataFeed(String downloadUrl, HttpClientSupplier httpClientSupplier) {
        this.downloadUrl = downloadUrl;
        this.httpClientSupplier = httpClientSupplier != null ? httpClientSupplier : HttpClientSupplier.getDefault();
    }

    /**
     * Downloads the EPSS data feed.
     *
     * @return a list of EPSS items, or null if the download fails
     */
    @Override
    public List<EpssItem> download() {
        List<EpssItem> list = null;
        HttpGet request = new HttpGet(downloadUrl);
        try (CloseableHttpClient client = httpClientSupplier.get()) {
            list = client.execute(request, new EpssResponseHandler());
        } catch (IOException e) {
            e.printStackTrace();
        }
        return list;
    }
}
