/*
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 * SPDX-License-Identifier: Apache-2.0
 * Copyright (c) 2023-2025 Jeremy Long. All Rights Reserved.
 */
package io.github.jeremylong.openvulnerability.client;

import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.Iterator;

/**
 * Interface for iterating over paged data sources.
 *
 * @param <T> the type of data in the source
 */
public interface PagedDataSource<T> extends AutoCloseable, Iterator<Collection<T>> {

    /**
     * Cleanup allocated resources.
     *
     * @throws Exception thrown if there is a problem.
     */
    @Override
    void close() throws Exception;

    /**
     * Only available after the first call to `next()`; returns the total number of records that will be available.
     *
     * @return the total number of records that will be returned
     */
    int getTotalAvailable();

    /**
     * Returns the last HTTP Status Code received.
     *
     * @return the last HTTP Status Code received.
     */
    int getLastStatusCode();

    /**
     * Returns <code>true</code> if there are more records available; otherwise <code>false</code>.
     *
     * @return <code>true</code> if there are more records available; otherwise <code>false</code>.
     */
    @Override
    boolean hasNext();

    /**
     * Returns the next collection of vulnerability data.
     *
     * @return a collection of vulnerability data.
     */
    @Override
    Collection<T> next();

    /**
     * Returns the latest updated date.
     *
     * @return the latest updated date
     */
    ZonedDateTime getLastUpdated();

}
