/*
 * Decompiled with CFR 0.152.
 */
package io.github.jbellis.jvector.graph.disk;

import io.github.jbellis.jvector.disk.RandomAccessWriter;
import io.github.jbellis.jvector.graph.ImmutableGraphIndex;
import io.github.jbellis.jvector.graph.disk.NodeRecordTask;
import io.github.jbellis.jvector.graph.disk.OrdinalMapper;
import io.github.jbellis.jvector.graph.disk.feature.Feature;
import io.github.jbellis.jvector.graph.disk.feature.FeatureId;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.AsynchronousFileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.IntFunction;

class ParallelGraphWriter
implements AutoCloseable {
    private final RandomAccessWriter writer;
    private final ImmutableGraphIndex graph;
    private final ExecutorService executor;
    private final ThreadLocal<ImmutableGraphIndex.View> viewPerThread;
    private final ThreadLocal<ByteBuffer> bufferPerThread;
    private final CopyOnWriteArrayList<ImmutableGraphIndex.View> allViews = new CopyOnWriteArrayList();
    private final int recordSize;
    private final Path filePath;
    private final int taskMultiplier;
    private static final AtomicInteger threadCounter = new AtomicInteger(0);

    public ParallelGraphWriter(RandomAccessWriter writer, ImmutableGraphIndex graph, List<Feature> inlineFeatures, Config config, Path filePath) {
        this.writer = writer;
        this.graph = graph;
        this.filePath = Objects.requireNonNull(filePath);
        this.taskMultiplier = config.taskMultiplier;
        this.executor = Executors.newFixedThreadPool(config.workerThreads, r -> {
            Thread t = new Thread(r);
            t.setName("ParallelGraphWriter-Worker-" + threadCounter.getAndIncrement());
            t.setDaemon(false);
            return t;
        });
        this.recordSize = 4 + inlineFeatures.stream().mapToInt(Feature::featureSize).sum() + 4 + graph.getDegree(0) * 4;
        this.viewPerThread = ThreadLocal.withInitial(() -> {
            ImmutableGraphIndex.View view = graph.getView();
            this.allViews.add(view);
            return view;
        });
        int bufferSize = this.recordSize;
        boolean useDirect = config.useDirectBuffers;
        this.bufferPerThread = ThreadLocal.withInitial(() -> {
            ByteBuffer buffer = useDirect ? ByteBuffer.allocateDirect(bufferSize) : ByteBuffer.allocate(bufferSize);
            buffer.order(ByteOrder.BIG_ENDIAN);
            return buffer;
        });
    }

    public void writeL0Records(OrdinalMapper ordinalMapper, List<Feature> inlineFeatures, Map<FeatureId, IntFunction<Feature.State>> featureStateSuppliers, long baseOffset) throws IOException {
        int maxOrdinal = ordinalMapper.maxOrdinal();
        int totalOrdinals = maxOrdinal + 1;
        int numCores = Runtime.getRuntime().availableProcessors();
        int numTasks = Math.max(1, Math.min(numCores * this.taskMultiplier, totalOrdinals));
        int ordinalsPerTask = (totalOrdinals + numTasks - 1) / numTasks;
        HashSet<StandardOpenOption> opts = new HashSet<StandardOpenOption>();
        opts.add(StandardOpenOption.WRITE);
        opts.add(StandardOpenOption.READ);
        try (AsynchronousFileChannel channel = AsynchronousFileChannel.open(this.filePath, opts, null, new FileAttribute[0]);){
            ArrayList<Future<Void>> taskFutures = new ArrayList<Future<Void>>(numTasks);
            for (int i = 0; i < numTasks; ++i) {
                int n = i * ordinalsPerTask;
                int endOrdinal = Math.min(n + ordinalsPerTask, totalOrdinals);
                if (n >= totalOrdinals) break;
                int start = n;
                int end = endOrdinal;
                Future<Void> future = this.executor.submit(() -> {
                    ImmutableGraphIndex.View view = this.viewPerThread.get();
                    ByteBuffer buffer = this.bufferPerThread.get();
                    NodeRecordTask task = new NodeRecordTask(start, end, ordinalMapper, this.graph, view, inlineFeatures, featureStateSuppliers, this.recordSize, baseOffset, channel, buffer);
                    return task.call();
                });
                taskFutures.add(future);
            }
            for (Future future : taskFutures) {
                try {
                    future.get();
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new IOException("Interrupted while writing records", e);
                }
                catch (ExecutionException e) {
                    throw this.unwrapExecutionException(e);
                }
            }
            channel.force(true);
        }
    }

    private IOException unwrapExecutionException(ExecutionException e) {
        Throwable cause = e.getCause();
        if (cause instanceof IOException) {
            return (IOException)cause;
        }
        if (cause instanceof RuntimeException) {
            throw (RuntimeException)cause;
        }
        throw new RuntimeException("Error building node record", cause);
    }

    public int getRecordSize() {
        return this.recordSize;
    }

    @Override
    public void close() throws IOException {
        try {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(60L, TimeUnit.SECONDS)) {
                    this.executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                this.executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
            for (ImmutableGraphIndex.View view : this.allViews) {
                view.close();
            }
            this.allViews.clear();
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IOException("Error closing parallel writer", e);
        }
    }

    static class Config {
        final int workerThreads;
        final boolean useDirectBuffers;
        final int taskMultiplier;

        public Config(int workerThreads, boolean useDirectBuffers, int taskMultiplier) {
            this.workerThreads = workerThreads <= 0 ? Runtime.getRuntime().availableProcessors() : workerThreads;
            this.useDirectBuffers = useDirectBuffers;
            this.taskMultiplier = taskMultiplier <= 0 ? 4 : taskMultiplier;
        }

        public static Config defaultConfig() {
            return new Config(0, false, 4);
        }
    }
}

