/*
 * Decompiled with CFR 0.152.
 */
package nl.altindag.ssl.util;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.X509ExtendedTrustManager;
import nl.altindag.ssl.SSLFactory;
import nl.altindag.ssl.exception.GenericCertificateException;
import nl.altindag.ssl.exception.GenericIOException;
import nl.altindag.ssl.util.CertificateUtils;
import nl.altindag.ssl.util.SSLSessionUtils;
import nl.altindag.ssl.util.SSLSocketUtils;
import nl.altindag.ssl.util.TrustManagerUtils;

class CertificateExtractorUtils {
    private static final Pattern CA_ISSUERS_AUTHORITY_INFO_ACCESS = Pattern.compile("(?s)^AuthorityInfoAccess\\h+\\[\\R\\s*\\[\\R.*?accessMethod:\\h+caIssuers\\R\\h*accessLocation: URIName:\\h+(https?://\\S+)", 8);
    private static CertificateExtractorUtils instance;
    private final SSLFactory sslFactory;
    private final SSLSocketFactory unsafeSslSocketFactory;
    private final SSLSocketFactory certificateCapturingSslSocketFactory;
    private final List<X509Certificate> certificatesCollector = new ArrayList<X509Certificate>();

    private CertificateExtractorUtils() {
        X509ExtendedTrustManager certificateCapturingTrustManager = TrustManagerUtils.createCertificateCapturingTrustManager(this.certificatesCollector);
        this.sslFactory = SSLFactory.builder().withTrustMaterial(certificateCapturingTrustManager).build();
        this.certificateCapturingSslSocketFactory = this.sslFactory.getSslSocketFactory();
        this.unsafeSslSocketFactory = SSLSocketUtils.createUnsafeSslSocketFactory();
    }

    static CertificateExtractorUtils getInstance() {
        if (instance == null) {
            instance = new CertificateExtractorUtils();
        } else {
            CertificateExtractorUtils.instance.certificatesCollector.clear();
            SSLSessionUtils.invalidateCaches(CertificateExtractorUtils.instance.sslFactory);
        }
        return instance;
    }

    List<X509Certificate> getCertificateFromExternalSource(String url) {
        try {
            URL parsedUrl = new URL(url);
            if ("https".equalsIgnoreCase(parsedUrl.getProtocol())) {
                HttpsURLConnection connection = (HttpsURLConnection)parsedUrl.openConnection();
                connection.setSSLSocketFactory(this.certificateCapturingSslSocketFactory);
                connection.connect();
                connection.disconnect();
                List<X509Certificate> rootCa = this.getRootCaFromChainIfPossible(this.certificatesCollector);
                return Stream.of(this.certificatesCollector, rootCa).flatMap(Collection::stream).distinct().collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
            }
            return Collections.emptyList();
        }
        catch (IOException e) {
            throw new GenericIOException(String.format("Failed getting certificate from: [%s]", url), e);
        }
    }

    List<X509Certificate> getRootCaFromChainIfPossible(List<X509Certificate> certificates) {
        String subject;
        X509Certificate certificate;
        String issuer;
        boolean isSelfSignedCertificate;
        if (!certificates.isEmpty() && !(isSelfSignedCertificate = (issuer = (certificate = certificates.get(certificates.size() - 1)).getIssuerX500Principal().getName()).equals(subject = certificate.getSubjectX500Principal().getName()))) {
            return this.getRootCaIfPossible(certificate);
        }
        return Collections.emptyList();
    }

    List<X509Certificate> getRootCaIfPossible(X509Certificate x509Certificate) {
        List<X509Certificate> rootCaFromAuthorityInfoAccessExtension = this.getRootCaFromAuthorityInfoAccessExtensionIfPresent(x509Certificate);
        if (!rootCaFromAuthorityInfoAccessExtension.isEmpty()) {
            return rootCaFromAuthorityInfoAccessExtension;
        }
        List<X509Certificate> rootCaFromJdkTrustedCertificates = this.getRootCaFromJdkTrustedCertificates(x509Certificate);
        if (!rootCaFromJdkTrustedCertificates.isEmpty()) {
            return rootCaFromJdkTrustedCertificates;
        }
        return Collections.emptyList();
    }

    List<X509Certificate> getRootCaFromAuthorityInfoAccessExtensionIfPresent(X509Certificate certificate) {
        String certificateContent = certificate.toString();
        Matcher caIssuersMatcher = CA_ISSUERS_AUTHORITY_INFO_ACCESS.matcher(certificateContent);
        if (caIssuersMatcher.find()) {
            String issuerLocation = caIssuersMatcher.group(1);
            return this.getCertificatesFromRemoteFile(URI.create(issuerLocation), certificate);
        }
        return Collections.emptyList();
    }

    List<X509Certificate> getCertificatesFromRemoteFile(URI uri, X509Certificate intermediateCertificate) {
        try {
            URL url = uri.toURL();
            URLConnection connection = url.openConnection();
            if (connection instanceof HttpsURLConnection) {
                ((HttpsURLConnection)connection).setSSLSocketFactory(this.unsafeSslSocketFactory);
            }
            InputStream inputStream = connection.getInputStream();
            List certificates = CertificateUtils.parseDerCertificate(inputStream).stream().filter(X509Certificate.class::isInstance).map(X509Certificate.class::cast).filter(issuer -> this.isIssuerOfIntermediateCertificate(intermediateCertificate, (X509Certificate)issuer)).collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
            inputStream.close();
            return certificates;
        }
        catch (IOException e) {
            throw new GenericCertificateException(e);
        }
    }

    List<X509Certificate> getRootCaFromJdkTrustedCertificates(X509Certificate intermediateCertificate) {
        List<X509Certificate> jdkTrustedCertificates = CertificateUtils.getJdkTrustedCertificates();
        return jdkTrustedCertificates.stream().filter(issuer -> this.isIssuerOfIntermediateCertificate(intermediateCertificate, (X509Certificate)issuer)).collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
    }

    boolean isIssuerOfIntermediateCertificate(X509Certificate intermediateCertificate, X509Certificate issuer) {
        try {
            intermediateCertificate.verify(issuer.getPublicKey());
            return true;
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException | CertificateException e) {
            return false;
        }
    }
}

