/*
 * Decompiled with CFR 0.152.
 */
package nl.altindag.ssl.util;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.X509TrustManager;
import javax.security.auth.x500.X500Principal;
import nl.altindag.ssl.SSLFactory;
import nl.altindag.ssl.exception.GenericCertificateException;
import nl.altindag.ssl.exception.GenericIOException;
import nl.altindag.ssl.util.IOUtils;
import nl.altindag.ssl.util.TrustManagerUtils;

public final class CertificateUtils {
    private static final String CERTIFICATE_TYPE = "X.509";
    private static final String CERTIFICATE_HEADER = "-----BEGIN CERTIFICATE-----";
    private static final String CERTIFICATE_FOOTER = "-----END CERTIFICATE-----";
    private static final Pattern CERTIFICATE_PATTERN = Pattern.compile("-----BEGIN CERTIFICATE-----(.*?)-----END CERTIFICATE-----", 32);
    private static final String EMPTY = "";
    private static final Pattern CA_ISSUERS_AUTHORITY_INFO_ACCESS = Pattern.compile("(?s)^AuthorityInfoAccess\\h+\\[\\R\\s*\\[\\R.*?accessMethod:\\h+caIssuers\\R\\h*accessLocation: URIName:\\h+(https?://\\S+)", 8);
    private static SSLSocketFactory unsafeSslSocketFactory = null;

    private CertificateUtils() {
    }

    public static String generateAlias(Certificate certificate) {
        if (certificate instanceof X509Certificate) {
            return ((X509Certificate)certificate).getSubjectX500Principal().getName();
        }
        return UUID.randomUUID().toString();
    }

    public static List<Certificate> loadCertificate(String ... certificatePaths) {
        return CertificateUtils.loadCertificate((T certificatePath) -> CertificateUtils.class.getClassLoader().getResourceAsStream((String)certificatePath), certificatePaths);
    }

    public static List<Certificate> loadCertificate(Path ... certificatePaths) {
        return CertificateUtils.loadCertificate((T certificatePath) -> {
            try {
                return Files.newInputStream(certificatePath, StandardOpenOption.READ);
            }
            catch (IOException exception) {
                throw new UncheckedIOException(exception);
            }
        }, certificatePaths);
    }

    public static List<Certificate> loadCertificate(InputStream ... certificateStreams) {
        return CertificateUtils.loadCertificate(Function.identity(), certificateStreams);
    }

    private static <T> List<Certificate> loadCertificate(Function<T, InputStream> resourceMapper, T[] resources) {
        ArrayList<Certificate> certificates = new ArrayList<Certificate>();
        for (T resource : resources) {
            try (InputStream certificateStream = resourceMapper.apply(resource);){
                certificates.addAll(CertificateUtils.parseCertificate(certificateStream));
            }
            catch (Exception e) {
                throw new GenericIOException(e);
            }
        }
        return Collections.unmodifiableList(certificates);
    }

    private static List<Certificate> parseCertificate(InputStream certificateStream) {
        String content = IOUtils.getContent(certificateStream);
        return CertificateUtils.parseCertificate(content);
    }

    public static List<Certificate> parseCertificate(String certificateContent) {
        ArrayList<Certificate> certificates = new ArrayList<Certificate>();
        Matcher certificateMatcher = CERTIFICATE_PATTERN.matcher(certificateContent);
        while (certificateMatcher.find()) {
            String sanitizedCertificate = certificateMatcher.group(1).replace(System.lineSeparator(), EMPTY).trim();
            byte[] decodedCertificate = Base64.getDecoder().decode(sanitizedCertificate);
            try {
                ByteArrayInputStream certificateAsInputStream = new ByteArrayInputStream(decodedCertificate);
                Throwable throwable = null;
                try {
                    CertificateFactory certificateFactory = CertificateFactory.getInstance(CERTIFICATE_TYPE);
                    Certificate certificate = certificateFactory.generateCertificate(certificateAsInputStream);
                    certificates.add(certificate);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (certificateAsInputStream == null) continue;
                    if (throwable != null) {
                        try {
                            certificateAsInputStream.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    certificateAsInputStream.close();
                }
            }
            catch (IOException | CertificateException e) {
                throw new GenericCertificateException(e);
            }
        }
        if (certificates.isEmpty()) {
            throw new GenericCertificateException(String.format("There are no valid certificates present to parse. Please make sure to supply at least one valid pem formatted certificate containing the header %s and the footer %s", CERTIFICATE_HEADER, CERTIFICATE_FOOTER));
        }
        return Collections.unmodifiableList(certificates);
    }

    public static List<X509Certificate> getJdkTrustedCertificates() {
        return Collections.unmodifiableList(Arrays.asList(TrustManagerUtils.createTrustManagerWithJdkTrustedCertificates().getAcceptedIssuers()));
    }

    public static List<X509Certificate> getSystemTrustedCertificates() {
        return TrustManagerUtils.createTrustManagerWithSystemTrustedCertificates().map(X509TrustManager::getAcceptedIssuers).map(Arrays::asList).map(Collections::unmodifiableList).orElse(Collections.emptyList());
    }

    public static Map<String, List<String>> getCertificateAsPem(String ... urls) {
        return CertificateUtils.getCertificateAsPem(Arrays.asList(urls));
    }

    public static Map<String, List<String>> getCertificateAsPem(List<String> urls) {
        Map<String, List> certificates = CertificateUtils.getCertificate(urls).entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, entry -> CertificateUtils.convertToPem((List)entry.getValue())));
        return Collections.unmodifiableMap(certificates);
    }

    public static Map<String, List<Certificate>> getCertificate(String ... urls) {
        return CertificateUtils.getCertificate(Arrays.asList(urls));
    }

    public static Map<String, List<Certificate>> getCertificate(List<String> urls) {
        HashMap<String, List<Certificate>> certificates = new HashMap<String, List<Certificate>>();
        for (String url : urls) {
            List<Certificate> serverCertificates = CertificateUtils.getCertificateFromExternalSource(url);
            certificates.put(url, serverCertificates);
        }
        return Collections.unmodifiableMap(certificates);
    }

    private static List<Certificate> getCertificateFromExternalSource(String url) {
        try {
            URL parsedUrl = new URL(url);
            if ("https".equalsIgnoreCase(parsedUrl.getProtocol())) {
                HttpsURLConnection connection = (HttpsURLConnection)parsedUrl.openConnection();
                SSLSocketFactory unsafeSslSocketFactory = CertificateUtils.getUnsafeSslSocketFactory();
                connection.setSSLSocketFactory(unsafeSslSocketFactory);
                connection.connect();
                Certificate[] serverCertificates = connection.getServerCertificates();
                List<X509Certificate> rootCa = CertificateUtils.getRootCaFromChainIfPossible(serverCertificates);
                ArrayList<Certificate> certificates = new ArrayList<Certificate>();
                certificates.addAll(Arrays.asList(serverCertificates));
                certificates.addAll(rootCa);
                connection.disconnect();
                return certificates.stream().distinct().collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
            }
            return Collections.emptyList();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static SSLSocketFactory getUnsafeSslSocketFactory() {
        if (unsafeSslSocketFactory == null) {
            unsafeSslSocketFactory = SSLFactory.builder().withTrustingAllCertificatesWithoutValidation().build().getSslSocketFactory();
        }
        return unsafeSslSocketFactory;
    }

    static List<X509Certificate> getRootCaFromChainIfPossible(Certificate[] certificates) {
        String subject;
        X509Certificate certificate;
        String issuer;
        boolean isSelfSignedCertificate;
        if (certificates.length > 0 && certificates[certificates.length - 1] instanceof X509Certificate && !(isSelfSignedCertificate = (issuer = (certificate = (X509Certificate)certificates[certificates.length - 1]).getIssuerX500Principal().getName()).equals(subject = certificate.getSubjectX500Principal().getName()))) {
            return CertificateUtils.getRootCaIfPossible(certificate);
        }
        return Collections.emptyList();
    }

    static List<X509Certificate> getRootCaIfPossible(X509Certificate x509Certificate) {
        List<X509Certificate> rootCaFromAuthorityInfoAccessExtension = CertificateUtils.getRootCaFromAuthorityInfoAccessExtensionIfPresent(x509Certificate);
        if (!rootCaFromAuthorityInfoAccessExtension.isEmpty()) {
            return rootCaFromAuthorityInfoAccessExtension;
        }
        List<X509Certificate> rootCaFromJdkTrustedCertificates = CertificateUtils.getRootCaFromJdkTrustedCertificates(x509Certificate);
        if (!rootCaFromJdkTrustedCertificates.isEmpty()) {
            return rootCaFromJdkTrustedCertificates;
        }
        return Collections.emptyList();
    }

    static List<X509Certificate> getRootCaFromAuthorityInfoAccessExtensionIfPresent(X509Certificate certificate) {
        String certificateContent = certificate.toString();
        Matcher caIssuersMatcher = CA_ISSUERS_AUTHORITY_INFO_ACCESS.matcher(certificateContent);
        if (caIssuersMatcher.find()) {
            String issuerLocation = caIssuersMatcher.group(1);
            return CertificateUtils.getCertificatesFromRemoteFile(URI.create(issuerLocation), certificate);
        }
        return Collections.emptyList();
    }

    static List<X509Certificate> getCertificatesFromRemoteFile(URI uri, X509Certificate intermediateCertificate) {
        try {
            URL url = uri.toURL();
            URLConnection connection = url.openConnection();
            if (connection instanceof HttpsURLConnection) {
                ((HttpsURLConnection)connection).setSSLSocketFactory(CertificateUtils.getUnsafeSslSocketFactory());
            }
            InputStream inputStream = connection.getInputStream();
            CertificateFactory certificateFactory = CertificateFactory.getInstance(CERTIFICATE_TYPE);
            List certificates = certificateFactory.generateCertificates(inputStream).stream().filter(X509Certificate.class::isInstance).map(X509Certificate.class::cast).filter(issuer -> CertificateUtils.isIssuerOfIntermediateCertificate(intermediateCertificate, issuer)).collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
            inputStream.close();
            return certificates;
        }
        catch (IOException | CertificateException e) {
            throw new GenericCertificateException(e);
        }
    }

    static List<X509Certificate> getRootCaFromJdkTrustedCertificates(X509Certificate intermediateCertificate) {
        List<X509Certificate> jdkTrustedCertificates = CertificateUtils.getJdkTrustedCertificates();
        return jdkTrustedCertificates.stream().filter(issuer -> CertificateUtils.isIssuerOfIntermediateCertificate(intermediateCertificate, issuer)).collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
    }

    static boolean isIssuerOfIntermediateCertificate(X509Certificate intermediateCertificate, X509Certificate issuer) {
        try {
            intermediateCertificate.verify(issuer.getPublicKey());
            return true;
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException | CertificateException e) {
            return false;
        }
    }

    public static List<String> convertToPem(List<Certificate> certificates) {
        return certificates.stream().map(CertificateUtils::convertToPem).collect(Collectors.collectingAndThen(Collectors.toList(), Collections::unmodifiableList));
    }

    public static String convertToPem(Certificate certificate) {
        try {
            byte[] encodedCertificate = certificate.getEncoded();
            byte[] base64EncodedCertificate = Base64.getEncoder().encode(encodedCertificate);
            String parsedCertificate = new String(base64EncodedCertificate);
            List certificateContainer = Stream.of(parsedCertificate.split("(?<=\\G.{64})")).collect(Collectors.toCollection(ArrayList::new));
            certificateContainer.add(0, CERTIFICATE_HEADER);
            certificateContainer.add(CERTIFICATE_FOOTER);
            if (certificate instanceof X509Certificate) {
                X509Certificate x509Certificate = (X509Certificate)certificate;
                X500Principal issuer = x509Certificate.getIssuerX500Principal();
                certificateContainer.add(0, String.format("issuer=%s", issuer.getName()));
                X500Principal subject = x509Certificate.getSubjectX500Principal();
                certificateContainer.add(0, String.format("subject=%s", subject.getName()));
            }
            return String.join((CharSequence)System.lineSeparator(), certificateContainer);
        }
        catch (CertificateEncodingException e) {
            throw new GenericCertificateException(e);
        }
    }
}

