package com.iflytek.aiui.player.core

import android.content.Context
import android.media.AudioAttributes
import android.media.AudioFocusRequest
import android.media.AudioManager
import android.os.Build

/**
 * 音频焦点管理
 *
 * 封装不同版本上对音频焦点处理的处理方式
 */

/** @suppress */
class AudioFocus(context: Context, private val focusListener: AudioManager.OnAudioFocusChangeListener) {
    private var mAudioFocus = false
    private val mAudioManager = context.getSystemService(Context.AUDIO_SERVICE) as AudioManager
    private val mAudioFocusListener = AudioManager.OnAudioFocusChangeListener { focusChange ->
        when(focusChange) {
            AudioManager.AUDIOFOCUS_LOSS, AudioManager.AUDIOFOCUS_LOSS_TRANSIENT -> {
                mAudioFocus = false
            }
        }

        focusListener.onAudioFocusChange(focusChange)
    }

    private lateinit var mAudioFocusRequest: AudioFocusRequest

    init {
        if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            mAudioFocusRequest = AudioFocusRequest.Builder(AudioManager.AUDIOFOCUS_GAIN).run {
                setAudioAttributes(AudioAttributes.Builder().run {
                    setUsage(AudioAttributes.USAGE_MEDIA)
                    setContentType(AudioAttributes.CONTENT_TYPE_MUSIC)
                    build()
                })

                setOnAudioFocusChangeListener(mAudioFocusListener)
                build()
            }
        }
    }

    /**
     * 获取音频焦点
     *
     * @return Int 音频焦点获取请求结果
     *
     * 返回值见AudioManager中说明
     */
    fun requestAudioFocusIfNeed(): Int{
        if(mAudioFocus) return AudioManager.AUDIOFOCUS_REQUEST_GRANTED

        val result = if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            mAudioManager.requestAudioFocus(mAudioFocusRequest)
        } else {
            mAudioManager.requestAudioFocus(mAudioFocusListener, AudioManager.STREAM_MUSIC, AudioManager.AUDIOFOCUS_GAIN)
        }

        mAudioFocus = result == AudioManager.AUDIOFOCUS_REQUEST_GRANTED

        return result
    }

    /**
     * 放弃音频焦点
     */
    fun abandonAudioFocus() {
        mAudioFocus = false

        if(Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            mAudioManager.abandonAudioFocusRequest(mAudioFocusRequest)
        } else {
            mAudioManager.abandonAudioFocus(mAudioFocusListener)
        }
    }



}