/*
 * Copyright (c) 2008, 2009, 2011 Oracle, Inc. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 and Eclipse Distribution License v. 1.0
 * which accompanies this distribution.  The Eclipse Public License is available
 * at http://www.eclipse.org/legal/epl-v10.html and the Eclipse Distribution License
 * is available at http://www.eclipse.org/org/documents/edl-v10.php.
 */
package javax.persistence;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Specifies a column that is used to maintain the persistent order of a list. The persistence provider is
 * responsible for maintaining the order upon retrieval and in the database. The persistence provider is
 * responsible for updating the ordering upon flushing to the database to reflect any insertion, deletion, or
 * reordering affecting the list.
 * <p>
 * The <code>OrderColumn</code> annotation is specified on a OneToMany or ManyToMany relationship or on an
 * element collection. The <code>OrderColumn</code> annotation is specified on the side of the relationship
 * that references the collection that is to be ordered. The order column is not visible as part of the state
 * of the entity or embeddable class.
 * <p>
 * The {@link OrderBy} annotation should be used for ordering that is visible as persistent state and
 * maintained by the application. The <code>OrderBy</code> annotation is not used when
 * <code>OrderColumn</code> is specified.
 * <p>
 * The order column must be of integral type. The persistence provider maintains a contiguous (non-sparse)
 * ordering of the values of the order column when updating the association or element collection. The order
 * column value for the first element is 0.
 * <p>
 * <pre>
 *
 *    Example:
 *
 *    &#064;Entity
 *    public class CreditCard {
 *
 *       &#064;Id long ccNumber;
 *
 *       &#064;OneToMany  // unidirectional
 *       &#064;OrderColumn
 *       List&#060;CardTransaction&#062; transactionHistory;
 *       ...
 *    }
 *
 * </pre>
 *
 * @see OrderBy
 * @since Java Persistence 2.0
 */
@Target({FIELD})
@Retention(RUNTIME)
public @interface OrderColumn {

  /**
   * (Optional) The name of the ordering column. Defaults to the concatenation of the name of the
   * referencing property or field; "_"; "ORDER".
   *
   * @return name
   */
  String name() default "";

  /**
   * (Optional) Whether the database column is nullable.
   *
   * @return whether nullable
   */
  boolean nullable() default true;

  /**
   * (Optional) Whether the column is included in SQL INSERT statements generated by the persistence
   * provider.
   *
   * @return whether insertable
   */
  boolean insertable() default true;

  /**
   * (Optional) Whether the column is included in SQL UPDATE statements generated by the persistence
   * provider.
   *
   * @return whether updateable
   */
  boolean updatable() default true;

  /**
   * (Optional) The SQL fragment that is used when generating the DDL for the column. Defaults to generated
   * SQL to create a column of the inferred type.
   *
   * @return column definition
   */
  String columnDefinition() default "";
}
