/*
 * Decompiled with CFR 0.152.
 */
package io.ebean.datasource.pool;

import io.ebean.datasource.DataSourceAlert;
import io.ebean.datasource.DataSourceBuilder;
import io.ebean.datasource.DataSourceConfig;
import io.ebean.datasource.DataSourceInitialiseException;
import io.ebean.datasource.DataSourcePool;
import io.ebean.datasource.DataSourcePoolListener;
import io.ebean.datasource.InitDatabase;
import io.ebean.datasource.PoolStatus;
import io.ebean.datasource.pool.DriverDataSource;
import io.ebean.datasource.pool.Log;
import io.ebean.datasource.pool.PooledConnection;
import io.ebean.datasource.pool.PooledConnectionQueue;
import io.ebean.datasource.pool.PstmtCache;
import io.ebean.datasource.pool.TransactionIsolation;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLClientInfoException;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;
import java.util.List;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Logger;
import javax.sql.DataSource;

final class ConnectionPool
implements DataSourcePool {
    private static final String APPLICATION_NAME = "ApplicationName";
    private static final long LAMBDA_MILLIS = 60000L;
    private final ReentrantLock heartbeatLock = new ReentrantLock(false);
    private final ReentrantLock notifyLock = new ReentrantLock(false);
    private final String name;
    private final AtomicInteger size = new AtomicInteger(0);
    private final DataSourceConfig config;
    private final DataSourceAlert notify;
    private final DataSourcePoolListener poolListener;
    private final List<String> initSql;
    private final String user;
    private final String schema;
    private final String heartbeatsql;
    private final int heartbeatFreqSecs;
    private final int heartbeatTimeoutSeconds;
    private final long trimPoolFreqMillis;
    private final int transactionIsolation;
    private final boolean autoCommit;
    private final boolean readOnly;
    private final boolean failOnStart;
    private final int maxInactiveMillis;
    private final long maxAgeMillis;
    private final boolean captureStackTrace;
    private final int maxStackTraceSize;
    private final Properties clientInfo;
    private final String applicationName;
    private final DataSource source;
    private long nextTrimTime;
    private long nextLambdaTrimTime;
    private final AtomicBoolean dataSourceUp = new AtomicBoolean(false);
    private SQLException dataSourceDownReason;
    private final AtomicBoolean inWarningMode = new AtomicBoolean();
    private int minConnections;
    private int maxConnections;
    private int warningSize;
    private final int waitTimeoutMillis;
    private final int pstmtCacheSize;
    private final PooledConnectionQueue queue;
    private Timer heartBeatTimer;
    private final long leakTimeMinutes;
    private final LongAdder pscHit = new LongAdder();
    private final LongAdder pscMiss = new LongAdder();
    private final LongAdder pscPut = new LongAdder();
    private final LongAdder pscRem = new LongAdder();
    private final boolean shutdownOnJvmExit;
    private Thread shutdownHook;

    ConnectionPool(String name, DataSourceConfig params) {
        this.config = params;
        this.name = name;
        this.notify = params.getAlert();
        this.poolListener = params.getListener();
        this.autoCommit = params.isAutoCommit();
        this.readOnly = params.isReadOnly();
        this.failOnStart = params.isFailOnStart();
        this.initSql = params.getInitSql();
        this.transactionIsolation = params.getIsolationLevel();
        this.maxInactiveMillis = 1000 * params.getMaxInactiveTimeSecs();
        this.maxAgeMillis = 60000L * (long)params.getMaxAgeMinutes();
        this.leakTimeMinutes = params.getLeakTimeMinutes();
        this.captureStackTrace = params.isCaptureStackTrace();
        this.maxStackTraceSize = params.getMaxStackTraceSize();
        this.pstmtCacheSize = params.getPstmtCacheSize();
        this.minConnections = params.getMinConnections();
        this.maxConnections = params.getMaxConnections();
        this.waitTimeoutMillis = params.getWaitTimeoutMillis();
        this.heartbeatsql = params.getHeartbeatSql();
        this.heartbeatFreqSecs = params.getHeartbeatFreqSecs();
        this.heartbeatTimeoutSeconds = params.getHeartbeatTimeoutSeconds();
        this.trimPoolFreqMillis = 1000L * (long)params.getTrimPoolFreqSecs();
        this.applicationName = params.getApplicationName();
        this.clientInfo = params.getClientInfo();
        this.queue = new PooledConnectionQueue(this);
        this.schema = params.getSchema();
        this.user = params.getUsername();
        this.shutdownOnJvmExit = params.isShutdownOnJvmExit();
        this.source = DriverDataSource.of(name, (DataSourceBuilder.Settings)params);
        if (!params.isOffline()) {
            this.init();
        }
        this.nextLambdaTrimTime = System.currentTimeMillis() + this.trimPoolFreqMillis + 60000L;
    }

    private void init() {
        try {
            if (this.config.useInitDatabase()) {
                this.initialiseDatabase();
            }
            this.initialiseConnections();
        }
        catch (SQLException e) {
            throw new DataSourceInitialiseException("Error initialising DataSource with user: " + this.user + " error:" + e.getMessage(), (Throwable)e);
        }
    }

    void pstmtCacheMetrics(PstmtCache pstmtCache) {
        this.pscHit.add(pstmtCache.hitCount());
        this.pscMiss.add(pstmtCache.missCount());
        this.pscPut.add(pstmtCache.putCount());
        this.pscRem.add(pstmtCache.removeCount());
    }

    public Logger getParentLogger() throws SQLFeatureNotSupportedException {
        throw new SQLFeatureNotSupportedException("We do not support java.util.logging");
    }

    private void tryEnsureMinimumConnections() {
        this.notifyLock.lock();
        try {
            this.queue.ensureMinimumConnections();
            if (this.notify != null) {
                this.notify.dataSourceUp((DataSource)((Object)this));
            }
        }
        catch (SQLException e) {
            Log.error("Error trying to ensure minimum connections, maybe db server is down - message:" + e.getMessage(), e);
        }
        finally {
            this.notifyLock.unlock();
        }
    }

    private void initialiseConnections() throws SQLException {
        long start = System.currentTimeMillis();
        this.dataSourceUp.set(true);
        if (this.failOnStart) {
            this.queue.ensureMinimumConnections();
        } else {
            this.tryEnsureMinimumConnections();
        }
        this.startHeartBeatIfStopped();
        if (this.shutdownOnJvmExit && this.shutdownHook == null) {
            this.shutdownHook = new Thread(() -> this.shutdownPool(true, true));
            this.shutdownHook.setName("DataSourcePool-ShutdownHook");
            Runtime.getRuntime().addShutdownHook(this.shutdownHook);
        }
        String ro = this.readOnly ? "readOnly[true] " : "";
        Log.info("DataSource [{0}] {1}autoCommit[{2}] transIsolation[{3}] min[{4}] max[{5}] in[{6}ms]", this.name, ro, this.autoCommit, TransactionIsolation.description(this.transactionIsolation), this.minConnections, this.maxConnections, System.currentTimeMillis() - start);
    }

    private void initialiseDatabase() throws SQLException {
        try (Connection connection = this.createConnection();){
            connection.clearWarnings();
        }
        catch (SQLException e) {
            Log.info("Obtaining connection using ownerUsername:{0} to initialise database", this.config.getOwnerUsername());
            try (Connection ownerConnection = this.getConnection(this.config.getOwnerUsername(), this.config.getOwnerPassword());){
                InitDatabase initDatabase = this.config.getInitDatabase();
                initDatabase.run(ownerConnection, this.config);
                ownerConnection.commit();
            }
            catch (SQLException e2) {
                throw new SQLException("Failed to run InitDatabase with ownerUsername:" + this.config.getOwnerUsername() + " message:" + e2.getMessage(), e2);
            }
        }
    }

    public boolean isWrapperFor(Class<?> arg0) {
        return false;
    }

    public <T> T unwrap(Class<T> arg0) throws SQLException {
        throw new SQLException("Not Implemented");
    }

    public String name() {
        return this.name;
    }

    public int size() {
        return this.size.get();
    }

    void inc() {
        this.size.incrementAndGet();
    }

    void dec() {
        this.size.decrementAndGet();
    }

    int maxStackTraceSize() {
        return this.maxStackTraceSize;
    }

    public SQLException dataSourceDownReason() {
        return this.dataSourceDownReason;
    }

    void notifyWarning(String msg) {
        if (this.inWarningMode.compareAndSet(false, true)) {
            Log.warn(msg, new Object[0]);
            if (this.notify != null) {
                this.notify.dataSourceWarning((DataSource)((Object)this), msg);
            }
        }
    }

    private void notifyDataSourceIsDown(SQLException reason) {
        if (this.dataSourceUp.get()) {
            this.reset();
            this.notifyDown(reason);
        }
    }

    private void notifyDown(SQLException reason) {
        this.notifyLock.lock();
        try {
            if (this.dataSourceUp.get()) {
                this.dataSourceUp.set(false);
                this.dataSourceDownReason = reason;
                Log.error("FATAL: DataSource [" + this.name + "] is down or has network error!!!", reason);
                if (this.notify != null) {
                    this.notify.dataSourceDown((DataSource)((Object)this), reason);
                }
            }
        }
        finally {
            this.notifyLock.unlock();
        }
    }

    private void notifyDataSourceIsUp() {
        if (!this.dataSourceUp.get()) {
            this.reset();
            this.notifyUp();
        }
    }

    private void notifyUp() {
        this.notifyLock.lock();
        try {
            if (!this.dataSourceUp.get()) {
                this.dataSourceUp.set(true);
                this.startHeartBeatIfStopped();
                this.dataSourceDownReason = null;
                Log.error("RESOLVED FATAL: DataSource [" + this.name + "] is back up!", new Object[0]);
                if (this.notify != null) {
                    this.notify.dataSourceUp((DataSource)((Object)this));
                }
            } else {
                Log.info("DataSource [{0}] is back up!", this.name);
            }
        }
        finally {
            this.notifyLock.unlock();
        }
    }

    void checkLambdaIdle() {
        if (System.currentTimeMillis() > this.nextLambdaTrimTime) {
            long timeGapSeconds = (System.currentTimeMillis() - this.nextTrimTime) / 1000L;
            PoolStatus status = this.status(false);
            Log.info("DataSource [{0}] detected lambda restore, trimming idle connections - timeGap {1}s {2}", this.name, timeGapSeconds, status);
            this.trimIdleConnections();
        }
    }

    private void trimIdleConnections() {
        if (System.currentTimeMillis() > this.nextTrimTime) {
            try {
                this.queue.trim(this.maxInactiveMillis, this.maxAgeMillis);
                this.nextTrimTime = System.currentTimeMillis() + this.trimPoolFreqMillis;
                this.nextLambdaTrimTime = this.nextTrimTime + 60000L;
            }
            catch (Exception e) {
                Log.error("Error trying to trim idle connections - message:" + e.getMessage(), e);
            }
        }
    }

    private void checkDataSource() {
        this.trimIdleConnections();
        Connection conn = null;
        try {
            conn = this.getConnection();
            if (this.testConnection(conn)) {
                this.notifyDataSourceIsUp();
            } else {
                this.notifyDataSourceIsDown(null);
            }
        }
        catch (SQLException ex) {
            this.notifyDataSourceIsDown(ex);
        }
        finally {
            try {
                if (conn != null) {
                    conn.close();
                }
            }
            catch (SQLException ex) {
                Log.warn("Can't close connection in checkDataSource!", new Object[0]);
            }
        }
    }

    private Connection initConnection(Connection conn) throws SQLException {
        conn.setAutoCommit(this.autoCommit);
        if (conn.getTransactionIsolation() != this.transactionIsolation) {
            conn.setTransactionIsolation(this.transactionIsolation);
        }
        if (this.readOnly) {
            conn.setReadOnly(true);
        }
        if (this.schema != null) {
            conn.setSchema(this.schema);
        }
        if (this.applicationName != null) {
            try {
                conn.setClientInfo(APPLICATION_NAME, this.applicationName);
            }
            catch (SQLClientInfoException e) {
                Log.error("Error setting clientInfo ApplicationName", e);
            }
        }
        if (this.clientInfo != null) {
            try {
                conn.setClientInfo(this.clientInfo);
            }
            catch (SQLClientInfoException e) {
                Log.error("Error setting clientInfo", e);
            }
        }
        if (this.initSql != null) {
            for (String query : this.initSql) {
                try (Statement stmt = conn.createStatement();){
                    stmt.execute(query);
                }
            }
        }
        return conn;
    }

    private Connection createConnection() throws SQLException {
        return this.initConnection(this.source.getConnection());
    }

    public void setMaxSize(int max) {
        this.queue.setMaxSize(max);
        this.maxConnections = max;
    }

    public int getMaxSize() {
        return this.maxConnections;
    }

    public void setMinSize(int min) {
        this.queue.setMinSize(min);
        this.minConnections = min;
    }

    public int getMinSize() {
        return this.minConnections;
    }

    public void setWarningSize(int warningSize) {
        this.queue.setWarningSize(warningSize);
        this.warningSize = warningSize;
    }

    public int getWarningSize() {
        return this.warningSize;
    }

    int waitTimeoutMillis() {
        return this.waitTimeoutMillis;
    }

    long maxAgeMillis() {
        return this.maxAgeMillis;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean testConnection(Connection conn) throws SQLException {
        if (this.heartbeatsql == null) {
            return conn.isValid(this.heartbeatTimeoutSeconds);
        }
        Statement stmt = null;
        ResultSet rset = null;
        try {
            stmt = conn.createStatement();
            if (this.heartbeatTimeoutSeconds > 0) {
                stmt.setQueryTimeout(this.heartbeatTimeoutSeconds);
            }
            rset = stmt.executeQuery(this.heartbeatsql);
            conn.commit();
            boolean bl = true;
            return bl;
        }
        finally {
            try {
                if (rset != null) {
                    rset.close();
                }
            }
            catch (SQLException e) {
                Log.error("Error closing resultSet", e);
            }
            try {
                if (stmt != null) {
                    stmt.close();
                }
            }
            catch (SQLException e) {
                Log.error("Error closing statement", e);
            }
        }
    }

    boolean validateConnection(PooledConnection conn) {
        try {
            return this.testConnection(conn);
        }
        catch (Exception e) {
            Log.warn("Heartbeat test failed on connection:{0} message: {1}", conn.name(), e.getMessage());
            return false;
        }
    }

    void returnConnection(PooledConnection pooledConnection) {
        this.returnTheConnection(pooledConnection, false);
    }

    void returnConnectionForceClose(PooledConnection pooledConnection) {
        this.returnTheConnection(pooledConnection, true);
    }

    void removeClosedConnection(PooledConnection pooledConnection) {
        this.queue.returnPooledConnection(pooledConnection, true);
    }

    private void returnTheConnection(PooledConnection pooledConnection, boolean forceClose) {
        if (this.poolListener != null && !forceClose) {
            this.poolListener.onBeforeReturnConnection((Connection)pooledConnection);
        }
        this.queue.returnPooledConnection(pooledConnection, forceClose);
        if (forceClose) {
            this.checkDataSource();
        }
    }

    void returnConnectionReset(PooledConnection pooledConnection) {
        this.queue.returnPooledConnection(pooledConnection, true);
        Log.warn("Resetting DataSource on read-only failure [{0}]", this.name);
        this.reset();
    }

    PooledConnection createConnectionForQueue(int connId) throws SQLException {
        try {
            PooledConnection pooledConnection = new PooledConnection(this, connId, this.createConnection());
            pooledConnection.resetForUse();
            this.notifyDataSourceIsUp();
            return pooledConnection;
        }
        catch (SQLException ex) {
            this.notifyDataSourceIsDown(ex);
            throw ex;
        }
    }

    private void reset() {
        this.queue.reset(this.leakTimeMinutes);
        this.inWarningMode.set(false);
    }

    public Connection getConnection(String username, String password) throws SQLException {
        return this.initConnection(this.source.getConnection(username, password));
    }

    public Connection getConnection() throws SQLException {
        return this.getPooledConnection();
    }

    private PooledConnection getPooledConnection() throws SQLException {
        PooledConnection c = this.queue.obtainConnection();
        if (this.captureStackTrace) {
            c.setStackTrace(Thread.currentThread().getStackTrace());
        }
        if (this.poolListener != null) {
            this.poolListener.onAfterBorrowConnection((Connection)c);
        }
        return c;
    }

    public void shutdown() {
        this.shutdownPool(true, false);
    }

    public void offline() {
        this.shutdownPool(false, false);
    }

    private void shutdownPool(boolean closeBusyConnections, boolean fromHook) {
        this.stopHeartBeatIfRunning();
        PoolStatus status = this.queue.shutdown(closeBusyConnections);
        this.dataSourceUp.set(false);
        if (fromHook) {
            Log.info("DataSource [{0}] shutdown on JVM exit {1}  psc[hit:{2} miss:{3} put:{4} rem:{5}]", this.name, status, this.pscHit, this.pscMiss, this.pscPut, this.pscRem);
        } else {
            Log.info("DataSource [{0}] shutdown {1}  psc[hit:{2} miss:{3} put:{4} rem:{5}]", this.name, status, this.pscHit, this.pscMiss, this.pscPut, this.pscRem);
            this.removeShutdownHook();
        }
    }

    private void removeShutdownHook() {
        if (this.shutdownHook != null) {
            try {
                Runtime.getRuntime().removeShutdownHook(this.shutdownHook);
            }
            catch (IllegalStateException illegalStateException) {
                // empty catch block
            }
            this.shutdownHook = null;
        }
    }

    public void online() throws SQLException {
        if (!this.dataSourceUp.get()) {
            this.initialiseConnections();
        }
    }

    public boolean isOnline() {
        return this.dataSourceUp.get();
    }

    public boolean isDataSourceUp() {
        return this.dataSourceUp.get();
    }

    private void startHeartBeatIfStopped() {
        this.heartbeatLock.lock();
        try {
            int freqMillis;
            if (this.heartBeatTimer == null && (freqMillis = this.heartbeatFreqSecs * 1000) > 0) {
                this.heartBeatTimer = new Timer(this.name + ".heartBeat", true);
                this.heartBeatTimer.scheduleAtFixedRate((TimerTask)new HeartBeatRunnable(), freqMillis, (long)freqMillis);
            }
        }
        finally {
            this.heartbeatLock.unlock();
        }
    }

    private void stopHeartBeatIfRunning() {
        this.heartbeatLock.lock();
        try {
            if (this.heartBeatTimer != null) {
                this.heartBeatTimer.cancel();
                this.heartBeatTimer = null;
            }
        }
        finally {
            this.heartbeatLock.unlock();
        }
    }

    public boolean isAutoCommit() {
        return this.autoCommit;
    }

    int transactionIsolation() {
        return this.transactionIsolation;
    }

    boolean captureStackTrace() {
        return this.captureStackTrace;
    }

    long leakTimeMinutes() {
        return this.leakTimeMinutes;
    }

    int pstmtCacheSize() {
        return this.pstmtCacheSize;
    }

    public int getLoginTimeout() throws SQLException {
        throw new SQLException("Method not supported");
    }

    public void setLoginTimeout(int seconds) throws SQLException {
        throw new SQLException("Method not supported");
    }

    public PrintWriter getLogWriter() {
        return null;
    }

    public void setLogWriter(PrintWriter writer) throws SQLException {
        throw new SQLException("Method not supported");
    }

    public PoolStatus status(boolean reset) {
        return this.queue.status(reset);
    }

    class HeartBeatRunnable
    extends TimerTask {
        HeartBeatRunnable() {
        }

        @Override
        public void run() {
            ConnectionPool.this.checkDataSource();
        }
    }

    static final class Status
    implements PoolStatus {
        private final int minSize;
        private final int maxSize;
        private final int free;
        private final int busy;
        private final int waiting;
        private final int highWaterMark;
        private final int waitCount;
        private final int hitCount;

        Status(int minSize, int maxSize, int free, int busy, int waiting, int highWaterMark, int waitCount, int hitCount) {
            this.minSize = minSize;
            this.maxSize = maxSize;
            this.free = free;
            this.busy = busy;
            this.waiting = waiting;
            this.highWaterMark = highWaterMark;
            this.waitCount = waitCount;
            this.hitCount = hitCount;
        }

        public String toString() {
            return "min[" + this.minSize + "] max[" + this.maxSize + "] free[" + this.free + "] busy[" + this.busy + "] waiting[" + this.waiting + "] highWaterMark[" + this.highWaterMark + "] waitCount[" + this.waitCount + "] hitCount[" + this.hitCount + "]";
        }

        public int minSize() {
            return this.minSize;
        }

        public int maxSize() {
            return this.maxSize;
        }

        public int free() {
            return this.free;
        }

        public int busy() {
            return this.busy;
        }

        public int waiting() {
            return this.waiting;
        }

        public int highWaterMark() {
            return this.highWaterMark;
        }

        public int waitCount() {
            return this.waitCount;
        }

        public int hitCount() {
            return this.hitCount;
        }
    }
}

