/*
 * Decompiled with CFR 0.152.
 */
package io.ebean.datasource.pool;

import io.ebean.datasource.DataSourceAlert;
import io.ebean.datasource.DataSourceConfig;
import io.ebean.datasource.DataSourceConfigurationException;
import io.ebean.datasource.DataSourceInitialiseException;
import io.ebean.datasource.DataSourcePool;
import io.ebean.datasource.DataSourcePoolListener;
import io.ebean.datasource.InitDatabase;
import io.ebean.datasource.PoolStatistics;
import io.ebean.datasource.PoolStatus;
import io.ebean.datasource.pool.PooledConnection;
import io.ebean.datasource.pool.PooledConnectionQueue;
import io.ebean.datasource.pool.TransactionIsolation;
import java.io.PrintWriter;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Statement;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.sql.DataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionPool
implements DataSourcePool {
    private static final Logger logger = LoggerFactory.getLogger(ConnectionPool.class);
    private final String name;
    private final DataSourceConfig config;
    private final DataSourceAlert notify;
    private final DataSourcePoolListener poolListener;
    private final Properties connectionProps;
    private final List<String> initSql;
    private final String databaseUrl;
    private final String databaseDriver;
    private final String heartbeatsql;
    private final int heartbeatFreqSecs;
    private final int heartbeatTimeoutSeconds;
    private final long trimPoolFreqMillis;
    private final int transactionIsolation;
    private final boolean autoCommit;
    private final boolean readOnly;
    private final boolean failOnStart;
    private final int maxInactiveMillis;
    private final long maxAgeMillis;
    private boolean captureStackTrace;
    private final int maxStackTraceSize;
    private boolean dataSourceDownAlertSent;
    private long lastTrimTime;
    private boolean dataSourceUp = true;
    private SQLException dataSourceDownReason;
    private AtomicBoolean inWarningMode = new AtomicBoolean();
    private int minConnections;
    private int maxConnections;
    private int warningSize;
    private final int waitTimeoutMillis;
    private int pstmtCacheSize;
    private final PooledConnectionQueue queue;
    private final Timer heartBeatTimer;
    private long leakTimeMinutes;

    public ConnectionPool(String name, DataSourceConfig params) {
        this.config = params;
        this.name = name;
        this.notify = params.getAlert();
        this.poolListener = params.getListener();
        this.autoCommit = params.isAutoCommit();
        this.readOnly = params.isReadOnly();
        this.failOnStart = params.isFailOnStart();
        this.initSql = params.getInitSql();
        this.transactionIsolation = params.getIsolationLevel();
        this.maxInactiveMillis = 1000 * params.getMaxInactiveTimeSecs();
        this.maxAgeMillis = 60000 * params.getMaxAgeMinutes();
        this.leakTimeMinutes = params.getLeakTimeMinutes();
        this.captureStackTrace = params.isCaptureStackTrace();
        this.maxStackTraceSize = params.getMaxStackTraceSize();
        this.databaseDriver = params.getDriver();
        this.databaseUrl = params.getUrl();
        this.pstmtCacheSize = params.getPstmtCacheSize();
        this.minConnections = params.getMinConnections();
        this.maxConnections = params.getMaxConnections();
        this.waitTimeoutMillis = params.getWaitTimeoutMillis();
        this.heartbeatsql = params.getHeartbeatSql();
        this.heartbeatFreqSecs = params.getHeartbeatFreqSecs();
        this.heartbeatTimeoutSeconds = params.getHeartbeatTimeoutSeconds();
        this.trimPoolFreqMillis = 1000 * params.getTrimPoolFreqSecs();
        this.queue = new PooledConnectionQueue(this);
        String un = params.getUsername();
        String pw = params.getPassword();
        if (un == null) {
            throw new DataSourceConfigurationException("DataSource user is null?");
        }
        if (pw == null) {
            throw new DataSourceConfigurationException("DataSource password is null?");
        }
        this.connectionProps = new Properties();
        this.connectionProps.setProperty("user", un);
        this.connectionProps.setProperty("password", pw);
        Map customProperties = params.getCustomProperties();
        if (customProperties != null) {
            Set entrySet = customProperties.entrySet();
            for (Map.Entry entry : entrySet) {
                this.connectionProps.setProperty((String)entry.getKey(), (String)entry.getValue());
            }
        }
        try {
            this.initialise();
            int freqMillis = this.heartbeatFreqSecs * 1000;
            this.heartBeatTimer = new Timer(name + ".heartBeat", true);
            if (freqMillis > 0) {
                this.heartBeatTimer.scheduleAtFixedRate((TimerTask)new HeartBeatRunnable(), freqMillis, (long)freqMillis);
            }
        }
        catch (SQLException ex) {
            throw new DataSourceInitialiseException("Error initialising DataSource", (Throwable)ex);
        }
    }

    public java.util.logging.Logger getParentLogger() throws SQLFeatureNotSupportedException {
        throw new SQLFeatureNotSupportedException("We do not support java.util.logging");
    }

    private void initialise() throws SQLException {
        try {
            ClassLoader contextLoader = Thread.currentThread().getContextClassLoader();
            if (contextLoader != null) {
                Class.forName(this.databaseDriver, true, contextLoader);
            } else {
                Class.forName(this.databaseDriver, true, this.getClass().getClassLoader());
            }
        }
        catch (Throwable e) {
            throw new IllegalStateException("Problem loading Database Driver [" + this.databaseDriver + "]: " + e.getMessage(), e);
        }
        String transIsolation = TransactionIsolation.getDescription(this.transactionIsolation);
        StringBuilder sb = new StringBuilder(70);
        sb.append("DataSourcePool [").append(this.name);
        sb.append("] autoCommit[").append(this.autoCommit);
        sb.append("] transIsolation[").append(transIsolation);
        sb.append("] min[").append(this.minConnections);
        sb.append("] max[").append(this.maxConnections).append("]");
        logger.info(sb.toString());
        if (this.config.useInitDatabase()) {
            this.initialiseDatabase();
        }
        try {
            this.queue.ensureMinimumConnections();
        }
        catch (SQLException e) {
            if (this.failOnStart) {
                throw e;
            }
            logger.error("Error trying to ensure minimum connections. Maybe db server is down.", (Throwable)e);
        }
    }

    private void initialiseDatabase() throws SQLException {
        try (Connection connection = this.createUnpooledConnection(this.connectionProps, false);){
            connection.clearWarnings();
        }
        catch (SQLException e) {
            logger.info("Obtaining connection using ownerUsername:{} to initialise database", (Object)this.config.getOwnerUsername());
            try (Connection ownerConnection = this.createUnpooledConnection(this.config.getOwnerUsername(), this.config.getOwnerPassword());){
                InitDatabase initDatabase = this.config.getInitDatabase();
                initDatabase.run(ownerConnection, this.config);
                ownerConnection.commit();
            }
            catch (SQLException e2) {
                throw new SQLException("Failed to run InitDatabase with ownerUsername:" + this.config.getOwnerUsername(), e2);
            }
        }
    }

    public boolean isWrapperFor(Class<?> arg0) throws SQLException {
        return false;
    }

    public <T> T unwrap(Class<T> arg0) throws SQLException {
        throw new SQLException("Not Implemented");
    }

    public String getName() {
        return this.name;
    }

    int getMaxStackTraceSize() {
        return this.maxStackTraceSize;
    }

    public boolean isDataSourceUp() {
        return this.dataSourceUp;
    }

    public SQLException getDataSourceDownReason() {
        return this.dataSourceDownReason;
    }

    protected void notifyWarning(String msg) {
        if (this.inWarningMode.compareAndSet(false, true)) {
            logger.warn(msg);
            if (this.notify != null) {
                this.notify.dataSourceWarning((DataSource)((Object)this), msg);
            }
        }
    }

    private synchronized void notifyDataSourceIsDown(SQLException ex) {
        if (this.dataSourceUp) {
            this.reset();
        }
        this.dataSourceUp = false;
        if (ex != null) {
            this.dataSourceDownReason = ex;
        }
        if (!this.dataSourceDownAlertSent) {
            this.dataSourceDownAlertSent = true;
            logger.error("FATAL: DataSourcePool [" + this.name + "] is down or has network error!!!", (Throwable)ex);
            if (this.notify != null) {
                this.notify.dataSourceDown((DataSource)((Object)this), ex);
            }
        }
    }

    private synchronized void notifyDataSourceIsUp() {
        if (this.dataSourceDownAlertSent) {
            this.dataSourceDownAlertSent = false;
            logger.error("RESOLVED FATAL: DataSourcePool [" + this.name + "] is back up!");
            if (this.notify != null) {
                this.notify.dataSourceUp((DataSource)((Object)this));
            }
        } else if (!this.dataSourceUp) {
            logger.info("DataSourcePool [" + this.name + "] is back up!");
        }
        if (!this.dataSourceUp) {
            this.dataSourceUp = true;
            this.dataSourceDownReason = null;
            this.reset();
        }
    }

    private void trimIdleConnections() {
        if (System.currentTimeMillis() > this.lastTrimTime + this.trimPoolFreqMillis) {
            try {
                this.queue.trim(this.maxInactiveMillis, this.maxAgeMillis);
                this.lastTrimTime = System.currentTimeMillis();
            }
            catch (Exception e) {
                logger.error("Error trying to trim idle connections", (Throwable)e);
            }
        }
    }

    private void checkDataSource() {
        this.trimIdleConnections();
        Connection conn = null;
        try {
            conn = this.getConnection();
            if (this.testConnection(conn)) {
                this.notifyDataSourceIsUp();
            } else {
                this.notifyDataSourceIsDown(null);
            }
        }
        catch (SQLException ex) {
            this.notifyDataSourceIsDown(ex);
        }
        finally {
            try {
                if (conn != null) {
                    conn.close();
                }
            }
            catch (SQLException ex) {
                logger.warn("Can't close connection in checkDataSource!");
            }
        }
    }

    private void initConnection(Connection conn) throws SQLException {
        conn.setAutoCommit(this.autoCommit);
        if (conn.getTransactionIsolation() != this.transactionIsolation) {
            conn.setTransactionIsolation(this.transactionIsolation);
        }
        if (this.readOnly) {
            conn.setReadOnly(this.readOnly);
        }
        if (this.initSql != null) {
            for (String query : this.initSql) {
                Statement stmt = conn.createStatement();
                Throwable throwable = null;
                try {
                    stmt.execute(query);
                }
                catch (Throwable throwable2) {
                    throwable = throwable2;
                    throw throwable2;
                }
                finally {
                    if (stmt == null) continue;
                    if (throwable != null) {
                        try {
                            stmt.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                        continue;
                    }
                    stmt.close();
                }
            }
        }
    }

    public Connection createUnpooledConnection(String username, String password) throws SQLException {
        Properties properties = new Properties(this.connectionProps);
        properties.setProperty("user", username);
        properties.setProperty("password", password);
        return this.createUnpooledConnection(properties, true);
    }

    public Connection createUnpooledConnection() throws SQLException {
        return this.createUnpooledConnection(this.connectionProps, true);
    }

    private Connection createUnpooledConnection(Properties properties, boolean notifyIsDown) throws SQLException {
        try {
            Connection conn = DriverManager.getConnection(this.databaseUrl, properties);
            this.initConnection(conn);
            return conn;
        }
        catch (SQLException ex) {
            if (notifyIsDown) {
                this.notifyDataSourceIsDown(null);
            }
            throw ex;
        }
    }

    public void setMaxSize(int max) {
        this.queue.setMaxSize(max);
        this.maxConnections = max;
    }

    public int getMaxSize() {
        return this.maxConnections;
    }

    public void setMinSize(int min) {
        this.queue.setMinSize(min);
        this.minConnections = min;
    }

    public int getMinSize() {
        return this.minConnections;
    }

    public void setWarningSize(int warningSize) {
        this.queue.setWarningSize(warningSize);
        this.warningSize = warningSize;
    }

    public int getWarningSize() {
        return this.warningSize;
    }

    public int getWaitTimeoutMillis() {
        return this.waitTimeoutMillis;
    }

    public int getMaxInactiveMillis() {
        return this.maxInactiveMillis;
    }

    public long getMaxAgeMillis() {
        return this.maxAgeMillis;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean testConnection(Connection conn) throws SQLException {
        if (this.heartbeatsql == null) {
            return conn.isValid(this.heartbeatTimeoutSeconds);
        }
        Statement stmt = null;
        ResultSet rset = null;
        try {
            stmt = conn.createStatement();
            if (this.heartbeatTimeoutSeconds > 0) {
                stmt.setQueryTimeout(this.heartbeatTimeoutSeconds);
            }
            rset = stmt.executeQuery(this.heartbeatsql);
            conn.commit();
            boolean bl = true;
            return bl;
        }
        finally {
            try {
                if (rset != null) {
                    rset.close();
                }
            }
            catch (SQLException e) {
                logger.error(null, (Throwable)e);
            }
            try {
                if (stmt != null) {
                    stmt.close();
                }
            }
            catch (SQLException e) {
                logger.error(null, (Throwable)e);
            }
        }
    }

    boolean validateConnection(PooledConnection conn) {
        try {
            return this.testConnection(conn);
        }
        catch (Exception e) {
            logger.warn("heartbeatsql test failed on connection[" + conn.getName() + "]");
            return false;
        }
    }

    void returnConnection(PooledConnection pooledConnection) {
        this.returnTheConnection(pooledConnection, false);
    }

    void returnConnectionForceClose(PooledConnection pooledConnection) {
        this.returnTheConnection(pooledConnection, true);
    }

    private void returnTheConnection(PooledConnection pooledConnection, boolean forceClose) {
        if (this.poolListener != null && !forceClose) {
            this.poolListener.onBeforeReturnConnection((Connection)pooledConnection);
        }
        this.queue.returnPooledConnection(pooledConnection, forceClose);
        if (forceClose) {
            this.checkDataSource();
        }
    }

    void reportClosingConnection(PooledConnection pooledConnection) {
        this.queue.reportClosingConnection(pooledConnection);
    }

    public String getBusyConnectionInformation() {
        return this.queue.getBusyConnectionInformation();
    }

    public void dumpBusyConnectionInformation() {
        this.queue.dumpBusyConnectionInformation();
    }

    public void closeBusyConnections(long leakTimeMinutes) {
        this.queue.closeBusyConnections(leakTimeMinutes);
    }

    PooledConnection createConnectionForQueue(int connId) throws SQLException {
        try {
            Connection c = this.createUnpooledConnection();
            PooledConnection pc = new PooledConnection(this, connId, c);
            pc.resetForUse();
            if (!this.dataSourceUp) {
                this.notifyDataSourceIsUp();
            }
            return pc;
        }
        catch (SQLException ex) {
            this.notifyDataSourceIsDown(ex);
            throw ex;
        }
    }

    public void reset() {
        this.queue.reset(this.leakTimeMinutes);
        this.inWarningMode.set(false);
    }

    public Connection getConnection() throws SQLException {
        return this.getPooledConnection();
    }

    private PooledConnection getPooledConnection() throws SQLException {
        PooledConnection c = this.queue.getPooledConnection();
        if (this.captureStackTrace) {
            c.setStackTrace(Thread.currentThread().getStackTrace());
        }
        if (this.poolListener != null) {
            this.poolListener.onAfterBorrowConnection((Connection)c);
        }
        return c;
    }

    public void testAlert() {
        String msg = "Just testing if alert message is sent successfully.";
        if (this.notify != null) {
            this.notify.dataSourceWarning((DataSource)((Object)this), msg);
        }
    }

    public void shutdown(boolean deregisterDriver) {
        this.heartBeatTimer.cancel();
        this.queue.shutdown();
        if (deregisterDriver) {
            this.deregisterDriver();
        }
    }

    public boolean isAutoCommit() {
        return this.autoCommit;
    }

    int getTransactionIsolation() {
        return this.transactionIsolation;
    }

    public boolean isCaptureStackTrace() {
        return this.captureStackTrace;
    }

    public void setCaptureStackTrace(boolean captureStackTrace) {
        this.captureStackTrace = captureStackTrace;
    }

    public Connection getConnection(String username, String password) throws SQLException {
        Properties props = new Properties();
        props.putAll((Map<?, ?>)this.connectionProps);
        props.setProperty("user", username);
        props.setProperty("password", password);
        Connection conn = DriverManager.getConnection(this.databaseUrl, props);
        this.initConnection(conn);
        return conn;
    }

    public int getLoginTimeout() throws SQLException {
        throw new SQLException("Method not supported");
    }

    public void setLoginTimeout(int seconds) throws SQLException {
        throw new SQLException("Method not supported");
    }

    public PrintWriter getLogWriter() {
        return null;
    }

    public void setLogWriter(PrintWriter writer) throws SQLException {
        throw new SQLException("Method not supported");
    }

    public void setLeakTimeMinutes(long leakTimeMinutes) {
        this.leakTimeMinutes = leakTimeMinutes;
    }

    public long getLeakTimeMinutes() {
        return this.leakTimeMinutes;
    }

    public int getPstmtCacheSize() {
        return this.pstmtCacheSize;
    }

    public void setPstmtCacheSize(int pstmtCacheSize) {
        this.pstmtCacheSize = pstmtCacheSize;
    }

    public PoolStatus getStatus(boolean reset) {
        return this.queue.getStatus(reset);
    }

    public PoolStatistics getStatistics(boolean reset) {
        return this.queue.getStatistics(reset);
    }

    private void deregisterDriver() {
        try {
            logger.debug("Deregister the JDBC driver " + this.databaseDriver);
            DriverManager.deregisterDriver(DriverManager.getDriver(this.databaseUrl));
        }
        catch (SQLException e) {
            logger.warn("Error trying to deregister the JDBC driver " + this.databaseDriver, (Throwable)e);
        }
    }

    public static class Status
    implements PoolStatus {
        private final int minSize;
        private final int maxSize;
        private final int free;
        private final int busy;
        private final int waiting;
        private final int highWaterMark;
        private final int waitCount;
        private final int hitCount;

        protected Status(int minSize, int maxSize, int free, int busy, int waiting, int highWaterMark, int waitCount, int hitCount) {
            this.minSize = minSize;
            this.maxSize = maxSize;
            this.free = free;
            this.busy = busy;
            this.waiting = waiting;
            this.highWaterMark = highWaterMark;
            this.waitCount = waitCount;
            this.hitCount = hitCount;
        }

        public String toString() {
            return "min[" + this.minSize + "] max[" + this.maxSize + "] free[" + this.free + "] busy[" + this.busy + "] waiting[" + this.waiting + "] highWaterMark[" + this.highWaterMark + "] waitCount[" + this.waitCount + "] hitCount[" + this.hitCount + "]";
        }

        public int getMinSize() {
            return this.minSize;
        }

        public int getMaxSize() {
            return this.maxSize;
        }

        public int getFree() {
            return this.free;
        }

        public int getBusy() {
            return this.busy;
        }

        public int getWaiting() {
            return this.waiting;
        }

        public int getHighWaterMark() {
            return this.highWaterMark;
        }

        public int getWaitCount() {
            return this.waitCount;
        }

        public int getHitCount() {
            return this.hitCount;
        }
    }

    class HeartBeatRunnable
    extends TimerTask {
        HeartBeatRunnable() {
        }

        @Override
        public void run() {
            ConnectionPool.this.checkDataSource();
        }
    }
}

