/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.storage.journal;

import io.atomix.storage.StorageException;
import io.atomix.storage.buffer.Buffer;
import io.atomix.storage.buffer.FileBuffer;
import io.atomix.storage.buffer.HeapBuffer;
import io.atomix.storage.buffer.MappedBuffer;
import io.atomix.storage.buffer.SlicedBuffer;
import io.atomix.storage.journal.Indexed;
import io.atomix.storage.journal.JournalSegmentCache;
import io.atomix.storage.journal.JournalSegmentDescriptor;
import io.atomix.storage.journal.JournalWriter;
import io.atomix.storage.journal.index.JournalIndex;
import io.atomix.utils.serializer.Serializer;
import java.nio.BufferOverflowException;
import java.util.zip.CRC32;

public class JournalSegmentWriter<E>
implements JournalWriter<E> {
    private final JournalSegmentDescriptor descriptor;
    private final int maxEntrySize;
    private final JournalSegmentCache cache;
    private final JournalIndex index;
    private final Buffer buffer;
    private final Serializer serializer;
    private final Buffer memory = HeapBuffer.allocate().flip();
    private final long firstIndex;
    private Indexed<E> lastEntry;

    public JournalSegmentWriter(JournalSegmentDescriptor descriptor, int maxEntrySize, JournalSegmentCache cache, JournalIndex index, Serializer serializer) {
        this.descriptor = descriptor;
        this.maxEntrySize = maxEntrySize;
        this.cache = cache;
        this.index = index;
        this.buffer = descriptor.buffer().slice();
        this.serializer = serializer;
        this.firstIndex = descriptor.index();
        this.reset(0L);
    }

    private void reset(long index) {
        long nextIndex = this.firstIndex;
        this.buffer.clear();
        this.memory.clear().flip();
        int position = this.buffer.position();
        if (this.memory.remaining() < this.maxEntrySize) {
            this.buffer.mark().read(this.memory.clear().limit(this.maxEntrySize * 2)).reset();
            this.memory.flip();
        }
        int length = this.memory.mark().readInt();
        while (0 < length && length <= this.maxEntrySize && (index == 0L || nextIndex <= index)) {
            long checksum = this.memory.readUnsignedInt();
            byte[] bytes = new byte[length];
            this.memory.read(bytes);
            CRC32 crc32 = new CRC32();
            crc32.update(bytes, 0, length);
            if (checksum != crc32.getValue()) break;
            Object entry = this.serializer.decode(bytes);
            this.lastEntry = new Indexed<Object>(nextIndex, entry, length);
            this.index.index(nextIndex, position);
            ++nextIndex;
            position = this.buffer.position() + this.memory.position();
            if (this.memory.remaining() < this.maxEntrySize) {
                this.buffer.skip(this.memory.position()).mark().read(this.memory.clear().limit(this.maxEntrySize * 2)).reset();
                this.memory.flip();
            }
            length = this.memory.mark().readInt();
        }
        this.buffer.skip(this.memory.reset().position());
    }

    @Override
    public long getLastIndex() {
        return this.lastEntry != null ? this.lastEntry.index() : this.descriptor.index() - 1L;
    }

    @Override
    public Indexed<E> getLastEntry() {
        return this.lastEntry;
    }

    @Override
    public long getNextIndex() {
        if (this.lastEntry != null) {
            return this.lastEntry.index() + 1L;
        }
        return this.firstIndex;
    }

    public long size() {
        return this.buffer.offset() + this.buffer.position();
    }

    public boolean isEmpty() {
        return this.lastEntry == null;
    }

    public boolean isFull() {
        return this.size() >= (long)this.descriptor.maxSegmentSize() || this.getNextIndex() - this.firstIndex >= (long)this.descriptor.maxEntries();
    }

    public long firstIndex() {
        return this.firstIndex;
    }

    @Override
    public void append(Indexed<E> entry) {
        long nextIndex = this.getNextIndex();
        if (entry.index() > nextIndex) {
            throw new IndexOutOfBoundsException("Entry index is not sequential");
        }
        if (entry.index() < nextIndex) {
            this.truncate(entry.index() - 1L);
        }
        this.append(entry.entry());
    }

    @Override
    public <T extends E> Indexed<T> append(T entry) {
        long index = this.getNextIndex();
        byte[] bytes = this.serializer.encode(entry);
        int length = bytes.length;
        if (this.buffer.remaining() < length + 4 + 4) {
            throw new BufferOverflowException();
        }
        if (length > this.maxEntrySize) {
            throw new StorageException.TooLarge("Entry size " + length + " exceeds maximum allowed bytes (" + this.maxEntrySize + ")");
        }
        CRC32 crc32 = new CRC32();
        crc32.update(bytes, 0, length);
        long checksum = crc32.getValue();
        int position = this.buffer.position();
        this.buffer.write(this.memory.clear().writeInt(length).writeUnsignedInt(checksum).write(bytes).flip());
        Indexed<T> indexedEntry = new Indexed<T>(index, entry, length);
        this.lastEntry = indexedEntry;
        this.cache.put(indexedEntry);
        this.index.index(index, position);
        return indexedEntry;
    }

    @Override
    public void truncate(long index) {
        if (index >= this.getLastIndex()) {
            return;
        }
        this.lastEntry = null;
        if (index < this.descriptor.index()) {
            ((Buffer)this.buffer.zero()).clear();
            this.cache.truncate(index);
            this.index.truncate(index);
            return;
        }
        this.cache.truncate(index);
        this.index.truncate(index);
        this.reset(index);
        this.buffer.zero(this.buffer.position());
    }

    @Override
    public void flush() {
        this.buffer.flush();
    }

    @Override
    public void close() {
        this.buffer.close();
    }

    void delete() {
        Buffer buffer;
        Buffer buffer2 = buffer = this.buffer instanceof SlicedBuffer ? ((SlicedBuffer)this.buffer).root() : this.buffer;
        if (buffer instanceof FileBuffer) {
            ((FileBuffer)buffer).delete();
        } else if (buffer instanceof MappedBuffer) {
            ((MappedBuffer)buffer).delete();
        }
    }
}

