/*
 * Decompiled with CFR 0.152.
 */
package io.ably.lib.http;

import com.google.gson.JsonParseException;
import io.ably.lib.debug.DebugOptions;
import io.ably.lib.http.HttpAuth;
import io.ably.lib.network.FailedConnectionException;
import io.ably.lib.network.HttpBody;
import io.ably.lib.network.HttpEngine;
import io.ably.lib.network.HttpEngineConfig;
import io.ably.lib.network.HttpEngineFactory;
import io.ably.lib.network.HttpRequest;
import io.ably.lib.network.HttpResponse;
import io.ably.lib.rest.Auth;
import io.ably.lib.transport.Defaults;
import io.ably.lib.transport.Hosts;
import io.ably.lib.types.AblyException;
import io.ably.lib.types.ClientOptions;
import io.ably.lib.types.ErrorInfo;
import io.ably.lib.types.ErrorResponse;
import io.ably.lib.types.Param;
import io.ably.lib.types.ProxyOptions;
import io.ably.lib.util.AgentHeaderCreator;
import io.ably.lib.util.Base64Coder;
import io.ably.lib.util.ClientOptionsUtils;
import io.ably.lib.util.Log;
import io.ably.lib.util.PlatformAgentProvider;
import java.io.IOException;
import java.lang.reflect.Field;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

public class HttpCore {
    private static final String TAG = HttpCore.class.getName();
    public final String scheme;
    public final int port;
    final ClientOptions options;
    final Hosts hosts;
    private final Auth auth;
    private final PlatformAgentProvider platformAgentProvider;
    private final HttpEngine engine;
    private HttpAuth proxyAuth;
    private Map<String, String> dynamicAgents;

    public HttpCore(ClientOptions options, Auth auth, PlatformAgentProvider platformAgentProvider) throws AblyException {
        this.options = options;
        this.auth = auth;
        this.platformAgentProvider = platformAgentProvider;
        this.scheme = options.tls ? "https://" : "http://";
        this.port = Defaults.getPort(options);
        this.hosts = new Hosts(options.restHost, "rest.ably.io", options);
        ProxyOptions proxyOptions = options.proxy;
        if (proxyOptions != null) {
            String proxyHost = proxyOptions.host;
            if (proxyHost == null) {
                throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy host", 40000, 400));
            }
            int proxyPort = proxyOptions.port;
            if (proxyPort == 0) {
                throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy port", 40000, 400));
            }
            String proxyUser = proxyOptions.username;
            if (proxyUser != null) {
                String proxyPassword = proxyOptions.password;
                if (proxyPassword == null) {
                    throw AblyException.fromErrorInfo(new ErrorInfo("Unable to configure proxy without proxy password", 40000, 400));
                }
                this.proxyAuth = new HttpAuth(proxyUser, proxyPassword, proxyOptions.prefAuthType);
            }
        }
        HttpEngineFactory engineFactory = HttpEngineFactory.getFirstAvailable();
        Log.v(TAG, String.format("Using %s HTTP Engine", engineFactory.getEngineType().name()));
        this.engine = engineFactory.create(new HttpEngineConfig(ClientOptionsUtils.convertToProxyConfig(options)));
    }

    private HttpCore(HttpCore underlyingHttpCore, Map<String, String> dynamicAgents) {
        this.options = underlyingHttpCore.options;
        this.auth = underlyingHttpCore.auth;
        this.platformAgentProvider = underlyingHttpCore.platformAgentProvider;
        this.scheme = underlyingHttpCore.scheme;
        this.port = underlyingHttpCore.port;
        this.hosts = underlyingHttpCore.hosts;
        this.proxyAuth = underlyingHttpCore.proxyAuth;
        this.engine = underlyingHttpCore.engine;
        this.dynamicAgents = dynamicAgents;
    }

    public <T> T httpExecuteWithRetry(URL url, String method, Param[] headers, RequestBody requestBody, ResponseHandler<T> responseHandler, boolean requireAblyAuth) throws AblyException {
        boolean renewPending = true;
        boolean proxyAuthPending = true;
        if (requireAblyAuth) {
            this.authorize(false);
        }
        while (true) {
            try {
                return this.httpExecute(url, method, headers, requestBody, true, responseHandler);
            }
            catch (AuthRequiredException are) {
                if (are.authChallenge != null && requireAblyAuth && are.expired && renewPending) {
                    this.authorize(true);
                    renewPending = false;
                    continue;
                }
                if (are.proxyAuthChallenge != null && proxyAuthPending && this.proxyAuth != null) {
                    this.proxyAuth.processAuthenticateHeaders(are.proxyAuthChallenge);
                    proxyAuthPending = false;
                    continue;
                }
                throw are;
            }
            break;
        }
    }

    public String getPreferredHost() {
        return this.hosts.getPreferredHost();
    }

    public void setPreferredHost(String host) {
        this.hosts.setPreferredHost(host, false);
    }

    public String getPrimaryHost() {
        return this.hosts.getPrimaryHost();
    }

    void authorize(boolean renew) throws AblyException {
        this.auth.assertAuthorizationHeader(renew);
    }

    public <T> T httpExecute(URL url, String method, Param[] headers, RequestBody requestBody, boolean withCredentials, ResponseHandler<T> responseHandler) throws AblyException {
        boolean withProxyCredentials = this.engine.isUsingProxy() && this.proxyAuth != null;
        return this.httpExecute(url, method, headers, requestBody, withCredentials, withProxyCredentials, responseHandler);
    }

    <T> T httpExecute(URL url, String method, Param[] headers, RequestBody requestBody, boolean withCredentials, boolean withProxyCredentials, ResponseHandler<T> responseHandler) throws AblyException {
        Response response;
        HttpRequest.HttpRequestBuilder requestBuilder = HttpRequest.builder();
        requestBuilder.url(url).method(method).httpOpenTimeout(this.options.httpOpenTimeout).httpReadTimeout(this.options.httpRequestTimeout).body(requestBody != null ? new HttpBody(requestBody.getContentType(), requestBody.getEncoded()) : null);
        Map<String, String> requestHeaders = this.collectRequestHeaders(url, method, headers, requestBody, withCredentials, withProxyCredentials);
        boolean credentialsIncluded = requestHeaders.containsKey("Authorization");
        String authHeader = requestHeaders.get("Authorization");
        requestBuilder.headers(requestHeaders);
        HttpRequest request = requestBuilder.build();
        if (Log.level <= 2 && request.getBody() != null && request.getBody().getContent() != null) {
            Log.v(TAG, System.lineSeparator() + new String(request.getBody().getContent()));
        }
        Map requestProperties = request.getHeaders();
        if (Log.level <= 2) {
            Log.v(TAG, "HTTP request: " + url + " " + method);
            if (credentialsIncluded) {
                Log.v(TAG, "  Authorization: " + authHeader);
            }
            for (Map.Entry entry : requestProperties.entrySet()) {
                for (String val : (List)entry.getValue()) {
                    Log.v(TAG, "  " + (String)entry.getKey() + ": " + val);
                }
            }
            if (requestBody != null) {
                Log.v(TAG, "  Content-Type: " + requestBody.getContentType());
                Log.v(TAG, "  Content-Length: " + (requestBody.getEncoded() != null ? requestBody.getEncoded().length : 0));
            }
        }
        DebugOptions.RawHttpListener rawHttpListener = null;
        String id = null;
        if (this.options instanceof DebugOptions && (rawHttpListener = ((DebugOptions)this.options).httpListener) != null && (response = rawHttpListener.onRawHttpRequest(id = String.valueOf(Math.random()).substring(2), request, credentialsIncluded ? authHeader : null, requestProperties, requestBody)) != null) {
            return this.handleResponse(credentialsIncluded, response, responseHandler);
        }
        try {
            response = this.executeRequest(request);
        }
        catch (FailedConnectionException exception) {
            throw AblyException.fromThrowable(exception);
        }
        catch (Exception e) {
            if (e.getCause() instanceof IOException) {
                throw AblyException.fromThrowable(e.getCause());
            }
            throw AblyException.fromThrowable(e);
        }
        if (rawHttpListener != null) {
            rawHttpListener.onRawHttpResponse(id, method, response);
        }
        return this.handleResponse(credentialsIncluded, response, responseHandler);
    }

    private Map<String, String> collectRequestHeaders(URL url, String method, Param[] headers, RequestBody requestBody, boolean withCredentials, boolean withProxyCredentials) throws AblyException {
        HashMap<String, String> requestHeaders = new HashMap<String, String>();
        String authHeader = Param.getFirst(headers, "Authorization");
        if (authHeader == null && this.auth != null) {
            authHeader = this.auth.getAuthorizationHeader();
        }
        if (withCredentials && authHeader != null) {
            requestHeaders.put("Authorization", authHeader);
        }
        if (withProxyCredentials && this.proxyAuth.hasChallenge()) {
            byte[] encodedRequestBody = requestBody != null ? requestBody.getEncoded() : null;
            String string = this.proxyAuth.getAuthorizationHeader(method, url.getPath(), encodedRequestBody);
            requestHeaders.put("Proxy-Authorization", string);
        }
        boolean acceptSet = false;
        if (headers != null) {
            for (Param header : headers) {
                requestHeaders.put(header.key, header.value);
                if (!header.key.equals("Accept")) continue;
                acceptSet = true;
            }
        }
        if (!acceptSet) {
            requestHeaders.put("Accept", "application/json");
        }
        if (!requestHeaders.containsKey("X-Ably-Version")) {
            requestHeaders.put("X-Ably-Version", "4");
        }
        HashMap<String, String> hashMap = new HashMap<String, String>();
        if (this.options.agents != null) {
            hashMap.putAll(this.options.agents);
        }
        if (this.dynamicAgents != null) {
            hashMap.putAll(this.dynamicAgents);
        }
        requestHeaders.put("Ably-Agent", AgentHeaderCreator.create(hashMap, this.platformAgentProvider));
        if (this.options.clientId != null) {
            requestHeaders.put("X-Ably-ClientId", Base64Coder.encodeString(this.options.clientId));
        }
        return requestHeaders;
    }

    private <T> T handleResponse(boolean credentialsIncluded, Response response, ResponseHandler<T> responseHandler) throws AblyException {
        List<String> proxyAuthHeaders;
        if (response.statusCode == 0) {
            return null;
        }
        if (response.statusCode >= 500 && response.statusCode <= 504) {
            ErrorInfo error = ErrorInfo.fromResponseStatus(response.statusLine, response.statusCode);
            throw AblyException.fromErrorInfo(error);
        }
        if (response.statusCode >= 200 && response.statusCode < 300) {
            return responseHandler != null ? (T)responseHandler.handleResponse(response, null) : null;
        }
        ErrorInfo error = null;
        if (response.body != null && response.body.length > 0) {
            if (response.contentType != null && response.contentType.contains("msgpack")) {
                try {
                    error = ErrorInfo.fromMsgpackBody(response.body);
                }
                catch (IOException e) {
                    System.err.println("Unable to parse msgpack error response");
                }
            } else {
                String bodyText = new String(response.body);
                try {
                    ErrorResponse errorResponse = ErrorResponse.fromJSON(bodyText);
                    if (errorResponse != null) {
                        error = errorResponse.error;
                    }
                }
                catch (JsonParseException jse) {
                    System.err.println("Error message in unexpected format: " + bodyText);
                }
            }
        }
        if (error == null) {
            String errorCodeHeader = response.getHeaderField("X-Ably-ErrorCode");
            String errorMessageHeader = response.getHeaderField("X-Ably-ErrorMessage");
            if (errorCodeHeader != null) {
                try {
                    error = new ErrorInfo(errorMessageHeader, response.statusCode, Integer.parseInt(errorCodeHeader));
                }
                catch (NumberFormatException numberFormatException) {
                    // empty catch block
                }
            }
        }
        if (response.statusCode == 401) {
            boolean stale = error != null && error.code == 40140;
            List<String> wwwAuthHeaders = response.getHeaderFields("WWW-Authenticate");
            if (wwwAuthHeaders != null && wwwAuthHeaders.size() > 0) {
                Map<HttpAuth.Type, String> headersByType = HttpAuth.sortAuthenticateHeaders(wwwAuthHeaders);
                String tokenHeader = headersByType.get((Object)HttpAuth.Type.X_ABLY_TOKEN);
                if (tokenHeader != null) {
                    stale |= tokenHeader.indexOf("stale") > -1;
                }
                AuthRequiredException exception = new AuthRequiredException(null, error);
                exception.authChallenge = headersByType;
                if (stale) {
                    exception.expired = true;
                    throw exception;
                }
                if (!credentialsIncluded) {
                    throw exception;
                }
            }
        }
        if (response.statusCode == 407 && (proxyAuthHeaders = response.getHeaderFields("Proxy-Authenticate")) != null && !proxyAuthHeaders.isEmpty()) {
            AuthRequiredException exception = new AuthRequiredException(null, error);
            exception.proxyAuthChallenge = HttpAuth.sortAuthenticateHeaders(proxyAuthHeaders);
            throw exception;
        }
        if (error == null) {
            error = ErrorInfo.fromResponseStatus(response.statusLine, response.statusCode);
        }
        Log.e(TAG, "Error response from server: err = " + error);
        if (responseHandler != null) {
            return responseHandler.handleResponse(response, error);
        }
        throw AblyException.fromErrorInfo(error);
    }

    private Response executeRequest(HttpRequest request) {
        HttpResponse rawResponse = this.engine.call(request).execute();
        Response response = new Response();
        response.statusCode = rawResponse.getCode();
        response.statusLine = rawResponse.getMessage();
        Log.v(TAG, "HTTP response:");
        Map caseSensitiveHeaders = rawResponse.getHeaders();
        response.headers = new HashMap<String, List<String>>(caseSensitiveHeaders.size(), 1.0f);
        for (Map.Entry entry : caseSensitiveHeaders.entrySet()) {
            if (entry.getKey() == null) continue;
            response.headers.put(((String)entry.getKey()).toLowerCase(Locale.ROOT), (List)entry.getValue());
            if (Log.level > 2) continue;
            for (String val : (List)entry.getValue()) {
                Log.v(TAG, (String)entry.getKey() + ": " + val);
            }
        }
        if (response.statusCode == 204 || rawResponse.getBody() == null) {
            return response;
        }
        response.contentType = rawResponse.getBody().getContentType();
        response.body = rawResponse.getBody().getContent();
        int n = response.contentLength = response.body == null ? 0 : response.body.length;
        if (Log.level <= 2 && response.body != null) {
            Log.v(TAG, System.lineSeparator() + new String(response.body));
        }
        return response;
    }

    public HttpCore injectDynamicAgents(Map<String, String> wrapperSDKAgents) {
        return new HttpCore(this, wrapperSDKAgents);
    }

    static {
        Field androidVersionField = null;
        int androidVersion = 0;
        try {
            androidVersionField = Class.forName("android.os.Build$VERSION").getField("SDK_INT");
            androidVersion = androidVersionField.getInt(androidVersionField);
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (androidVersionField != null && androidVersion < 8) {
            System.setProperty("httpCore.keepAlive", "false");
        }
    }

    public static interface RequestBody {
        public byte[] getEncoded();

        public String getContentType();
    }

    public static interface ResponseHandler<T> {
        public T handleResponse(Response var1, ErrorInfo var2) throws AblyException;
    }

    public static class AuthRequiredException
    extends AblyException {
        private static final long serialVersionUID = 1L;
        public boolean expired;
        public Map<HttpAuth.Type, String> authChallenge;
        public Map<HttpAuth.Type, String> proxyAuthChallenge;

        public AuthRequiredException(Throwable throwable, ErrorInfo reason) {
            super(throwable, reason);
        }
    }

    public static class Response {
        public int statusCode;
        public String statusLine;
        public Map<String, List<String>> headers;
        public String contentType;
        public int contentLength;
        public byte[] body;

        public List<String> getHeaderFields(String name) {
            if (this.headers == null) {
                return null;
            }
            return this.headers.get(name.toLowerCase(Locale.ROOT));
        }

        public String getHeaderField(String name) {
            if (this.headers == null) {
                return null;
            }
            List<String> values = this.headers.get(name.toLowerCase(Locale.ROOT));
            if (values == null || values.isEmpty()) {
                return null;
            }
            return values.get(0);
        }
    }

    public static interface BodyHandler<T> {
        public T[] handleResponseBody(String var1, byte[] var2) throws AblyException;
    }
}

