/*
 * Decompiled with CFR 0.152.
 */
package io.ably.lib.transport;

import io.ably.lib.debug.DebugOptions;
import io.ably.lib.http.HttpHelpers;
import io.ably.lib.realtime.AblyRealtime;
import io.ably.lib.realtime.Channel;
import io.ably.lib.realtime.CompletionListener;
import io.ably.lib.realtime.Connection;
import io.ably.lib.realtime.ConnectionState;
import io.ably.lib.realtime.ConnectionStateListener;
import io.ably.lib.rest.Auth;
import io.ably.lib.transport.Defaults;
import io.ably.lib.transport.Hosts;
import io.ably.lib.transport.ITransport;
import io.ably.lib.transport.NetworkConnectivity;
import io.ably.lib.types.AblyException;
import io.ably.lib.types.ClientOptions;
import io.ably.lib.types.ConnectionDetails;
import io.ably.lib.types.ErrorInfo;
import io.ably.lib.types.ProtocolMessage;
import io.ably.lib.types.ProtocolSerializer;
import io.ably.lib.util.Log;
import io.ably.lib.util.PlatformAgentProvider;
import io.ably.lib.util.ReconnectionStrategy;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

public class ConnectionManager
implements ITransport.ConnectListener {
    final ExecutorService singleThreadExecutor = Executors.newSingleThreadExecutor();
    private static final String TAG = ConnectionManager.class.getName();
    private static final String INTERNET_CHECK_URL = "https://internet-up.ably-realtime.com/is-the-internet-up.txt";
    private static final String INTERNET_CHECK_OK = "yes";
    static ErrorInfo REASON_CLOSED = new ErrorInfo("Can't attach when not in an active state", 200, 10000);
    static ErrorInfo REASON_DISCONNECTED = new ErrorInfo("Connection temporarily unavailable", 503, 80003);
    static ErrorInfo REASON_SUSPENDED = new ErrorInfo("Connection unavailable", 503, 80002);
    static ErrorInfo REASON_FAILED = new ErrorInfo("Connection failed", 400, 80000);
    static ErrorInfo REASON_REFUSED = new ErrorInfo("Access refused", 401, 40100);
    static ErrorInfo REASON_TOO_BIG = new ErrorInfo("Connection closed; message too large", 400, 40000);
    final AblyRealtime ably;
    private final Channels channels;
    private final Connection connection;
    private final ITransport.Factory transportFactory;
    private final List<QueuedMessage> queuedMessages = new ArrayList<QueuedMessage>();
    private final PendingMessageQueue pendingMessages = new PendingMessageQueue();
    private final HashSet<Object> heartbeatWaiters = new HashSet();
    private final ActionQueue actionQueue = new ActionQueue();
    private final Hosts hosts;
    private final PlatformAgentProvider platformAgentProvider;
    private Thread handlerThread;
    private final Map<ConnectionState, State> states = new HashMap<ConnectionState, State>();
    private State currentState;
    private ErrorInfo stateError;
    private ConnectParams pendingConnect;
    private boolean suppressRetry;
    private ITransport transport;
    private long suspendTime;
    public long msgSerial;
    private long lastActivity;
    private CMConnectivityListener connectivityListener;
    private long connectionStateTtl = Defaults.connectionStateTtl;
    long maxIdleInterval = Defaults.maxIdleInterval;
    private int disconnectedRetryAttempt = 0;
    private final DebugOptions.RawProtocolListener protocolListener;
    private String lastUsedHost;
    private static final long HEARTBEAT_TIMEOUT = 5000L;

    public ErrorInfo getStateErrorInfo() {
        return this.stateError != null ? this.stateError : this.currentState.defaultErrorInfo;
    }

    public boolean isActive() {
        return this.currentState.queueEvents || this.currentState.sendEvents;
    }

    private synchronized void addAction(Action action) {
        this.actionQueue.add(action);
        this.notifyAll();
    }

    public ConnectionManager(AblyRealtime ably, Connection connection, Channels channels, PlatformAgentProvider platformAgentProvider) throws AblyException {
        this.ably = ably;
        this.connection = connection;
        this.channels = channels;
        this.platformAgentProvider = platformAgentProvider;
        ClientOptions options = ably.options;
        this.hosts = new Hosts(options.realtimeHost, "realtime.ably.io", options);
        ITransport.Factory transportFactory = null;
        DebugOptions.RawProtocolListener protocolListener = null;
        if (options instanceof DebugOptions) {
            protocolListener = ((DebugOptions)options).protocolListener;
            transportFactory = ((DebugOptions)options).transportFactory;
        }
        this.protocolListener = protocolListener;
        this.transportFactory = transportFactory != null ? transportFactory : Defaults.TRANSPORT;
        this.states.put(ConnectionState.initialized, new Initialized());
        this.states.put(ConnectionState.connecting, new Connecting());
        this.states.put(ConnectionState.connected, new Connected());
        this.states.put(ConnectionState.disconnected, new Disconnected());
        this.states.put(ConnectionState.suspended, new Suspended());
        this.states.put(ConnectionState.closing, new Closing());
        this.states.put(ConnectionState.closed, new Closed());
        this.states.put(ConnectionState.failed, new Failed());
        this.currentState = this.states.get((Object)ConnectionState.initialized);
        this.setSuspendTime();
    }

    public String getHost() {
        return this.lastUsedHost;
    }

    public synchronized State getConnectionState() {
        return this.currentState;
    }

    public synchronized void connect() {
        if (this.currentState.terminal || this.currentState.state == ConnectionState.initialized) {
            this.startup();
        }
        this.requestState(ConnectionState.connecting);
    }

    public void close() {
        this.requestState(ConnectionState.closing);
    }

    public void requestState(ConnectionState state) {
        this.requestState(new StateIndication(state, null));
    }

    public void requestState(StateIndication state) {
        this.requestState(null, state);
    }

    private synchronized void requestState(ITransport transport, StateIndication stateIndication) {
        Log.v(TAG, "requestState(): requesting " + (Object)((Object)stateIndication.state) + "; id = " + this.connection.id);
        this.addAction(new AsynchronousStateChangeAction(transport, stateIndication));
    }

    private synchronized ConnectionStateListener.ConnectionStateChange setState(ITransport transport, StateIndication stateIndication) {
        if (transport != null && transport != this.transport) {
            Log.v(TAG, "setState: action received for superseded transport; discarding");
            return null;
        }
        StateIndication validatedStateIndication = this.currentState.validateTransition(stateIndication);
        if (validatedStateIndication == null) {
            Log.v(TAG, "setState(): not transitioning; not a valid transition " + (Object)((Object)stateIndication.state));
            return null;
        }
        if (stateIndication.state == ConnectionState.connected || stateIndication.state == ConnectionState.suspended) {
            this.disconnectedRetryAttempt = 0;
        }
        if (stateIndication.state == ConnectionState.disconnected) {
            this.states.get((Object)((Object)ConnectionState.disconnected)).timeout = ReconnectionStrategy.getRetryTime(this.ably.options.disconnectedRetryTimeout, ++this.disconnectedRetryAttempt);
        }
        if (stateIndication.state == ConnectionState.closing || stateIndication.state == ConnectionState.closed || stateIndication.state == ConnectionState.suspended || stateIndication.state == ConnectionState.failed) {
            this.connection.id = null;
            this.connection.key = null;
        }
        ConnectionState newConnectionState = validatedStateIndication.state;
        State newState = this.states.get((Object)newConnectionState);
        ErrorInfo reason = validatedStateIndication.reason;
        if (reason == null) {
            reason = newState.defaultErrorInfo;
        }
        Log.v(TAG, "setState(): setting " + (Object)((Object)newState.state) + "; reason " + reason);
        ConnectionStateListener.ConnectionStateChange change = new ConnectionStateListener.ConnectionStateChange(this.currentState.state, newConnectionState, newState.timeout, reason);
        this.currentState = newState;
        this.stateError = reason;
        return change;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void ping(CompletionListener listener) {
        HeartbeatWaiter waiter = new HeartbeatWaiter(listener);
        if (this.currentState.state != ConnectionState.connected) {
            waiter.onError(new ErrorInfo("Unable to ping service; not connected", 40000, 400));
            return;
        }
        HashSet<Object> hashSet = this.heartbeatWaiters;
        synchronized (hashSet) {
            this.heartbeatWaiters.add(waiter);
            waiter.start();
        }
        try {
            this.send(new ProtocolMessage(ProtocolMessage.Action.heartbeat), false, null);
        }
        catch (AblyException e) {
            waiter.onError(e.errorInfo);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void onAuthUpdated(String token, boolean waitForResponse) throws AblyException {
        ConnectionWaiter waiter = new ConnectionWaiter();
        try {
            switch (this.currentState.state) {
                case connected: {
                    try {
                        ProtocolMessage msg = new ProtocolMessage(ProtocolMessage.Action.auth);
                        msg.auth = new ProtocolMessage.AuthDetails(token);
                        this.send(msg, false, null);
                    }
                    catch (AblyException e) {
                        Log.v(TAG, "onAuthUpdated: closing transport after send failure");
                        this.transport.close();
                    }
                    break;
                }
                case connecting: {
                    Log.v(TAG, "onAuthUpdated: closing connecting transport");
                    ErrorInfo disconnectError = new ErrorInfo("Aborting incomplete connection with superseded auth params", 503, 80003);
                    this.requestState(new StateIndication(ConnectionState.disconnected, disconnectError, null, null));
                    this.connect();
                    break;
                }
                default: {
                    this.connect();
                }
            }
            if (!waitForResponse) {
                return;
            }
            boolean waitingForConnected = true;
            block14: while (waitingForConnected) {
                ErrorInfo reason = waiter.waitForChange();
                ConnectionState connectionState = this.currentState.state;
                switch (connectionState) {
                    case connected: {
                        Log.v(TAG, "onAuthUpdated: got connected");
                        waitingForConnected = false;
                        continue block14;
                    }
                    case connecting: 
                    case disconnected: {
                        Log.v(TAG, "onAuthUpdated: " + (Object)((Object)connectionState));
                        continue block14;
                    }
                }
                Log.v(TAG, "onAuthUpdated: throwing exception");
                throw AblyException.fromErrorInfo(reason);
            }
        }
        finally {
            waiter.close();
        }
    }

    public void onAuthUpdatedAsync(String token, Auth.AuthUpdateResult authUpdateResult) {
        ConnectionWaiter waiter = new ConnectionWaiter();
        switch (this.currentState.state) {
            case connected: {
                try {
                    ProtocolMessage msg = new ProtocolMessage(ProtocolMessage.Action.auth);
                    msg.auth = new ProtocolMessage.AuthDetails(token);
                    this.send(msg, false, null);
                }
                catch (AblyException e) {
                    Log.v(TAG, "onAuthUpdated: closing transport after send failure");
                    this.transport.close();
                }
                break;
            }
            case connecting: {
                Log.v(TAG, "onAuthUpdated: closing connecting transport");
                ErrorInfo disconnectError = new ErrorInfo("Aborting incomplete connection with superseded auth params", 503, 80003);
                this.requestState(new StateIndication(ConnectionState.disconnected, disconnectError, null, null));
                this.connect();
                break;
            }
            default: {
                this.connect();
            }
        }
        this.singleThreadExecutor.execute(() -> {
            boolean waitingForConnected = true;
            block4: while (waitingForConnected) {
                ErrorInfo reason = waiter.waitForChange();
                ConnectionState connectionState = this.currentState.state;
                switch (connectionState) {
                    case connected: {
                        authUpdateResult.onUpdate(true, null);
                        Log.v(TAG, "onAuthUpdated: got connected");
                        waitingForConnected = false;
                        continue block4;
                    }
                    case connecting: 
                    case disconnected: {
                        Log.v(TAG, "onAuthUpdated: " + (Object)((Object)connectionState));
                        continue block4;
                    }
                }
                Log.v(TAG, "onAuthUpdated: throwing exception");
                authUpdateResult.onUpdate(false, reason);
                waitingForConnected = false;
            }
            waiter.close();
        });
    }

    public void onAuthError(ErrorInfo errorInfo) {
        Log.i(TAG, String.format(Locale.ROOT, "onAuthError: (%d) %s", errorInfo.code, errorInfo.message));
        if (errorInfo.statusCode == 403) {
            ConnectionStateListener.ConnectionStateChange failedStateChange = new ConnectionStateListener.ConnectionStateChange(this.connection.state, ConnectionState.failed, 0L, errorInfo);
            this.connection.onConnectionStateChange(failedStateChange);
            return;
        }
        switch (this.currentState.state) {
            case connecting: {
                ITransport transport = this.transport;
                if (transport == null) break;
                this.requestState(new StateIndication(ConnectionState.disconnected, errorInfo));
                break;
            }
            case connected: {
                this.addAction(new UpdateAction(errorInfo));
                break;
            }
        }
    }

    public void onMessage(ITransport transport, ProtocolMessage message) throws AblyException {
        if (transport != null && this.transport != transport) {
            return;
        }
        if (Log.level <= 2) {
            Log.v(TAG, "onMessage() (transport = " + transport + "): " + (Object)((Object)message.action) + ": " + new String(ProtocolSerializer.writeJSON(message)));
        }
        try {
            if (this.protocolListener != null) {
                this.protocolListener.onRawMessageRecv(message);
            }
            switch (message.action) {
                case heartbeat: {
                    this.onHeartbeat(message);
                    break;
                }
                case error: {
                    ErrorInfo reason = message.error;
                    if (reason == null) {
                        Log.e(TAG, "onMessage(): ERROR message received (no error detail)");
                    } else {
                        Log.e(TAG, "onMessage(): ERROR message received; message = " + reason.message + "; code = " + reason.code);
                    }
                    if (message.channel != null) {
                        this.onChannelMessage(message);
                        break;
                    }
                    this.onError(message);
                    break;
                }
                case connected: {
                    if (this.currentState.state == ConnectionState.closing) {
                        if (!this.trySendCloseProtocolMessage()) {
                            this.requestState(ConnectionState.closed);
                        }
                        break;
                    }
                    this.onConnected(message);
                    break;
                }
                case disconnect: 
                case disconnected: {
                    this.onDisconnected(message);
                    break;
                }
                case closed: {
                    this.onClosed(message);
                    break;
                }
                case ack: {
                    this.onAck(message);
                    break;
                }
                case nack: {
                    this.onNack(message);
                    break;
                }
                case auth: {
                    this.addAction(new ReauthAction());
                    break;
                }
                default: {
                    this.onChannelMessage(message);
                    break;
                }
            }
        }
        catch (Exception e) {
            throw AblyException.fromThrowable(e);
        }
    }

    private void onChannelMessage(ProtocolMessage message) {
        this.channels.onMessage(message);
        this.connection.recoveryKey = this.connection.createRecoveryKey();
    }

    private synchronized void onConnected(ProtocolMessage message) {
        this.ably.options.recover = null;
        this.connection.reason = message.error;
        if (this.connection.id != null) {
            Log.d(TAG, "There was a connection resume");
            if (message.connectionId.equals(this.connection.id)) {
                if (message.error == null) {
                    Log.d(TAG, "connection has reconnected and resumed successfully");
                } else {
                    Log.d(TAG, "connection resume success with non-fatal error: " + message.error.message);
                }
                this.addPendingMessagesToQueuedMessages(false);
            } else {
                if (message.error != null) {
                    Log.d(TAG, "connection resume failed with error: " + message.error.message);
                } else {
                    Log.d(TAG, "connection resume failed without error");
                }
                this.addPendingMessagesToQueuedMessages(true);
                this.channels.transferToChannelQueue(this.extractConnectionQueuePresenceMessages());
            }
        }
        this.connection.id = message.connectionId;
        ConnectionDetails connectionDetails = message.connectionDetails;
        this.connection.key = connectionDetails.connectionKey;
        this.maxIdleInterval = connectionDetails.maxIdleInterval;
        this.connectionStateTtl = connectionDetails.connectionStateTtl;
        String clientId = connectionDetails.clientId;
        try {
            this.ably.auth.setClientId(clientId);
        }
        catch (AblyException e) {
            this.requestState(this.transport, new StateIndication(ConnectionState.failed, e.errorInfo));
            return;
        }
        this.connection.recoveryKey = this.connection.createRecoveryKey();
        StateIndication stateIndication = new StateIndication(ConnectionState.connected, message.error, null, null);
        this.requestState(stateIndication);
    }

    private synchronized List<QueuedMessage> extractConnectionQueuePresenceMessages() {
        Iterator<QueuedMessage> queuedIterator = this.queuedMessages.iterator();
        ArrayList<QueuedMessage> queuedPresenceMessages = new ArrayList<QueuedMessage>();
        while (queuedIterator.hasNext()) {
            QueuedMessage queuedMessage = queuedIterator.next();
            if (queuedMessage.msg.presence == null) continue;
            queuedPresenceMessages.add(queuedMessage);
            queuedIterator.remove();
        }
        return queuedPresenceMessages;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addPendingMessagesToQueuedMessages(boolean resetMessageSerial) {
        ConnectionManager connectionManager = this;
        synchronized (connectionManager) {
            List<QueuedMessage> allPendingMessages = this.pendingMessages.popAll();
            if (resetMessageSerial) {
                this.msgSerial = 0L;
            } else if (!allPendingMessages.isEmpty()) {
                this.msgSerial = allPendingMessages.get((int)0).msg.msgSerial;
            }
            this.queuedMessages.addAll(0, allPendingMessages);
        }
    }

    public List<QueuedMessage> getPendingMessages() {
        return this.pendingMessages.queue;
    }

    private synchronized void onDisconnected(ProtocolMessage message) {
        ErrorInfo reason = message.error;
        if (reason != null && this.isTokenError(reason)) {
            this.ably.auth.onAuthError(reason);
        }
        this.requestState(new StateIndication(ConnectionState.disconnected, reason));
    }

    private synchronized void onClosed(ProtocolMessage message) {
        if (message.error != null) {
            this.onError(message);
        } else {
            this.connection.key = null;
            this.requestState(new StateIndication(ConnectionState.closed, null));
        }
    }

    private synchronized void onError(ProtocolMessage message) {
        this.connection.key = null;
        ErrorInfo reason = message.error;
        if (this.isTokenError(reason)) {
            this.ably.auth.onAuthError(reason);
        }
        ConnectionState destinationState = this.isFatalError(reason) ? ConnectionState.failed : ConnectionState.disconnected;
        this.requestState(this.transport, new StateIndication(destinationState, reason));
    }

    private void onAck(ProtocolMessage message) {
        this.pendingMessages.ack(message.msgSerial, message.count, message.error);
    }

    private void onNack(ProtocolMessage message) {
        this.pendingMessages.nack(message.msgSerial, message.count, message.error);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void onHeartbeat(ProtocolMessage message) {
        HashSet<Object> hashSet = this.heartbeatWaiters;
        synchronized (hashSet) {
            this.heartbeatWaiters.clear();
            this.heartbeatWaiters.notifyAll();
        }
    }

    private synchronized void startup() {
        if (this.handlerThread == null) {
            this.handlerThread = new Thread(new ActionHandler());
            this.handlerThread.start();
            this.startConnectivityListener();
        }
    }

    private boolean checkConnectionStale() {
        if (this.lastActivity == 0L) {
            return false;
        }
        long now = System.currentTimeMillis();
        long intervalSinceLastActivity = now - this.lastActivity;
        if (intervalSinceLastActivity > this.maxIdleInterval + this.connectionStateTtl) {
            if (this.connection.key != null) {
                Log.v(TAG, "Clearing stale connection key to suppress resume");
                this.connection.key = null;
                this.connection.recoveryKey = null;
            }
            return true;
        }
        return false;
    }

    private synchronized void setSuspendTime() {
        this.suspendTime = System.currentTimeMillis() + this.connectionStateTtl;
    }

    private StateIndication checkFallback(ErrorInfo reason) {
        String hostFallback;
        if (this.pendingConnect != null && (reason == null || reason.statusCode >= 500) && this.checkConnectivity() && (hostFallback = this.hosts.getFallback(this.pendingConnect.host)) != null) {
            Log.v(TAG, "checkFallback: fallback to " + hostFallback);
            return new StateIndication(ConnectionState.connecting, null, hostFallback, this.pendingConnect.host);
        }
        this.pendingConnect = null;
        return null;
    }

    private synchronized StateIndication checkSuspended(ErrorInfo reason) {
        long currentTime = System.currentTimeMillis();
        long timeToSuspend = this.suspendTime - currentTime;
        boolean suspendMode = timeToSuspend <= 0L;
        Log.v(TAG, "checkSuspended: timeToSuspend = " + timeToSuspend + "ms; suspendMode = " + suspendMode);
        ConnectionState expiredState = suspendMode ? ConnectionState.suspended : ConnectionState.disconnected;
        return new StateIndication(expiredState, reason);
    }

    private void tryWait(long timeout) {
        try {
            if (timeout == 0L) {
                this.wait();
            } else {
                this.wait(timeout);
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    private void handleReauth() {
        if (this.currentState.state == ConnectionState.connected) {
            Log.v(TAG, "Server initiated reauth");
            ErrorInfo errorInfo = null;
            try {
                this.ably.auth.renew();
            }
            catch (AblyException e) {
                errorInfo = e.errorInfo;
            }
            if (this.currentState.state == ConnectionState.connected) {
                this.connection.emitUpdate(errorInfo);
            }
        }
    }

    @Override
    public synchronized void onTransportAvailable(ITransport transport) {
        if (this.transport != transport) {
            Log.v(TAG, "onTransportAvailable: ignoring connection event from superseded transport");
            return;
        }
        if (this.protocolListener != null) {
            this.protocolListener.onRawConnect(transport.getURL());
        }
    }

    @Override
    public synchronized void onTransportUnavailable(ITransport transport, ErrorInfo reason) {
        Log.v(TAG, "onTransportUnavailable()");
        if (this.transport != transport) {
            Log.v(TAG, "onTransportUnavailable: ignoring disconnection event from superseded transport");
            return;
        }
        if (this.currentState.state == ConnectionState.connected) {
            this.setSuspendTime();
        }
        if (this.currentState.state == ConnectionState.closing) {
            this.requestState(ConnectionState.closed);
            return;
        }
        StateIndication fallbackAttempt = this.checkFallback(reason);
        if (fallbackAttempt != null) {
            this.requestState(fallbackAttempt);
            return;
        }
        StateIndication stateIndication = null;
        if (reason != null) {
            if (this.isFatalError(reason)) {
                Log.e(TAG, "onTransportUnavailable: unexpected transport error: " + reason.message);
                stateIndication = new StateIndication(ConnectionState.failed, reason);
            } else if (this.isTokenError(reason)) {
                this.ably.auth.onAuthError(reason);
            }
        }
        if (stateIndication == null) {
            stateIndication = this.checkSuspended(reason);
        }
        this.addAction(new SynchronousStateChangeAction(transport, stateIndication));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void connectImpl(StateIndication request) {
        ITransport oldTransport;
        ITransport transport;
        String host = request.fallback;
        if (host == null) {
            host = this.hosts.getPreferredHost();
        }
        this.checkConnectionStale();
        this.pendingConnect = new ConnectParams(this.ably.options, this.platformAgentProvider);
        this.pendingConnect.host = host;
        this.lastUsedHost = host;
        try {
            transport = this.transportFactory.getTransport(this.pendingConnect, this);
        }
        catch (Exception e) {
            String msg = "Unable to instance transport class";
            Log.e(this.getClass().getName(), msg, e);
            throw new RuntimeException(msg, e);
        }
        ConnectionManager connectionManager = this;
        synchronized (connectionManager) {
            oldTransport = this.transport;
            this.transport = transport;
        }
        if (oldTransport != null) {
            oldTransport.close();
        }
        transport.connect(this);
        if (this.protocolListener != null) {
            this.protocolListener.onRawConnectRequested(transport.getURL());
        }
    }

    private boolean closeImpl(boolean shouldAwaitConnection) {
        if (this.transport == null) {
            return true;
        }
        if (shouldAwaitConnection) {
            return false;
        }
        return !this.trySendCloseProtocolMessage();
    }

    private boolean trySendCloseProtocolMessage() {
        try {
            Log.v(TAG, "Requesting connection close");
            this.transport.send(new ProtocolMessage(ProtocolMessage.Action.close));
            return true;
        }
        catch (AblyException e) {
            Log.v(TAG, "Closing incomplete transport");
            this.clearTransport();
            return false;
        }
    }

    private void clearTransport() {
        if (this.transport != null) {
            this.transport.close();
            this.transport = null;
        }
    }

    protected boolean checkConnectivity() {
        try {
            return HttpHelpers.getUrlString(this.ably.httpCore, INTERNET_CHECK_URL).contains(INTERNET_CHECK_OK);
        }
        catch (AblyException e) {
            Log.d(TAG, "Exception whilst checking connectivity", e);
            return false;
        }
    }

    protected void setLastActivity(long lastActivityTime) {
        this.lastActivity = lastActivityTime;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void send(ProtocolMessage msg, boolean queueEvents, CompletionListener listener) throws AblyException {
        State state;
        ConnectionManager connectionManager = this;
        synchronized (connectionManager) {
            state = this.currentState;
            if (state.sendEvents) {
                this.sendImpl(msg, listener);
                return;
            }
            if (state.queueEvents && queueEvents) {
                this.queuedMessages.add(new QueuedMessage(msg, listener));
                return;
            }
        }
        throw AblyException.fromErrorInfo(state.defaultErrorInfo);
    }

    private void sendImpl(ProtocolMessage message, CompletionListener listener) throws AblyException {
        if (this.transport == null) {
            Log.v(TAG, "sendImpl(): Discarding message; transport unavailable");
            return;
        }
        if (ProtocolMessage.ackRequired(message)) {
            message.msgSerial = this.msgSerial++;
            this.pendingMessages.push(new QueuedMessage(message, listener));
        }
        if (this.protocolListener != null) {
            this.protocolListener.onRawMessageSend(message);
        }
        this.transport.send(message);
    }

    private void sendImpl(QueuedMessage msg) throws AblyException {
        if (this.transport == null) {
            Log.v(TAG, "sendImpl(): Discarding message; transport unavailable");
            return;
        }
        ProtocolMessage message = msg.msg;
        if (ProtocolMessage.ackRequired(message)) {
            message.msgSerial = this.msgSerial++;
            this.pendingMessages.push(msg);
        }
        if (this.protocolListener != null) {
            this.protocolListener.onRawMessageSend(message);
        }
        this.transport.send(message);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void sendQueuedMessages() {
        ConnectionManager connectionManager = this;
        synchronized (connectionManager) {
            while (this.queuedMessages.size() > 0) {
                try {
                    this.sendImpl(this.queuedMessages.get(0));
                }
                catch (AblyException e) {
                    Log.e(TAG, "sendQueuedMessages(): Unexpected error sending queued messages", e);
                }
                finally {
                    this.queuedMessages.remove(0);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void failQueuedMessages(ErrorInfo reason) {
        ConnectionManager connectionManager = this;
        synchronized (connectionManager) {
            for (QueuedMessage queued : this.queuedMessages) {
                if (queued.listener == null) continue;
                try {
                    queued.listener.onError(reason);
                }
                catch (Throwable t) {
                    Log.e(TAG, "failQueuedMessages(): Unexpected error calling listener", t);
                }
            }
            this.queuedMessages.clear();
            this.pendingMessages.fail(reason);
        }
    }

    private void startConnectivityListener() {
        this.connectivityListener = new CMConnectivityListener();
        this.ably.platform.getNetworkConnectivity().addListener(this.connectivityListener);
    }

    private void stopConnectivityListener() {
        this.ably.platform.getNetworkConnectivity().removeListener(this.connectivityListener);
        this.connectivityListener = null;
    }

    void disconnectAndSuppressRetries() {
        if (this.transport != null) {
            this.transport.close();
        }
        this.suppressRetry = true;
    }

    private boolean isTokenError(ErrorInfo err) {
        return err.code >= 40140 && err.code < 40150 || err.code == 80019 && err.statusCode == 401;
    }

    private boolean isFatalError(ErrorInfo err) {
        if (err.code != 0) {
            if (this.isTokenError(err)) {
                return false;
            }
            if (err.code >= 40000 && err.code < 50000) {
                return true;
            }
        }
        return err.statusCode != 0 && err.statusCode < 500;
    }

    private static class ActionQueue
    extends ArrayDeque<Action> {
        private ActionQueue() {
        }

        @Override
        public synchronized boolean add(Action action) {
            return super.add(action);
        }

        @Override
        public synchronized Action poll() {
            return (Action)super.poll();
        }

        @Override
        public synchronized Action peek() {
            return (Action)super.peek();
        }

        @Override
        public synchronized int size() {
            return super.size();
        }
    }

    public static class StateIndication {
        final ConnectionState state;
        final ErrorInfo reason;
        final String fallback;
        final String currentHost;

        StateIndication(ConnectionState state) {
            this(state, null);
        }

        public StateIndication(ConnectionState state, ErrorInfo reason) {
            this(state, reason, null, null);
        }

        StateIndication(ConnectionState state, ErrorInfo reason, String fallback, String currentHost) {
            this.state = state;
            this.reason = reason;
            this.fallback = fallback;
            this.currentHost = currentHost;
        }
    }

    public abstract class State {
        public final ConnectionState state;
        public final ErrorInfo defaultErrorInfo;
        public final boolean queueEvents;
        public final boolean sendEvents;
        final boolean terminal;
        public long timeout;

        State(ConnectionState state, boolean queueEvents, boolean sendEvents, boolean terminal, long timeout, ErrorInfo defaultErrorInfo) {
            this.state = state;
            this.queueEvents = queueEvents;
            this.sendEvents = sendEvents;
            this.terminal = terminal;
            this.timeout = timeout;
            this.defaultErrorInfo = defaultErrorInfo;
        }

        abstract StateIndication validateTransition(StateIndication var1);

        StateIndication onTimeout() {
            return null;
        }

        void enact(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change) {
            if (change != null) {
                if (this.sendEvents) {
                    ConnectionManager.this.sendQueuedMessages();
                } else if (!this.queueEvents) {
                    ConnectionManager.this.failQueuedMessages(stateIndication.reason);
                }
                for (Channel channel : ConnectionManager.this.channels.values()) {
                    this.enactForChannel(stateIndication, change, channel);
                }
            }
        }

        void enactForChannel(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change, Channel channel) {
        }
    }

    private class ConnectParams
    extends ITransport.TransportParams {
        ConnectParams(ClientOptions options, PlatformAgentProvider platformAgentProvider) {
            super(options, platformAgentProvider);
            this.connectionKey = ((ConnectionManager)ConnectionManager.this).connection.key;
            this.port = Defaults.getPort(options);
        }
    }

    private static interface Action
    extends Runnable {
    }

    public static interface Channels {
        public void onMessage(ProtocolMessage var1);

        public void suspendAll(ErrorInfo var1, boolean var2);

        public Iterable<Channel> values();

        public void transferToChannelQueue(List<QueuedMessage> var1);
    }

    private static class PendingMessageQueue {
        private final List<QueuedMessage> queue = new ArrayList<QueuedMessage>();

        private PendingMessageQueue() {
        }

        public synchronized void push(QueuedMessage msg) {
            this.queue.add(msg);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void ack(long msgSerial, int count, ErrorInfo reason) {
            QueuedMessage[] ackMessages = null;
            QueuedMessage[] nackMessages = null;
            PendingMessageQueue pendingMessageQueue = this;
            synchronized (pendingMessageQueue) {
                if (this.queue.isEmpty()) {
                    return;
                }
                long startSerial = this.queue.get((int)0).msg.msgSerial;
                if (msgSerial < startSerial) {
                    if ((count -= (int)(startSerial - msgSerial)) < 0) {
                        count = 0;
                    }
                    msgSerial = startSerial;
                }
                if (msgSerial > startSerial) {
                    int n = (int)(msgSerial - startSerial);
                    List<QueuedMessage> nackList = this.queue.subList(0, n);
                    nackMessages = nackList.toArray(new QueuedMessage[n]);
                    nackList.clear();
                    startSerial = msgSerial;
                }
                if (msgSerial == startSerial) {
                    List<QueuedMessage> list = this.queue.subList(0, count);
                    ackMessages = list.toArray(new QueuedMessage[count]);
                    list.clear();
                }
            }
            if (nackMessages != null) {
                if (reason == null) {
                    reason = new ErrorInfo("Unknown error", 500, 50000);
                }
                for (PendingMessageQueue pendingMessageQueue2 : nackMessages) {
                    try {
                        if (((QueuedMessage)((Object)pendingMessageQueue2)).listener == null) continue;
                        ((QueuedMessage)((Object)pendingMessageQueue2)).listener.onError(reason);
                    }
                    catch (Throwable t) {
                        Log.e(TAG, "ack(): listener exception", t);
                    }
                }
            }
            if (ackMessages != null) {
                for (PendingMessageQueue pendingMessageQueue3 : ackMessages) {
                    try {
                        if (((QueuedMessage)((Object)pendingMessageQueue3)).listener == null) continue;
                        ((QueuedMessage)((Object)pendingMessageQueue3)).listener.onSuccess();
                    }
                    catch (Throwable t) {
                        Log.e(TAG, "ack(): listener exception", t);
                    }
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized void nack(long serial, int count, ErrorInfo reason) {
            QueuedMessage[] nackMessages = null;
            QueuedMessage[] queuedMessageArray = this;
            synchronized (this) {
                if (this.queue.isEmpty()) {
                    // ** MonitorExit[var6_5] (shouldn't be in output)
                    return;
                }
                long startSerial = this.queue.get((int)0).msg.msgSerial;
                if (serial != startSerial) {
                    count -= (int)(startSerial - serial);
                    serial = startSerial;
                }
                List<QueuedMessage> nackList = this.queue.subList(0, count);
                nackMessages = nackList.toArray(new QueuedMessage[count]);
                nackList.clear();
                startSerial += (long)count;
                // ** MonitorExit[var6_5] (shouldn't be in output)
                if (nackMessages != null) {
                    if (reason == null) {
                        reason = new ErrorInfo("Unknown error", 500, 50000);
                    }
                    for (QueuedMessage msg : nackMessages) {
                        try {
                            if (msg.listener == null) continue;
                            msg.listener.onError(reason);
                        }
                        catch (Throwable t) {
                            Log.e(TAG, "nack(): listener exception", t);
                        }
                    }
                }
                return;
            }
        }

        synchronized List<QueuedMessage> popAll() {
            ArrayList<QueuedMessage> allPendingMessages = new ArrayList<QueuedMessage>(this.queue);
            this.queue.clear();
            return allPendingMessages;
        }

        synchronized void fail(ErrorInfo reason) {
            for (QueuedMessage queuedMessage : this.queue) {
                if (queuedMessage.listener == null) continue;
                queuedMessage.listener.onError(reason);
            }
            this.queue.clear();
        }
    }

    class Initialized
    extends State {
        Initialized() {
            super(ConnectionState.initialized, true, false, false, 0L, null);
        }

        @Override
        StateIndication validateTransition(StateIndication target) {
            if (target.state == this.state) {
                return null;
            }
            return target;
        }
    }

    class Connecting
    extends State {
        Connecting() {
            super(ConnectionState.connecting, true, false, false, Defaults.TIMEOUT_CONNECT, null);
        }

        @Override
        StateIndication onTimeout() {
            return ConnectionManager.this.checkSuspended(null);
        }

        @Override
        StateIndication validateTransition(StateIndication target) {
            return target;
        }

        @Override
        void enact(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change) {
            super.enact(stateIndication, change);
            ConnectionManager.this.connectImpl(stateIndication);
        }
    }

    class Connected
    extends State {
        Connected() {
            super(ConnectionState.connected, false, true, false, 0L, null);
        }

        @Override
        StateIndication validateTransition(StateIndication target) {
            if (target.state == this.state) {
                ConnectionManager.this.addAction(new UpdateAction(null));
                return null;
            }
            return target;
        }

        @Override
        void enactForChannel(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change, Channel channel) {
            channel.setConnected();
        }

        @Override
        void enact(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change) {
            super.enact(stateIndication, change);
            ConnectionManager.this.pendingConnect = null;
        }
    }

    class Disconnected
    extends State {
        Disconnected() {
            super(ConnectionState.disconnected, true, false, false, ConnectionManager.this.ably.options.disconnectedRetryTimeout, REASON_DISCONNECTED);
        }

        @Override
        StateIndication validateTransition(StateIndication target) {
            if (target.state == this.state) {
                return null;
            }
            if (target.state == ConnectionState.closing) {
                return new StateIndication(ConnectionState.closed);
            }
            return target;
        }

        @Override
        StateIndication onTimeout() {
            return new StateIndication(ConnectionState.connecting);
        }

        @Override
        void enactForChannel(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change, Channel channel) {
        }

        @Override
        void enact(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change) {
            super.enact(stateIndication, change);
            ConnectionManager.this.clearTransport();
            if (change.previous == ConnectionState.connected) {
                ConnectionManager.this.setSuspendTime();
                if (!ConnectionManager.this.suppressRetry) {
                    Log.v(TAG, "Was previously connected, retrying immediately");
                    ConnectionManager.this.requestState(ConnectionState.connecting);
                }
            }
        }
    }

    class Suspended
    extends State {
        Suspended() {
            super(ConnectionState.suspended, false, false, false, ConnectionManager.this.ably.options.suspendedRetryTimeout, REASON_SUSPENDED);
        }

        @Override
        StateIndication validateTransition(StateIndication target) {
            if (target.state == this.state) {
                return null;
            }
            if (target.state == ConnectionState.closing) {
                return new StateIndication(ConnectionState.closed);
            }
            return target;
        }

        @Override
        StateIndication onTimeout() {
            return new StateIndication(ConnectionState.connecting);
        }

        @Override
        void enactForChannel(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change, Channel channel) {
            channel.setSuspended(this.defaultErrorInfo, true);
        }
    }

    class Closing
    extends State {
        Closing() {
            super(ConnectionState.closing, false, false, false, Defaults.TIMEOUT_CONNECT, REASON_CLOSED);
        }

        @Override
        StateIndication validateTransition(StateIndication target) {
            if (target.state == this.state) {
                return null;
            }
            if (target.state == ConnectionState.disconnected || target.state == ConnectionState.suspended) {
                return new StateIndication(ConnectionState.closed);
            }
            return target;
        }

        @Override
        StateIndication onTimeout() {
            return new StateIndication(ConnectionState.closed);
        }

        @Override
        void enact(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change) {
            super.enact(stateIndication, change);
            boolean shouldAwaitConnection = change.previous == ConnectionState.connecting;
            boolean closed = ConnectionManager.this.closeImpl(shouldAwaitConnection);
            if (closed) {
                ConnectionManager.this.addAction(new AsynchronousStateChangeAction(ConnectionState.closed));
            }
        }
    }

    class Closed
    extends State {
        Closed() {
            super(ConnectionState.closed, false, false, true, 0L, REASON_CLOSED);
        }

        @Override
        StateIndication validateTransition(StateIndication target) {
            if (target.state == ConnectionState.connecting) {
                return target;
            }
            return null;
        }

        @Override
        void enactForChannel(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change, Channel channel) {
            channel.setConnectionClosed(REASON_CLOSED);
        }
    }

    class Failed
    extends State {
        Failed() {
            super(ConnectionState.failed, false, false, true, 0L, REASON_FAILED);
        }

        @Override
        StateIndication validateTransition(StateIndication target) {
            if (target.state == ConnectionState.connecting) {
                return target;
            }
            return null;
        }

        @Override
        void enactForChannel(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change, Channel channel) {
            channel.setConnectionFailed(stateIndication.reason);
        }

        @Override
        void enact(StateIndication stateIndication, ConnectionStateListener.ConnectionStateChange change) {
            super.enact(stateIndication, change);
            ConnectionManager.this.clearTransport();
        }
    }

    private class AsynchronousStateChangeAction
    extends StateChangeAction
    implements Action {
        AsynchronousStateChangeAction(ConnectionState state) {
            super(null, new StateIndication(state, null));
        }

        AsynchronousStateChangeAction(ITransport transport, StateIndication stateIndication) {
            super(transport, stateIndication);
        }

        @Override
        public void run() {
            this.setState();
            this.enactState();
        }
    }

    private class HeartbeatWaiter
    extends Thread {
        private final CompletionListener listener;

        HeartbeatWaiter(CompletionListener listener) {
            this.listener = listener;
        }

        private void onSuccess() {
            this.clear();
            if (this.listener != null) {
                this.listener.onSuccess();
            }
        }

        private void onError(ErrorInfo reason) {
            this.clear();
            if (this.listener != null) {
                this.listener.onError(reason);
            }
        }

        private boolean clear() {
            boolean pending = ConnectionManager.this.heartbeatWaiters.remove(this);
            if (pending) {
                this.interrupt();
            }
            return pending;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            boolean pending;
            HashSet hashSet = ConnectionManager.this.heartbeatWaiters;
            synchronized (hashSet) {
                try {
                    ConnectionManager.this.heartbeatWaiters.wait(5000L);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                pending = this.clear();
            }
            if (pending) {
                this.onError(new ErrorInfo("Timed out waiting for heartbeat response", 50000, 500));
            } else {
                this.onSuccess();
            }
        }
    }

    private class ConnectionWaiter
    implements ConnectionStateListener {
        private ConnectionStateListener.ConnectionStateChange change;
        private boolean closed = false;

        private ConnectionWaiter() {
            ConnectionManager.this.connection.on(this);
        }

        private synchronized ErrorInfo waitForChange() {
            if (this.closed) {
                throw new IllegalStateException("Already closed.");
            }
            Log.d(TAG, "ConnectionWaiter.waitFor()");
            if (this.change == null) {
                try {
                    this.wait();
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
            Log.d(TAG, "ConnectionWaiter.waitFor done: currentState=" + ConnectionManager.this.currentState + ")");
            ErrorInfo reason = this.change.reason;
            this.change = null;
            return reason;
        }

        @Override
        public synchronized void onConnectionStateChanged(ConnectionStateListener.ConnectionStateChange state) {
            this.change = state;
            this.notify();
        }

        private void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            ConnectionManager.this.connection.off(this);
        }
    }

    private class UpdateAction
    implements Action {
        private final ErrorInfo reason;

        UpdateAction(ErrorInfo reason) {
            this.reason = reason;
        }

        @Override
        public void run() {
            ConnectionManager.this.connection.emitUpdate(this.reason);
        }
    }

    private class ReauthAction
    implements Action {
        private ReauthAction() {
        }

        @Override
        public void run() {
            ConnectionManager.this.handleReauth();
        }
    }

    public static class QueuedMessage {
        public final ProtocolMessage msg;
        public final CompletionListener listener;

        public QueuedMessage(ProtocolMessage msg, CompletionListener listener) {
            this.msg = msg;
            this.listener = listener;
        }
    }

    class ActionHandler
    implements Runnable {
        ActionHandler() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            block5: while (true) {
                ConnectionManager connectionManager = ConnectionManager.this;
                synchronized (connectionManager) {
                    while (ConnectionManager.this.actionQueue.size() == 0) {
                        StateIndication nextState;
                        if (((ConnectionManager)ConnectionManager.this).currentState.terminal) {
                            ConnectionManager.this.handlerThread = null;
                            ConnectionManager.this.stopConnectivityListener();
                            return;
                        }
                        ConnectionManager.this.tryWait(((ConnectionManager)ConnectionManager.this).currentState.timeout);
                        Action act = ConnectionManager.this.actionQueue.peek();
                        if (act != null) {
                            Log.d(TAG, "Wait ended by action: " + act.toString());
                            break;
                        }
                        if (ConnectionManager.this.suppressRetry || (nextState = ConnectionManager.this.currentState.onTimeout()) == null) continue;
                        ConnectionManager.this.requestState(nextState);
                    }
                }
                while (true) {
                    Action deferredAction;
                    if ((deferredAction = ConnectionManager.this.actionQueue.poll()) == null) continue block5;
                    try {
                        deferredAction.run();
                        continue;
                    }
                    catch (Exception e) {
                        Log.e(TAG, "Action invocation failed with exception: action = " + deferredAction.toString(), e);
                        continue;
                    }
                    break;
                }
                break;
            }
        }
    }

    private class SynchronousStateChangeAction
    extends StateChangeAction
    implements Action {
        SynchronousStateChangeAction(ITransport transport, StateIndication stateIndication) {
            super(transport, stateIndication);
            this.setState();
        }

        @Override
        public void run() {
            this.enactState();
        }
    }

    private class CMConnectivityListener
    implements NetworkConnectivity.NetworkConnectivityListener {
        private CMConnectivityListener() {
        }

        @Override
        public void onNetworkAvailable() {
            ConnectionManager cm = ConnectionManager.this;
            ConnectionState currentState = cm.getConnectionState().state;
            Log.i(TAG, "onNetworkAvailable(): currentState = " + currentState.name());
            if (currentState == ConnectionState.disconnected || currentState == ConnectionState.suspended) {
                Log.i(TAG, "onNetworkAvailable(): initiating reconnect");
                cm.connect();
            }
        }

        @Override
        public void onNetworkUnavailable(ErrorInfo reason) {
            ConnectionManager cm = ConnectionManager.this;
            ConnectionState currentState = cm.getConnectionState().state;
            Log.i(TAG, "onNetworkUnavailable(); currentState = " + currentState.name() + "; reason = " + reason.toString());
            if (currentState == ConnectionState.connected || currentState == ConnectionState.connecting) {
                Log.i(TAG, "onNetworkUnavailable(): closing connected transport");
                cm.requestState(new StateIndication(ConnectionState.disconnected, reason));
            }
        }
    }

    private abstract class StateChangeAction {
        protected final ITransport transport;
        protected final StateIndication stateIndication;
        protected ConnectionStateListener.ConnectionStateChange change;

        StateChangeAction(ITransport transport, StateIndication stateIndication) {
            this.transport = transport;
            this.stateIndication = stateIndication;
        }

        protected void setState() {
            this.change = ConnectionManager.this.setState(this.transport, this.stateIndication);
        }

        protected void enactState() {
            if (this.change != null) {
                if (this.change.current != this.change.previous) {
                    ConnectionManager.this.connection.onConnectionStateChange(this.change);
                }
                ((State)ConnectionManager.this.states.get((Object)this.stateIndication.state)).enact(this.stateIndication, this.change);
                if (((ConnectionManager)ConnectionManager.this).currentState.terminal) {
                    ConnectionManager.this.clearTransport();
                }
            }
        }
    }
}

