/*
 * Decompiled with CFR 0.152.
 */
package info.setmy.models.storage;

import info.setmy.exceptions.ForbiddenException;
import info.setmy.exceptions.UncheckedException;
import info.setmy.models.storage.DirectoryStructurePattern;
import info.setmy.models.storage.StorageFile;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Optional;
import java.util.UUID;
import org.apache.commons.lang3.StringUtils;

public class Storage {
    private final String directoryLocation;
    private File storageDirectory;

    public Storage(String directoryLocation) {
        this.directoryLocation = StringUtils.trim((String)directoryLocation);
    }

    public void init() {
        if (this.directoryLocation == null) {
            throw new UncheckedException("Storage directory is not set");
        }
        this.validateFileName(this.directoryLocation);
        File file = new File(this.directoryLocation);
        if (!file.isDirectory()) {
            throw new UncheckedException(String.format("Storage location '%s' is not directory", this.directoryLocation));
        }
        this.storageDirectory = file;
    }

    public Optional<StorageFile> createStorageFile() {
        return this.createStorageFile(new DirectoryStructurePattern().setDefault());
    }

    public Optional<StorageFile> createStorageFile(DirectoryStructurePattern pattern) {
        String fileName;
        String directoryName = StringUtils.trim((String)pattern.toString());
        if (pattern.getName() == null) {
            UUID uuid = UUID.randomUUID();
            fileName = uuid.toString();
        } else {
            fileName = pattern.getName();
        }
        this.validateFileName(fileName);
        Optional<File> directory = this.newDirectory(directoryName);
        if (directory.isPresent()) {
            File directoryFile = directory.get();
            File file = new File(directoryFile, fileName);
            try {
                if (file.createNewFile()) {
                    return Optional.of(new StorageFile(directoryFile, file, directoryName, fileName));
                }
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    public File[] listStorageFiles(String directoryPath) {
        Optional<StorageFile> dir = this.getStorageDirectory(directoryPath);
        if (dir.isPresent()) {
            return dir.get().getParent().listFiles();
        }
        return new File[0];
    }

    public Optional<FileInputStream> getStorageFileStream(String directoryPath) {
        try {
            return Optional.of(new FileInputStream(this.getStorageFile(directoryPath).get().getChild()));
        }
        catch (FileNotFoundException ex) {
            return Optional.empty();
        }
    }

    public Optional<StorageFile> getStorageFile(String fileNamePath) {
        File file = this.getValidatedFileObject(StringUtils.trim((String)fileNamePath));
        if (file.exists() && file.isFile()) {
            return Optional.of(new StorageFile(null, file, null, fileNamePath));
        }
        return Optional.empty();
    }

    public Optional<StorageFile> getStorageDirectory(String directoryPath) {
        File directory = this.getValidatedFileObject(StringUtils.trim((String)directoryPath));
        if (directory.exists() && directory.isDirectory()) {
            return Optional.of(new StorageFile(directory, null, directoryPath, null));
        }
        return Optional.empty();
    }

    private Optional<File> newDirectory(String directoryName) {
        this.validateFileName(directoryName);
        String dirFullPath = this.getPath(directoryName);
        File directory = new File(dirFullPath);
        if (directory.exists()) {
            return Optional.of(directory);
        }
        boolean result = directory.mkdirs();
        if (result) {
            return Optional.of(directory);
        }
        return Optional.empty();
    }

    private File getValidatedFileObject(String directoryPath) {
        this.validateUperAndHomeChange(directoryPath);
        File file = new File(this.storageDirectory, directoryPath);
        return file;
    }

    private void validateFileName(String fileName) {
        this.validateRootChange(fileName);
        this.validateUperAndHomeChange(fileName);
    }

    private void validateUperAndHomeChange(String fileName) throws ForbiddenException {
        this.validateUperChange(fileName);
        this.validateHomeChange(fileName);
    }

    private void validateRootChange(String fileName) throws ForbiddenException {
        if (fileName.indexOf("/") == 0) {
            throw new ForbiddenException(String.format("File or directory '%s' is not allowed", fileName));
        }
    }

    private void validateUperChange(String fileName) throws ForbiddenException {
        if (-1 != fileName.indexOf("..")) {
            throw new ForbiddenException(String.format("File or directory '%s' is not allowed", fileName));
        }
    }

    private void validateHomeChange(String fileName) throws ForbiddenException {
        if (-1 != fileName.indexOf("~")) {
            throw new ForbiddenException(String.format("File or directory '%s' is not allowed", fileName));
        }
    }

    private String getPath(String directoryPath) {
        return this.directoryLocation + "/" + directoryPath;
    }
}

