//    Copyright (c) 2014 - 2015 payu@india.com
//
//    Permission is hereby granted, free of charge, to any person obtaining a copy
//    of this software and associated documentation files (the "Software"), to deal
//    in the Software without restriction, including without limitation the rights
//    to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//    copies of the Software, and to permit persons to whom the Software is
//    furnished to do so, subject to the following conditions:
//
//    The above copyright notice and this permission notice shall be included in
//    all copies or substantial portions of the Software.
//
//    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
//    THE SOFTWARE.

package com.payu.paymentparamhelper;

import static com.payu.paymentparamhelper.PayuErrors.TRANSACTION_ID_LENGTH_LIMIT;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

/**
 * Created by franklin on 6/18/15.
 * Payu util class provides protected, public functions to sdk and App.
 * Use the {@link PayuUtils#SBI_MAES_BIN } set to find whether the number is state bank maestro number or not
 * Use the {@link PayuUtils#validateCardNumber(String)} method to validate user card number, it includes length luhn validation.
 * Use the {@link PayuUtils#luhn(String)} method to validate card number, use this function only if you validate length at your end.
 * Use the {@link PayuUtils#getIssuer(String)} method to get the issuer, it will return the issuer from {VISA, AMEX, MAES, MAST, SMAE, LASER, DINR, JCB}
 * use the {@link PayuUtils#validateCvv(String, String)} method to validate cvv, AMEX has 4 digit, SMAE has no cvv, other has 3 dight cvv.
 * use the {@link PayuUtils#validateExpiry(int, int)} method to validate whether the card is expired or not!.
 * use the {@link PayuUtils#concatParams(String, String, Boolean)} method to concat both params1 and param2 using equal to (=) and add a ampersand at the end!
 * adding default arguments{@link PayuUtils#getReturnData(String)} {@link PayuUtils#getReturnData(int, String)}
 * use the {@link PayuUtils#getReturnData(int, String, String)} to send the data back to app as {@link PostData}
 * use the {@link PayuUtils#trimAmpersand(String)} remove leading ampersand if any!.
 */
@SuppressWarnings("JavadocReference")
public class PayuUtils {


    /**
     * Known Sate bank maestro bins
     * helps us to validate cvv expmon, expyr,
     * SMAE cards do not have CVV, Expiry month, Expiry year.
     */
     static Set<String> SBI_MAES_BIN;
     static String dateFormat = "yyyy-MM-dd";

    static {
        SBI_MAES_BIN = new HashSet<>();
        SBI_MAES_BIN.add("504435");
        SBI_MAES_BIN.add("504645");
        SBI_MAES_BIN.add("504775");
        SBI_MAES_BIN.add("504809");
        SBI_MAES_BIN.add("504993");
        SBI_MAES_BIN.add("600206");
        SBI_MAES_BIN.add("603845");
        SBI_MAES_BIN.add("622018");
        SBI_MAES_BIN.add("504774");
    }

    public static String valueOf(Object o) {
      String value=String.valueOf(o);
      if(value.equals("null")){
        return null;
      }
      return value;
    }

    /**
     * includes length validation also
     * use {@link PayuUtils#luhn(String)} to validate.
     *
     * @param cardNumber any card number
     * @return true if valid else false
     */
     Boolean validateCardNumber(String cardNumber) {
        if (cardNumber.length() < 12) {
            return false;
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.RUPAY) && cardNumber.length() == 16){
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.VISA) && cardNumber.length() == 16) {
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.MAST) && cardNumber.length() == 16) {
            return luhn(cardNumber);
        } else if ((getIssuer(cardNumber).contentEquals(PayuConstants.MAES) || getIssuer(cardNumber).contentEquals(PayuConstants.SMAE)) && cardNumber.length() >= 12 && cardNumber.length() <= 19) {
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.DINR) && cardNumber.length() == 14) {
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.AMEX) && cardNumber.length() == 15) {
            return luhn(cardNumber);
        } else if (getIssuer(cardNumber).contentEquals(PayuConstants.JCB) && cardNumber.length() == 16) {
            return luhn(cardNumber);
        }else if (getIssuer(cardNumber).contentEquals(PayuConstants.SODEXO) && cardNumber.length() == 16) {
            return luhn(cardNumber);
        }
        else {
            return luhn(cardNumber); //If no issuer is found return true if card number passes luhn algorithm.
        }
    }

    /**
     * Universal luhn validation for Credit, Debit cards.
     *
     * @param cardNumber any card number
     * @return true if valid else false
     */
     Boolean luhn(String cardNumber) {
        int sum = 0;
        boolean alternate = false;
        for (int i = cardNumber.length() - 1; i >= 0; i--) {
            int n = Integer.parseInt(cardNumber.substring(i, i + 1));
            if (alternate) {
                n *= 2;
                if (n > 9) {
                    n = (n % 10) + 1;
                }
            }
            sum += n;
            alternate = !alternate;
        }
        if (sum % 10 == 0) {
            return true;
        }
        return false;
    }


    /**
     * Gives the Issuer of the card number.
     * using simple regex we can actually figure out the card belong to which issuer amoung VISA, AMEX, MAES, MAST, SMAE, LASER, DINR, JCB
     *
     * @param mCardNumber first 6 digit of card number is more than enough to figure out. (Card Bin)
     * @return Issuer of the card
     */
     String getIssuer(String mCardNumber) {
        if (mCardNumber.startsWith("4")) {
            return PayuConstants.VISA;
        }else if (mCardNumber.matches("^508[5-9][0-9][0-9]|60698[5-9]|60699[0-9]|607[0-8][0-9][0-9]|6079[0-7][0-9]|60798[0-4]|(?!608000)608[0-4][0-9][0-9]|608500|6521[5-9][0-9]|652[2-9][0-9][0-9]|6530[0-9][0-9]|6531[0-4][0-9]")){
            return PayuConstants.RUPAY;
        } else if (mCardNumber.matches("^((6304)|(6706)|(6771)|(6709))[\\d]+")) {
            return PayuConstants.LASER;
        } else if (mCardNumber.matches("6(?:011|5[0-9]{2})[0-9]{12}[\\d]+")) {
            return PayuConstants.LASER;
        } else if (mCardNumber.matches("(5[06-8]|6\\d)\\d{14}(\\d{2,3})?[\\d]+") || mCardNumber.matches("(5[06-8]|6\\d)[\\d]+") || mCardNumber.matches("((504([435|645|774|775|809|993]))|(60([0206]|[3845]))|(622[018])\\d)[\\d]+")) {
            if (mCardNumber.length() >= 6) { // wel we have 6 digit bin
                if (SBI_MAES_BIN.contains(mCardNumber.substring(0, 6))) {
                    return PayuConstants.SMAE;
                }
            }
            return PayuConstants.MAES;
        } else if (mCardNumber.matches("^5[1-5][\\d]+") || mCardNumber.matches("^(222[1-9][0-9]{2}|22[3-9][0-9]{3}|2[3-6][0-9]{4}|27[0-1][0-9]{3}|2720[0-9]{2})[\\d]*$")) {
            return PayuConstants.MAST;
        } else if (mCardNumber.matches("^3[47][\\d]+")) {
            return PayuConstants.AMEX;
        } else if (mCardNumber.startsWith("36") || mCardNumber.matches("^30[0-5][\\d]+") || mCardNumber.matches("2(014|149)[\\d]+")) {
            return PayuConstants.DINR;
        } else if (mCardNumber.matches("^35(2[89]|[3-8][0-9])[\\d]+")) {
            return PayuConstants.JCB;
        }else if (mCardNumber.matches("^35(2[89]|[3-8][0-9])[\\d]+")) {
            return PayuConstants.JCB;
        }
        return PayuConstants.UNDEFINED;
    }

    /**
     * helps to validate cvv
     * we need card bin to figure out issuer,
     * from issuer we validate cvv
     * Oh! AMEX has 4 digit,
     * Crap! SMAE has no cvv,
     * okay, other has 3 dight cvv.
     *
     * @param cardNumber Card bin
     * @param cvv        cvv
     * @return true if valid cvv else false
     */
     boolean validateCvv(String cardNumber, String cvv) {
        String issuer = getIssuer(cardNumber);
        if (issuer.contentEquals(PayuConstants.SMAE)||issuer.contentEquals(PayuConstants.UNDEFINED)) {
            return true;
        }else if (issuer.contentEquals("")){
            return false;
        }
        else if (issuer.contentEquals(PayuConstants.AMEX) & cvv.length() == 4) {
            return true;
        } else if (!issuer.contentEquals(PayuConstants.AMEX) && cvv.length() == 3) {
            return true;
        }
        return false;
    }

    /**
     * helps to validate whether the card is expired or not!.
     *
     * @param expiryMonth expmon
     * @param expiryYear  expyr
     * @return true if valid else false
     */
     boolean validateExpiry(int expiryMonth, int expiryYear) {
        Calendar calendar = Calendar.getInstance();
        if (expiryMonth < 1 || expiryMonth > 12 || String.valueOf(expiryYear).length() != 4) { // expiry month validation
            return false;
        } else if (calendar.get(Calendar.YEAR) > expiryYear || (calendar.get(Calendar.YEAR) == expiryYear && calendar.get(Calendar.MONTH) + 1 > expiryMonth)) { // expiry date validation.
            return false;
        }
        return true;
    }

    /**
     * just to make our life easier, lets define a function
     * concat two strings with equal sign and add an ampersand at the end
     *
     * @param key   example txnid
     * @param value example payu12345
     * @param shouldEncode
     * @return concatenated String
     */

    public String concatParams(String key, String value, boolean shouldEncode) {
        if (shouldEncode)
            value = encodeValue(value);

        StringBuilder sb = new StringBuilder(key);
        sb.append( '=' );
        sb.append( value );
        sb.append( '&' );
        return sb.toString();
    }

    public static String appendKeyValueToStingBuilder(String key, Object value) {
        return "\"" + key + "\":\"" + value + "\",";
    }

    /**
     * Defalut param of {@link PayuUtils#getReturnData(int, String, String)}
     * Use this if your first param is {@link PayuErrors#MISSING_PARAMETER_EXCEPTION} and second param is {@link PayuConstants#ERROR}
     *
     * @param result result it the result of your PostData
     * @return PostData object.
     */
     PostData getReturnData(String result) {
        return getReturnData(PayuErrors.MISSING_PARAMETER_EXCEPTION, PayuConstants.ERROR, result);
    }

    /**
     * Defalut param of {@link PayuUtils#getReturnData(int, String, String)}
     * Use this param if there is something wrong while validating params {@link PayuConstants#ERROR}
     *
     * @param code   error code to be returned with PostData
     * @param result error message (result) to be returned with PostData
     * @return PostData object.
     */
     PostData getReturnData(int code, String result) {
        return getReturnData(code, PayuConstants.ERROR, result);
    }

    /**
     * Use this is method to return the PostData to the user {@link PostData#code} {@link PostData#status} {@link PostData#result}
     *
     * @param code    error code to be returned with PostData
     * @param status, status message it can either be {@link PayuConstants#SUCCESS} or {@link PayuConstants#ERROR}
     * @param result  error message (result) to be returned with PostData
     * @return PostData object.
     */
     PostData getReturnData(int code, String status, String result) {
        PostData postData = new PostData();
        postData.setCode(code);
        postData.setStatus(status);
        postData.setResult(result);
        return postData;
    }

    /**
     * This method is used to trim the leading Ampersand sign
     *
     * @param data , should be concatenated data with equal sign and ampersand.
     * @return example txnid=payu12345
     */
     String trimAmpersand(String data) {
        return data.charAt(data.length() - 1) == '&' ? data.substring(0, data.length() - 1) : data;
    }


    /**
     * Return Analytics JSON String so as to append it to postdata
     * @return JSONified String
     */
     static String getAnalyticsString(String analyticsData){
        JSONArray jsonArray=new JSONArray();
        try {
            if (analyticsData == null) {
                jsonArray = new JSONArray();
            } else {
                jsonArray = new JSONArray(analyticsData);
            }
            JSONObject object = new JSONObject();
            object.put(PayuConstants.PLATFORM_KEY, PayuConstants.PLATFORM_VALUE);
            object.put(PayuConstants.NAME_KEY, PayuConstants.NAME_VALUE);
            object.put(PayuConstants.VERSION_KEY, BuildConfig.VERSION_NAME);
            jsonArray.put(object);
        }catch (JSONException e){

        }
        return jsonArray.toString();
    }

    //format YYYY-MM-DD
    static boolean validateCardStartDate(String date){
        DateFormat format = new SimpleDateFormat(dateFormat);
        try {
            Date parsedDate = format.parse(date);
            return  getDatePart(parsedDate)>=getDatePart(new Date());
        } catch (ParseException e) {

            return false;
        }
    }

    private static long getDatePart(Date date) {
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);

        return cal.getTimeInMillis();
    }

   public static boolean validateCardEndDate(String date) {
       DateFormat format = new SimpleDateFormat(dateFormat);
       try {
           format.parse(date);
           return true;
       } catch (ParseException e) {

           return false;
       }
   }

   public static boolean compareDates(String startDate, String endDate) {
       DateFormat format = new SimpleDateFormat(dateFormat);
       try {
           Date dStartDate = format.parse(startDate);
           Date dEndDate =format.parse(endDate);
           return dEndDate.getTime() > dStartDate.getTime();
       } catch (ParseException e) {

           return false;
       }
   }

    private String encodeValue(String value) {
        String encodedvalue = value;
        try {
            if (value != null)
                encodedvalue = URLEncoder.encode(value, "UTF-8");
        } catch (UnsupportedEncodingException ignored) {
        }
        return encodedvalue;
    }

    public static Boolean supportedBankCodesForDCEmi(String bankCode) {
        SupportedBankCodesForEmi.SupportedBankCodesDCEmi supportedBankCodes[] = SupportedBankCodesForEmi.SupportedBankCodesDCEmi.values();
        for (SupportedBankCodesForEmi.SupportedBankCodesDCEmi bankCodes : supportedBankCodes) {
            if (bankCode.startsWith(bankCodes.toString())) {
                return true;
            }
        }
        return false;
    }


    public static int parseInt(String s){
        try {
          if(s==null)
            return 0;
          return Integer.parseInt(s);
        }catch (NumberFormatException e){
            return 0;
        }
    }

    public static Boolean supportedBankCodesForCardlessEmi(String bankCode) {
        SupportedBankCodesForEmi.SupportedBankCodesCardlessEmi supportedBankCodes[] = SupportedBankCodesForEmi.SupportedBankCodesCardlessEmi.values();
        for (SupportedBankCodesForEmi.SupportedBankCodesCardlessEmi bankCodes : supportedBankCodes) {
            if (bankCode.startsWith(bankCodes.toString())) {
                return true;
            }
        }
        return false;
    }

    /**
     * function to getJsonArrayString for wealthTech tpv txn
     */
    public static String getProductDetailsJsonArray(PaymentParams mPaymentParams) {
        StringBuilder jsonBuilder = new StringBuilder();
        jsonBuilder.append("[");
        List<Products> productList = mPaymentParams.getProductsList();
        for (int i = 0; i < productList.size(); i++) {
            Products products = productList.get(i);
            jsonBuilder.append("{");

            jsonBuilder.append("\"").append(PayuConstants.PAYU_TYPE).append("\":\"").append(products.getType()).append("\",");
            if (products.getPlan() != null && !products.getPlan().isBlank())
                jsonBuilder.append("\"").append(PayuConstants.PAYU_PLAN).append("\":\"").append(products.getPlan()).append("\",");
            if (products.getFolio() != null && !products.getFolio().isBlank())
                jsonBuilder.append("\"").append(PayuConstants.PAYU_FOLIO).append("\":\"").append(products.getFolio()).append("\",");
            jsonBuilder.append("\"").append(PayuConstants.AMOUNT).append("\":\"").append(products.getAmount()).append("\",");
            if (products.getOption() != null && !products.getOption().isBlank())
                jsonBuilder.append("\"").append(PayuConstants.PAYU_OPTION).append("\":\"").append(products.getOption()).append("\",");
            if (products.getScheme() != null && !products.getScheme().isBlank())
                jsonBuilder.append("\"").append(PayuConstants.PAYU_SCHEME).append("\":\"").append(products.getScheme()).append("\",");
            jsonBuilder.append("\"").append(PayuConstants.PAYU_RECEIPT).append("\":\"").append(products.getReceipt()).append("\",");
            jsonBuilder.append("\"").append(PayuConstants.PAYU_MF_MEMBER_ID).append("\":\"").append(products.getMfMemberId()).append("\",");
            jsonBuilder.append("\"").append(PayuConstants.PAYU_MF_USER_ID).append("\":\"").append(products.getMfUserId()).append("\",");
            jsonBuilder.append("\"").append(PayuConstants.PAYU_MF_PARTNER).append("\":\"").append(products.getMfPartner()).append("\",");
            jsonBuilder.append("\"").append(PayuConstants.PAYU_MF_INVESTMENT_TYPE).append("\":\"").append(products.getMfInvestmentType()).append("\",");
            if (products.getMfAmcCode() != null && !products.getMfAmcCode().isBlank())
                jsonBuilder.append("\"").append(PayuConstants.PAYU_MF_AMC_CODE).append("\":\"").append(products.getMfAmcCode()).append("\"");

            jsonBuilder.append("}");

            if (i != productList.size() - 1) {
                jsonBuilder.append(",");
            }
        }

        jsonBuilder.append("]");
        return jsonBuilder.toString();
    }

    public boolean isValidTxnId(String txnId) {
        if (txnId == null || txnId.trim().isEmpty()) {
            return false;
        } else if (txnId.length() > TRANSACTION_ID_LENGTH_LIMIT) {
            return false;
        }

        return txnId.matches("^[a-zA-Z0-9]*$");
    }
}

