/*
 * BSD 2-Clause License
 *
 * Copyright (c) 2021, Ondrej Fischer
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 *  Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

package fluent.validation;

import java.io.Serializable;
import java.lang.invoke.SerializedLambda;
import java.lang.reflect.Method;

import static java.util.Objects.isNull;

/**
 * Functional interface representing transformation of a data for partial check.
 *
 * @param <F> Original type, on which the transformation is applied (FROM).
 * @param <T> Result type, that the transformation returns (TO).
 */
@FunctionalInterface
public interface Transformation<F, T> extends Serializable {

    /**
     * Apply the transformation.
     * @param from Original object, on which to apply the transformation.
     * @return Returns transformed value.
     * @throws Exception Can throw any exception during transformation.
     */
    T apply(F from) throws Exception;

    default String getMethodName() {
        try {
            Method writeReplace = this.getClass().getDeclaredMethod("writeReplace");
            writeReplace.setAccessible(true);
            SerializedLambda sl = (SerializedLambda) writeReplace.invoke(this);
            writeReplace.setAccessible(false);
            String methodName = sl.getImplMethodName();
            return methodName.startsWith("get") ? methodName.substring(3) : methodName;
        } catch (Exception e) {
            return null;
        }
    }

    static <F, T> Transformation<F, T> dontTransformNull(Transformation<F, T> transformation) {
        return new Transformation<F, T>() {
            @Override
            public T apply(F f) throws Exception {
                return isNull(f) ? null : transformation.apply(f);
            }

            @Override
            public String getMethodName() {
                return transformation.getMethodName();
            }
        };
    }

}
