/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.security.engine.core;

import static java.util.Optional.ofNullable;

import java.util.List;
import java.util.function.BiPredicate;
import java.util.function.Function;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import org.springframework.security.util.FieldUtils;
import org.springframework.util.CollectionUtils;

public interface RoleHandler {

  List<Authority> getAuthorities();

  String getRole();

  BiPredicate<String, Object> getHandler();

  static RoleBuilder builder() {
    return new RoleBuilder.DefaultRoleBuilder();
  }

  default boolean handle(String authority, Object context) {
    BiPredicate<String, Object> handler = getHandler();
    if (handler == null) {
      handler = defaultHandler();
    }
    return handler.test(authority, context);
  }

  default BiPredicate<String, Object> defaultHandler() {

    return (authority, context) -> {
      final Authority currentAuthority = getAuthorities()
          .stream()
          .filter(auth -> authority.equalsIgnoreCase(auth.getName()))
          .findFirst()
          .orElse(null);

      if (currentAuthority == null) {
        return false;
      }

      final Fields fields = currentAuthority.getContext();
      final List<String> contextFields = ofNullable(fields)
          .map(Fields::getValue).orElse(null);

      if (CollectionUtils.isEmpty(contextFields)) {
        return currentAuthority.isAvailable();
      }

      final Function<String, Boolean> func = field -> getFieldValue(context, field);

      final Fields.Type type = fields.getType();
      final List<String> attr = fields.getValue();

      switch (type) {
        case AND:
          return attr.stream()
              .map(func)
              .filter(Boolean.FALSE::equals)
              .findFirst()
              .orElse(true);
        case OR:
          return attr.stream()
              .map(func)
              .filter(Boolean.TRUE::equals)
              .findFirst()
              .orElse(false);
        default:
          return false;
      }
    };
  }

  private boolean getFieldValue(final Object context, final String fieldName) {
    final Object value = FieldUtils.getProtectedFieldValue(fieldName, context);
    if (value instanceof Boolean) {
      return (Boolean) value;
    } else {
      throw new AuthorityException(
          String.format("Field [%s] of [%s] is not boolean.", fieldName, context.getClass())
      );
    }
  }

  @AllArgsConstructor(access = AccessLevel.PACKAGE)
  class DefaultRoleHandler implements RoleHandler {

    private String role;
    private List<Authority> authorities;
    private BiPredicate<String, Object> handler;

    @Override
    public List<Authority> getAuthorities() {
      return this.authorities;
    }

    @Override
    public String getRole() {
      return this.role;
    }

    @Override
    public BiPredicate<String, Object> getHandler() {
      return this.handler;
    }

  }
}
