/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.security.engine.core;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Optional;
import java.util.stream.Stream;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.stereotype.Component;

@Aspect
@Component
@Slf4j
@RequiredArgsConstructor
public class AuthorityHandlerAspect {

  private final SecurityEngine securityEngine;

  @Pointcut("@annotation(digital.nedra.commons.starter.security.engine.core.AuthorityCheck)")
  public void annotatedAuthorityPointcut() {
    //DO NOTHING
  }

  @Around("annotatedAuthorityPointcut()")
  public Object around(ProceedingJoinPoint joinPoint) throws Throwable {
    if (!securityEngine.isCalledFromRestController()) {
      return joinPoint.proceed();
    } else {
      extractAnnotation(joinPoint)
          .ifPresent(
              authorityCheck -> checkPermission(
                  joinPoint,
                  authorityCheck.handler(),
                  authorityCheck.value(),
                  authorityCheck.description()
              )
          );
      return joinPoint.proceed();
    }
  }

  @SuppressWarnings("all")
  private void checkPermission(final ProceedingJoinPoint joinPoint,
                               final Class<? extends ContextBuilder> builder,
                               final String authority,
                               final String errorMessage) {
    final MethodSignature signature = (MethodSignature) joinPoint.getSignature();
    final AuthorityContextPayload payload =
        getAuthorityContextPayload(
            joinPoint.getArgs(),
            signature.getMethod().getParameterAnnotations()
        );
    securityEngine.checkPermission(
        authority,
        errorMessage,
        (Class<? extends ContextBuilder<AuthorityContext>>) builder,
        payload
    );
  }

  private Optional<AuthorityCheck> extractAnnotation(final ProceedingJoinPoint joinPoint) {
    final MethodSignature methodSignature = (MethodSignature) joinPoint.getSignature();
    final Method method = methodSignature.getMethod();
    return Stream.of(method.getDeclaredAnnotations())
        .filter(annotation -> (AuthorityCheck.class.equals(annotation.annotationType())))
        .map(AuthorityCheck.class::cast)
        .findFirst();
  }

  private AuthorityContextPayload getAuthorityContextPayload(final Object[] args,
                                                             final Annotation[][] annotations) {
    final AuthorityContextPayload payload = new DefaultContextPayload();
    for (int i = 0; i < annotations.length; i++) {
      for (final Annotation annotation : annotations[i]) {
        final Class<? extends Annotation> clazz = annotation.annotationType();
        if ((AuthParam.class.equals(clazz))) {
          final String name = ((AuthParam) annotation).value();
          payload.set(name, args[i]);
        }
      }
    }
    return payload;
  }

}
