package de.pco.example;

/*-
 * #%L
 * pco-example
 * %%
 * Copyright (C) 2020 PCO
 * %%
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 * #L%
 */

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;

import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageTypeSpecifier;
import javax.imageio.stream.ImageInputStream;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.SwingUtilities;
import javax.swing.border.BevelBorder;
import javax.swing.filechooser.FileNameExtensionFilter;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import com.twelvemonkeys.imageio.plugins.tiff.TIFFImageMetadata;
import com.twelvemonkeys.imageio.plugins.tiff.TIFFImageReaderSpi;

import de.pco.imageio.B16ImageReader;
import de.pco.imageio.B16ImageReaderSpi;
import de.pco.imageio.B16ImageWriter;
import de.pco.imageio.PcoIIOMetadata;

/**
 * Example application that takes advantage of the PCO.java functions.
 * @author PCO
 *
 */
public class SimpleExample extends JFrame 
{
    private static StringBuilder sb = null;
    private static final String APPLICATION_TITLE = "PCO.java Example Application";
    private JMenuBar menuMB = null;
    private JMenuItem showMetadataMI = null;
    private PaintPanel paintP = null;
    private PcoIIOMetadata metadata = null;
    private MetadataTableModel metadataTM = null;
    private MetadataTableEditor metadataTE = null;
    private IIOImage imageContainer = null;
    private JMenuItem saveMI = null;
    private JLabel statusL = null;
    
    private JFileChooser openFC = null;
    private JFileChooser saveFC = null;
    
    private SimpleExample() 
    {
        SimpleExample self = this;
        this.setTitle(APPLICATION_TITLE);
        this.setSize(new Dimension(800, 500));
        this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        
        
        Dimension minimumSize = new Dimension(500, 50);
        this.paintP = new PaintPanel();
        JScrollPane paintSP = new JScrollPane(this.paintP);
        
        this.metadataTM = new MetadataTableModel();
        JTable tableT = new JTable(this.metadataTM);
        this.metadataTE = new MetadataTableEditor();
        tableT.setDefaultEditor(Object.class, this.metadataTE);
        tableT.setDefaultRenderer(Object.class, new MetadataTableRenderer());
        JScrollPane metadataSP = new JScrollPane(tableT);
        
        JSplitPane splitP = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, false, paintSP, metadataSP);
        splitP.setDividerLocation(0.9);
        this.add(splitP);
        paintSP.setMinimumSize(minimumSize);
//        metadataSP.setMinimumSize(minimumSize);
                
        this.menuMB = new JMenuBar();
        this.openFC = new JFileChooser();
        this.saveFC = new JFileChooser();
        JMenu fileM = new JMenu("File");
        fileM.setMnemonic('F');
        JMenuItem openMI = new JMenuItem("Open...");
        openMI.setMnemonic('O');
        openMI.addActionListener(new ActionListener() 
        {
            @Override
            public void actionPerformed(ActionEvent e) 
            {
                self.metadataTE.stopCellEditing();
                self.openFC.setFileFilter(new FileNameExtensionFilter("B16 File, TIFF File", "b16", "tif", "tiff"));
                if (self.openFC.getSelectedFile() != null) {
                    self.openFC.setCurrentDirectory(openFC.getSelectedFile().getParentFile());
                }
                if (self.openFC.showOpenDialog(self) == JFileChooser.APPROVE_OPTION) 
                {
                    File file = self.openFC.getSelectedFile();
                    BufferedImage image = null;
                    ImageReader reader = null;
                    boolean isB16 = false;
                    boolean isTiff = false;
                    try {
                        // simple method of obtaining image (without metadata) is
                        // image = ImageIO.read(file);
                        
                        // or this one:
                        ImageInputStream iis = ImageIO.createImageInputStream(file);
                        B16ImageReaderSpi b16Spi = new B16ImageReaderSpi();
                        TIFFImageReaderSpi tiffSpi = new TIFFImageReaderSpi();
                        if (b16Spi.canDecodeInput(iis))  // if (file.getName().endsWith("b16"))
                        {
                            isB16 = true;
                            reader = new B16ImageReader();
                        }
                        else if (tiffSpi.canDecodeInput(iis))
                        {
                            isTiff = true;
                            reader = new TIFFImageReaderSpi().createReaderInstance();
                        }
                        else {
                            JOptionPane.showMessageDialog(self, "File cannot be opened.", "Error", JOptionPane.ERROR_MESSAGE);
                            return;
                        }
                        reader.setInput(iis);
                        self.imageContainer = reader.readAll(0, reader.getDefaultReadParam());
                    } 
                    catch (IOException ex) {
                        ex.printStackTrace();
                    }
                    if (self.imageContainer != null) 
                    {
                        image = (BufferedImage)self.imageContainer.getRenderedImage();
                        if (isB16 == true) {
                            self.metadata = (PcoIIOMetadata)self.imageContainer.getMetadata();
                        }
                        else {
                            TIFFImageMetadata tim = (TIFFImageMetadata)self.imageContainer.getMetadata();
                            B16ImageWriter writer = new B16ImageWriter();
                            ImageTypeSpecifier imageType = null;
                            try {
                                imageType = reader.getImageTypes(0).next();
                                self.metadata = (PcoIIOMetadata)writer.convertImageMetadata(tim, imageType, null);
                            } catch (IOException e1) {
                                // TODO Auto-generated catch block
                                e1.printStackTrace();
                            }
                        }
                        
                        double aspectRatio = ((double)image.getWidth()) / image.getHeight();
                        String newTitle = APPLICATION_TITLE + " - " + file.getName();
                        self.setTitle(newTitle);
                        newTitle = file.getName();
                        if (isTiff == true) 
                        {
                            //if (image.getType() == BufferedImage.TYPE_USHORT_GRAY)
                            if (    self.metadata.getMetadataBean().bitRes == 16 
                                &&  self.metadata.getMetadataBean().isColor == false)
                            {
                                newTitle += " (16-Bit grayscale), convertible in B16";
                                self.saveMI.setEnabled(true);
                            }
                            else {
                                newTitle += " (";
                                newTitle += self.metadata.getMetadataBean().bitRes;
                                newTitle += "-Bit ";
                                if (self.metadata.getMetadataBean().isColor == true) {
                                    newTitle += "RGB)";
                                }
                                else {
                                    newTitle += "grayscale)";
                                } 
                                newTitle += ", not convertible in B16";
                                self.saveMI.setEnabled(false);
                            }
                        }
                        else {
                            self.saveMI.setEnabled(true);
                        }
                        self.statusL.setText(newTitle);
                        self.setSize(new Dimension((int)(800 * aspectRatio), 500));
                        self.getPaintP().setImage(image);
                        
                        self.metadataTM.setMetadataBean(metadata.getMetadataBean());
                        self.metadataTM.setLoaded(true);
                        self.metadataTM.fireTableDataChanged();
                        self.showMetadataMI.setEnabled(true);
                       
                        self.repaint();
                    }
                }
            }
        });
        this.saveMI = new JMenuItem("Save...");
        this.saveMI.setEnabled(false);
        this.saveMI.setMnemonic('S');
        this.saveMI.addActionListener(new ActionListener() 
        {
            @Override
            public void actionPerformed(ActionEvent e) 
            {
                self.metadataTE.stopCellEditing();
                if (self.saveFC.getSelectedFile() != null) {
                    self.saveFC.setCurrentDirectory(saveFC.getSelectedFile().getParentFile());
                }
                self.saveFC.setFileFilter(new FileNameExtensionFilter("B16 File", "b16"));
                boolean closeSaveDialog = false;
                File file = null;
                while (closeSaveDialog == false) 
                {
                    if (self.saveFC.showSaveDialog(self) == JFileChooser.APPROVE_OPTION) 
                    {
                        file = self.saveFC.getSelectedFile();
                        if (    file.getName().endsWith("b16") == false
                            &&  file.getName().endsWith("B16") == false) 
                        {
                            file = new File(file.getParentFile(), file.getName() + ".b16");
                        }
                        if (file.exists() == true) 
                        {
                            int result = JOptionPane.showOptionDialog(self, "File exists. Do you want to overwrite it?", 
                                "Overwrite", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE, 
                                null, null, null);
                            if (result == JOptionPane.YES_OPTION) {
                                closeSaveDialog = true;
                            }
                        }
                        else {
                            closeSaveDialog = true;
                        }
                    }
                    else {
                        return;
                    }
                }
                    
                try {
                    B16ImageWriter writer = new B16ImageWriter();
                    writer.setOutput(ImageIO.createImageOutputStream(file));
                    writer.write(imageContainer);
                    JOptionPane.showMessageDialog(self, "Image saved");
                    self.openFC.setSelectedFile(file);
                    self.statusL.setText(file.getName());
                    String newTitle = APPLICATION_TITLE + " - " + file.getName();
                    self.setTitle(newTitle);
                } 
                catch (IOException e1) {
                    // TODO Auto-generated catch block
                    e1.printStackTrace();
                }
            }
        });
        this.showMetadataMI = new JMenuItem("Show metadata");
        this.showMetadataMI.setEnabled(false);
        this.showMetadataMI.addActionListener(new ActionListener() 
        {
            @Override
            public void actionPerformed(ActionEvent e) 
            {
                self.metadataTE.stopCellEditing();
                JDialog dialog = new JDialog(self, "Metadata in the native XML", true);
                
                JTextArea metadataTA = new JTextArea();
                displayMetadata(self.metadata.getAsTree(self.metadata.getNativeMetadataFormatName()));
                JScrollPane metadataSP = new JScrollPane(metadataTA);
                JPanel contentPane = new JPanel(new BorderLayout());
                contentPane.add(metadataSP, BorderLayout.CENTER);
                contentPane.setOpaque(true);
                dialog.setContentPane(contentPane);
                dialog.setSize(new Dimension(600, 850));
                dialog.setLocationRelativeTo(self);
                metadataTA.setText(sb.toString());
                metadataTA.setCaretPosition(0);
                dialog.setVisible(true);
            }
        });
        JMenuItem exitMI = new JMenuItem("Exit");
        exitMI.setMnemonic('x');
        exitMI.addActionListener(new ActionListener() 
        {
            @Override
            public void actionPerformed(ActionEvent e) {
                self.dispose();
            }
        });
                
        fileM.add(openMI);
        fileM.add(this.saveMI);
        //fileM.add(this.showMetadataMI);
        fileM.add(new JSeparator());
        fileM.add(exitMI);
        this.menuMB.add(fileM);
        this.setJMenuBar(menuMB);
        
        JPanel statusBar = new JPanel(new BorderLayout());
        statusBar.setBorder(new BevelBorder(BevelBorder.LOWERED));
        this.statusL = new JLabel(" ");
        statusBar.add(statusL, BorderLayout.LINE_START);
        this.add(statusBar, BorderLayout.SOUTH);
        
        this.setVisible(true);
    }

    /**
     * Main method
     * @param args
     */
    public static void main(String[] args) 
    {
        SwingUtilities.invokeLater(new Runnable() {
          public void run() {
              SimpleExample example = new SimpleExample(); 
          }
      });
    }

    private PaintPanel getPaintP() {
        return this.paintP;
    }
    
    private MetadataTableModel getMetadataTM() {
        return this.metadataTM;
    }
    
    private static void displayMetadata(Node root) {
        sb = new StringBuilder();
        displayMetadata(root, 0);
    }

    private static void indent(int level) {
        for (int i = 0; i < level; i++) {
            sb.append("    ");
        }
    } 

    private static void displayMetadata(Node node, int level) {
        indent(level); // emit open tag
        sb.append("<" + node.getNodeName());
        NamedNodeMap map = node.getAttributes();
        if (map != null) { // print attribute values
                int length = map.getLength();
                for (int i = 0; i < length; i++) {
                        Node attr = map.item(i);
                        sb.append(" " + attr.getNodeName() +
                                         "=\"" + attr.getNodeValue() + "\"");
                }
        }

        Node child = node.getFirstChild();
        if (child != null) {
                sb.append(">\n"); // close current tag
                while (child != null) { // emit child tags recursively
                        displayMetadata(child, level + 1);
                        child = child.getNextSibling();
                }
                indent(level); // emit close tag
                sb.append("</" + node.getNodeName() + ">\n");
        } else {
                sb.append("/>\n");
        }
    }
}

