package de.flapdoodle.embed.mongo.types;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;

/**
 * Immutable implementation of {@link SystemProperties}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableSystemProperties.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutableSystemProperties extends SystemProperties {
  private final Map<String, String> value;

  private ImmutableSystemProperties(Map<String, String> value) {
    this.value = value;
  }

  /**
   * @return The value of the {@code value} attribute
   */
  @Override
  public Map<String, String> value() {
    return value;
  }

  /**
   * Copy the current immutable object by replacing the {@link SystemProperties#value() value} map with the specified map.
   * Nulls are not permitted as keys or values.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param entries The entries to be added to the value map
   * @return A modified copy of {@code this} object
   */
  public final ImmutableSystemProperties withValue(Map<String, ? extends String> entries) {
    if (this.value == entries) return this;
    Map<String, String> newValue = createUnmodifiableMap(true, false, entries);
    return new ImmutableSystemProperties(newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableSystemProperties} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableSystemProperties
        && equalTo(0, (ImmutableSystemProperties) another);
  }

  private boolean equalTo(int synthetic, ImmutableSystemProperties another) {
    return value.equals(another.value);
  }

  /**
   * Computes a hash code from attributes: {@code value}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + value.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code SystemProperties} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "SystemProperties{"
        + "value=" + value
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link SystemProperties} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable SystemProperties instance
   */
  public static ImmutableSystemProperties copyOf(SystemProperties instance) {
    if (instance instanceof ImmutableSystemProperties) {
      return (ImmutableSystemProperties) instance;
    }
    return ImmutableSystemProperties.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableSystemProperties ImmutableSystemProperties}.
   * <pre>
   * ImmutableSystemProperties.builder()
   *    .putValue|putAllValue(String =&gt; String) // {@link SystemProperties#value() value} mappings
   *    .build();
   * </pre>
   * @return A new ImmutableSystemProperties builder
   */
  public static ImmutableSystemProperties.Builder builder() {
    return new ImmutableSystemProperties.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableSystemProperties ImmutableSystemProperties}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder {
    private Map<String, String> value = new LinkedHashMap<String, String>();

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code SystemProperties} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(SystemProperties instance) {
      Objects.requireNonNull(instance, "instance");
      putAllValue(instance.value());
      return this;
    }

    /**
     * Put one entry to the {@link SystemProperties#value() value} map.
     * @param key The key in the value map
     * @param value The associated value in the value map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putValue(String key, String value) {
      this.value.put(
          Objects.requireNonNull(key, "value key"),
          Objects.requireNonNull(value, value == null ? "value value for key: " + key : null));
      return this;
    }

    /**
     * Put one entry to the {@link SystemProperties#value() value} map. Nulls are not permitted
     * @param entry The key and value entry
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putValue(Map.Entry<String, ? extends String> entry) {
      String k = entry.getKey();
      String v = entry.getValue();
      this.value.put(
          Objects.requireNonNull(k, "value key"),
          Objects.requireNonNull(v, v == null ? "value value for key: " + k : null));
      return this;
    }

    /**
     * Sets or replaces all mappings from the specified map as entries for the {@link SystemProperties#value() value} map. Nulls are not permitted
     * @param entries The entries that will be added to the value map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder value(Map<String, ? extends String> entries) {
      this.value.clear();
      return putAllValue(entries);
    }

    /**
     * Put all mappings from the specified map as entries to {@link SystemProperties#value() value} map. Nulls are not permitted
     * @param entries The entries that will be added to the value map
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder putAllValue(Map<String, ? extends String> entries) {
      for (Map.Entry<String, ? extends String> e : entries.entrySet()) {
        String k = e.getKey();
        String v = e.getValue();
        this.value.put(
            Objects.requireNonNull(k, "value key"),
            Objects.requireNonNull(v, v == null ? "value value for key: " + k : null));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableSystemProperties ImmutableSystemProperties}.
     * @return An immutable instance of SystemProperties
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableSystemProperties build() {
      return new ImmutableSystemProperties(createUnmodifiableMap(false, false, value));
    }
  }

  private static <K, V> Map<K, V> createUnmodifiableMap(boolean checkNulls, boolean skipNulls, Map<? extends K, ? extends V> map) {
    switch (map.size()) {
    case 0: return Collections.emptyMap();
    case 1: {
      Map.Entry<? extends K, ? extends V> e = map.entrySet().iterator().next();
      K k = e.getKey();
      V v = e.getValue();
      if (checkNulls) {
        Objects.requireNonNull(k, "key");
        Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
      }
      if (skipNulls && (k == null || v == null)) {
        return Collections.emptyMap();
      }
      return Collections.singletonMap(k, v);
    }
    default: {
      Map<K, V> linkedMap = new LinkedHashMap<>(map.size() * 4 / 3 + 1);
      if (skipNulls || checkNulls) {
        for (Map.Entry<? extends K, ? extends V> e : map.entrySet()) {
          K k = e.getKey();
          V v = e.getValue();
          if (skipNulls) {
            if (k == null || v == null) continue;
          } else if (checkNulls) {
            Objects.requireNonNull(k, "key");
            Objects.requireNonNull(v, v == null ? "value for key: " + k : null);
          }
          linkedMap.put(k, v);
        }
      } else {
        linkedMap.putAll(map);
      }
      return Collections.unmodifiableMap(linkedMap);
    }
    }
  }
}
