
package de.flapdoodle.embed.mongo.packageresolver;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Immutable implementation of {@link PackageFinderRule}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutablePackageFinderRule.builder()}.
 */
@SuppressWarnings({"all"})
public final class ImmutablePackageFinderRule
    implements PackageFinderRule {
  private final DistributionMatch match;
  private final PackageFinder finder;

  private ImmutablePackageFinderRule(
      DistributionMatch match,
      PackageFinder finder) {
    this.match = match;
    this.finder = finder;
  }

  /**
   * @return The value of the {@code match} attribute
   */
  @Override
  public DistributionMatch match() {
    return match;
  }

  /**
   * @return The value of the {@code finder} attribute
   */
  @Override
  public PackageFinder finder() {
    return finder;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageFinderRule#match() match} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for match
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePackageFinderRule withMatch(DistributionMatch value) {
    if (this.match == value) return this;
    DistributionMatch newValue = Objects.requireNonNull(value, "match");
    return new ImmutablePackageFinderRule(newValue, this.finder);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link PackageFinderRule#finder() finder} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for finder
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePackageFinderRule withFinder(PackageFinder value) {
    if (this.finder == value) return this;
    PackageFinder newValue = Objects.requireNonNull(value, "finder");
    return new ImmutablePackageFinderRule(this.match, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePackageFinderRule} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePackageFinderRule
        && equalsByValue((ImmutablePackageFinderRule) another);
  }

  private boolean equalsByValue(ImmutablePackageFinderRule another) {
    return match.equals(another.match)
        && finder.equals(another.finder);
  }

  /**
   * Computes a hash code from attributes: {@code match}, {@code finder}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + match.hashCode();
    h += (h << 5) + finder.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code PackageFinderRule} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "PackageFinderRule{"
        + "match=" + match
        + ", finder=" + finder
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link PackageFinderRule} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable PackageFinderRule instance
   */
  public static ImmutablePackageFinderRule copyOf(PackageFinderRule instance) {
    if (instance instanceof ImmutablePackageFinderRule) {
      return (ImmutablePackageFinderRule) instance;
    }
    return ((ImmutablePackageFinderRule.Builder) ImmutablePackageFinderRule.builder())
        .match(instance.match())
        .finder(instance.finder())
        .build();
  }

  /**
   * Creates a builder for {@link ImmutablePackageFinderRule ImmutablePackageFinderRule}.
   * <pre>
   * ImmutablePackageFinderRule.builder()
   *    .match(de.flapdoodle.embed.mongo.packageresolver.DistributionMatch) // required {@link PackageFinderRule#match() match}
   *    .finder(de.flapdoodle.embed.mongo.packageresolver.PackageFinder) // required {@link PackageFinderRule#finder() finder}
   *    .build();
   * </pre>
   * @return A new ImmutablePackageFinderRule builder
   */
  public static MatchBuildStage builder() {
    return new ImmutablePackageFinderRule.Builder();
  }

  /**
   * Builds instances of type {@link ImmutablePackageFinderRule ImmutablePackageFinderRule}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  public static final class Builder implements MatchBuildStage, FinderBuildStage, BuildFinal {
    private static final long INIT_BIT_MATCH = 0x1L;
    private static final long INIT_BIT_FINDER = 0x2L;
    private long initBits = 0x3L;

    private DistributionMatch match;
    private PackageFinder finder;

    private Builder() {
    }

    /**
     * Initializes the value for the {@link PackageFinderRule#match() match} attribute.
     * @param match The value for match 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder match(DistributionMatch match) {
      checkNotIsSet(matchIsSet(), "match");
      this.match = Objects.requireNonNull(match, "match");
      initBits &= ~INIT_BIT_MATCH;
      return this;
    }

    /**
     * Initializes the value for the {@link PackageFinderRule#finder() finder} attribute.
     * @param finder The value for finder 
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder finder(PackageFinder finder) {
      checkNotIsSet(finderIsSet(), "finder");
      this.finder = Objects.requireNonNull(finder, "finder");
      initBits &= ~INIT_BIT_FINDER;
      return this;
    }

    /**
     * Builds a new {@link ImmutablePackageFinderRule ImmutablePackageFinderRule}.
     * @return An immutable instance of PackageFinderRule
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutablePackageFinderRule build() {
      checkRequiredAttributes();
      return new ImmutablePackageFinderRule(match, finder);
    }

    private boolean matchIsSet() {
      return (initBits & INIT_BIT_MATCH) == 0;
    }

    private boolean finderIsSet() {
      return (initBits & INIT_BIT_FINDER) == 0;
    }

    private static void checkNotIsSet(boolean isSet, String name) {
      if (isSet) throw new IllegalStateException("Builder of PackageFinderRule is strict, attribute is already set: ".concat(name));
    }

    private void checkRequiredAttributes() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if (!matchIsSet()) attributes.add("match");
      if (!finderIsSet()) attributes.add("finder");
      return "Cannot build PackageFinderRule, some of required attributes are not set " + attributes;
    }
  }

  public interface MatchBuildStage {
    /**
     * Initializes the value for the {@link PackageFinderRule#match() match} attribute.
     * @param match The value for match 
     * @return {@code this} builder for use in a chained invocation
     */
    FinderBuildStage match(DistributionMatch match);
  }

  public interface FinderBuildStage {
    /**
     * Initializes the value for the {@link PackageFinderRule#finder() finder} attribute.
     * @param finder The value for finder 
     * @return {@code this} builder for use in a chained invocation
     */
    BuildFinal finder(PackageFinder finder);
  }

  public interface BuildFinal {

    /**
     * Builds a new {@link ImmutablePackageFinderRule ImmutablePackageFinderRule}.
     * @return An immutable instance of PackageFinderRule
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    ImmutablePackageFinderRule build();
  }
}
