/*
 * Decompiled with CFR 0.152.
 */
package de.bwaldvogel.mongo.wire.bson;

import de.bwaldvogel.mongo.backend.Assert;
import de.bwaldvogel.mongo.bson.BsonJavaScript;
import de.bwaldvogel.mongo.bson.BsonRegularExpression;
import de.bwaldvogel.mongo.bson.BsonTimestamp;
import de.bwaldvogel.mongo.bson.Decimal128;
import de.bwaldvogel.mongo.bson.Document;
import de.bwaldvogel.mongo.bson.LegacyUUID;
import de.bwaldvogel.mongo.bson.MaxKey;
import de.bwaldvogel.mongo.bson.MinKey;
import de.bwaldvogel.mongo.bson.ObjectId;
import io.netty.buffer.ByteBuf;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

public final class BsonDecoder {
    private BsonDecoder() {
    }

    public static Document decodeBson(ByteBuf buffer) {
        int totalObjectLength = buffer.readIntLE();
        int length = totalObjectLength - 4;
        if (buffer.readableBytes() < length) {
            throw new IllegalArgumentException("Too few bytes to read: " + buffer.readableBytes() + ". Expected: " + length);
        }
        if (length > 0x1000000) {
            throw new IllegalArgumentException("BSON object too large: " + length + " bytes");
        }
        Document document = new Document();
        int start = buffer.readerIndex();
        while (buffer.readerIndex() - start < length) {
            byte type = buffer.readByte();
            if (type == 0) {
                return document;
            }
            String name = BsonDecoder.decodeCString(buffer);
            Object value = BsonDecoder.decodeValue(type, buffer);
            Object existingValue = document.put(name, value);
            Assert.isNull(existingValue, () -> "Document already contains field '" + name + "'");
        }
        throw new IllegalArgumentException("illegal BSON object. Terminating byte not found. totalObjectLength = " + totalObjectLength);
    }

    public static Object decodeValue(byte type, ByteBuf buffer) {
        switch (type) {
            case 1: {
                return Double.longBitsToDouble(buffer.readLongLE());
            }
            case 2: {
                return BsonDecoder.decodeString(buffer);
            }
            case 3: {
                return BsonDecoder.decodeBson(buffer);
            }
            case 4: {
                return BsonDecoder.decodeArray(buffer);
            }
            case 5: {
                return BsonDecoder.decodeBinary(buffer);
            }
            case 6: 
            case 10: {
                return null;
            }
            case 7: {
                return BsonDecoder.decodeObjectId(buffer);
            }
            case 8: {
                return BsonDecoder.decodeBoolean(buffer);
            }
            case 9: {
                return Instant.ofEpochMilli(buffer.readLongLE());
            }
            case 11: {
                return BsonDecoder.decodePattern(buffer);
            }
            case 16: {
                return buffer.readIntLE();
            }
            case 17: {
                return new BsonTimestamp(buffer.readLongLE());
            }
            case 18: {
                return buffer.readLongLE();
            }
            case 19: {
                return new Decimal128(buffer.readLongLE(), buffer.readLongLE());
            }
            case 127: {
                return MaxKey.getInstance();
            }
            case -1: {
                return MinKey.getInstance();
            }
            case 13: {
                return new BsonJavaScript(BsonDecoder.decodeString(buffer));
            }
            case 15: {
                throw new IllegalArgumentException("unhandled type: 0x" + Integer.toHexString(type));
            }
        }
        throw new IllegalArgumentException("unknown type: 0x" + Integer.toHexString(type));
    }

    private static BsonRegularExpression decodePattern(ByteBuf buffer) {
        String regex = BsonDecoder.decodeCString(buffer);
        String options = BsonDecoder.decodeCString(buffer);
        return new BsonRegularExpression(regex, options);
    }

    private static List<Object> decodeArray(ByteBuf buffer) {
        ArrayList<Object> array = new ArrayList<Object>();
        Document arrayObject = BsonDecoder.decodeBson(buffer);
        for (String key : arrayObject.keySet()) {
            array.add(arrayObject.get(key));
        }
        return array;
    }

    private static ObjectId decodeObjectId(ByteBuf buffer) {
        byte[] b = new byte[12];
        buffer.readBytes(b);
        return new ObjectId(b);
    }

    private static String decodeString(ByteBuf buffer) {
        int length = buffer.readIntLE();
        byte[] data = new byte[length - 1];
        buffer.readBytes(data);
        String value = new String(data, StandardCharsets.UTF_8);
        byte trail = buffer.readByte();
        if (trail != 0) {
            throw new IllegalArgumentException("Unexpected trailing byte: " + trail);
        }
        return value;
    }

    public static String decodeCString(ByteBuf buffer) {
        int length = buffer.bytesBefore((byte)0);
        if (length < 0) {
            throw new IllegalArgumentException("string termination not found");
        }
        String result = buffer.toString(buffer.readerIndex(), length, StandardCharsets.UTF_8);
        buffer.skipBytes(length + 1);
        return result;
    }

    private static Object decodeBinary(ByteBuf buffer) {
        int length = buffer.readIntLE();
        byte subtype = buffer.readByte();
        switch (subtype) {
            case -128: 
            case 0: {
                byte[] data = new byte[length];
                buffer.readBytes(data);
                return data;
            }
            case 3: {
                if (length != 16) {
                    throw new IllegalArgumentException("Illegal length: " + length);
                }
                return new LegacyUUID(buffer.readLongLE(), buffer.readLongLE());
            }
            case 4: {
                if (length != 16) {
                    throw new IllegalArgumentException("Illegal length: " + length);
                }
                return new UUID(buffer.readLong(), buffer.readLong());
            }
        }
        throw new IllegalArgumentException("Unknown subtype: " + subtype);
    }

    private static Object decodeBoolean(ByteBuf buffer) {
        byte value = buffer.readByte();
        switch (value) {
            case 0: {
                return Boolean.FALSE;
            }
            case 1: {
                return Boolean.TRUE;
            }
        }
        throw new IllegalArgumentException("Illegal boolean value:" + value);
    }
}

