/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.security;

import com.yahoo.security.Base58;
import com.yahoo.security.BouncyCastleProviderHolder;
import com.yahoo.security.KeyAlgorithm;
import com.yahoo.security.KeyFormat;
import com.yahoo.security.SideChannelSafe;
import java.io.IOException;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.UncheckedIOException;
import java.io.Writer;
import java.math.BigInteger;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.interfaces.RSAPrivateCrtKey;
import java.security.interfaces.XECPrivateKey;
import java.security.interfaces.XECPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.KeySpec;
import java.security.spec.NamedParameterSpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.security.spec.XECPrivateKeySpec;
import java.security.spec.XECPublicKeySpec;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Optional;
import javax.crypto.KeyAgreement;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.edec.EdECObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.PKCSObjectIdentifiers;
import org.bouncycastle.asn1.pkcs.PrivateKeyInfo;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.SubjectPublicKeyInfo;
import org.bouncycastle.asn1.x9.X9ObjectIdentifiers;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPrivateKey;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPublicKeySpec;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointCombMultiplier;
import org.bouncycastle.math.ec.rfc7748.X25519;
import org.bouncycastle.openssl.PEMKeyPair;
import org.bouncycastle.openssl.PEMParser;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.bouncycastle.util.Arrays;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemObjectGenerator;

public class KeyUtils {
    private KeyUtils() {
    }

    public static KeyPair generateKeypair(KeyAlgorithm algorithm, int keySize) {
        try {
            KeyPairGenerator keyGen = KeyPairGenerator.getInstance(algorithm.getAlgorithmName(), (Provider)BouncyCastleProviderHolder.getInstance());
            if (keySize != -1) {
                keyGen.initialize(keySize);
            }
            if (algorithm.getSpec().isPresent()) {
                keyGen.initialize(algorithm.getSpec().get());
            }
            return keyGen.genKeyPair();
        }
        catch (GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    public static KeyPair generateKeypair(KeyAlgorithm algorithm) {
        return KeyUtils.generateKeypair(algorithm, -1);
    }

    public static PublicKey extractPublicKey(PrivateKey privateKey) {
        KeyAlgorithm keyAlgorithm = KeyAlgorithm.from(privateKey.getAlgorithm());
        try {
            return switch (keyAlgorithm) {
                default -> throw new IncompatibleClassChangeError();
                case KeyAlgorithm.RSA -> {
                    KeyFactory keyFactory = KeyUtils.createKeyFactory(KeyAlgorithm.RSA);
                    RSAPrivateCrtKey rsaPrivateCrtKey = (RSAPrivateCrtKey)privateKey;
                    RSAPublicKeySpec keySpec = new RSAPublicKeySpec(rsaPrivateCrtKey.getModulus(), rsaPrivateCrtKey.getPublicExponent());
                    yield keyFactory.generatePublic(keySpec);
                }
                case KeyAlgorithm.EC -> {
                    KeyFactory keyFactory = KeyUtils.createKeyFactory(KeyAlgorithm.EC);
                    BCECPrivateKey ecPrivateKey = (BCECPrivateKey)privateKey;
                    ECParameterSpec ecParameterSpec = ecPrivateKey.getParameters();
                    ECPoint ecPoint = new FixedPointCombMultiplier().multiply(ecParameterSpec.getG(), ecPrivateKey.getD());
                    ECPublicKeySpec keySpec = new ECPublicKeySpec(ecPoint, ecParameterSpec);
                    yield keyFactory.generatePublic((KeySpec)keySpec);
                }
                case KeyAlgorithm.XDH -> {
                    byte[] privScalar = KeyUtils.toRawX25519PrivateKeyBytes((XECPrivateKey)privateKey);
                    byte[] pubPoint = new byte[32];
                    X25519.generatePublicKey((byte[])privScalar, (int)0, (byte[])pubPoint, (int)0);
                    yield KeyUtils.fromRawX25519PublicKey(pubPoint);
                }
            };
        }
        catch (GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    public static KeyPair toKeyPair(PrivateKey privateKey) {
        return new KeyPair(KeyUtils.extractPublicKey(privateKey), privateKey);
    }

    public static KeyPair keyPairFromPemEncodedPrivateKey(String pem) {
        return KeyUtils.toKeyPair(KeyUtils.fromPemEncodedPrivateKey(pem));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static PrivateKey fromPemEncodedPrivateKey(String pem) {
        try (PEMParser parser = new PEMParser((Reader)new StringReader(pem));){
            Object pemObject;
            ArrayList<Object> unknownObjects = new ArrayList<Object>();
            while ((pemObject = parser.readObject()) != null) {
                if (pemObject instanceof PrivateKeyInfo) {
                    PrivateKeyInfo keyInfo = (PrivateKeyInfo)pemObject;
                    PKCS8EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(keyInfo.getEncoded());
                    PrivateKey privateKey = KeyUtils.createKeyFactory(keyInfo.getPrivateKeyAlgorithm()).generatePrivate(keySpec);
                    return privateKey;
                }
                if (pemObject instanceof PEMKeyPair) {
                    PEMKeyPair pemKeypair = (PEMKeyPair)pemObject;
                    PrivateKeyInfo keyInfo = pemKeypair.getPrivateKeyInfo();
                    PrivateKey privateKey = KeyUtils.createKeyFactory(keyInfo.getPrivateKeyAlgorithm()).generatePrivate(new PKCS8EncodedKeySpec(keyInfo.getEncoded()));
                    return privateKey;
                }
                unknownObjects.add(pemObject);
            }
            throw new IllegalArgumentException("Expected a private key, but found " + String.valueOf(unknownObjects));
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        catch (GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static PublicKey fromPemEncodedPublicKey(String pem) {
        try (PEMParser parser = new PEMParser((Reader)new StringReader(pem));){
            SubjectPublicKeyInfo keyInfo;
            block12: {
                Object pemObject;
                ArrayList<Object> unknownObjects = new ArrayList<Object>();
                while ((pemObject = parser.readObject()) != null) {
                    if (pemObject instanceof SubjectPublicKeyInfo) {
                        keyInfo = (SubjectPublicKeyInfo)pemObject;
                        break block12;
                    }
                    if (pemObject instanceof PEMKeyPair) {
                        PEMKeyPair pemKeypair = (PEMKeyPair)pemObject;
                        keyInfo = pemKeypair.getPublicKeyInfo();
                        break block12;
                    }
                    unknownObjects.add(pemObject);
                }
                throw new IllegalArgumentException("Expected a public key, but found " + ((Object)unknownObjects).toString());
            }
            PublicKey publicKey = KeyUtils.createKeyFactory(keyInfo.getAlgorithm()).generatePublic(new X509EncodedKeySpec(keyInfo.getEncoded()));
            return publicKey;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        catch (GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    public static String toPem(PrivateKey privateKey) {
        return KeyUtils.toPem(privateKey, KeyFormat.PKCS1);
    }

    public static String toPem(PrivateKey privateKey, KeyFormat format) {
        return switch (format) {
            default -> throw new IncompatibleClassChangeError();
            case KeyFormat.PKCS1 -> KeyUtils.toPkcs1Pem(privateKey);
            case KeyFormat.PKCS8 -> KeyUtils.toPkcs8Pem(privateKey);
        };
    }

    /*
     * Enabled aggressive exception aggregation
     */
    public static String toPem(PublicKey publicKey) {
        try (StringWriter stringWriter = new StringWriter();){
            String string;
            try (JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)stringWriter);){
                pemWriter.writeObject((Object)publicKey);
                pemWriter.flush();
                string = stringWriter.toString();
            }
            return string;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static String toPkcs1Pem(PrivateKey privateKey) {
        try (StringWriter stringWriter = new StringWriter();){
            String string;
            try (JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)stringWriter);){
                KeyAlgorithm keyAlgorithm = KeyAlgorithm.from(privateKey.getAlgorithm());
                String type = switch (keyAlgorithm) {
                    default -> throw new IncompatibleClassChangeError();
                    case KeyAlgorithm.RSA -> "RSA PRIVATE KEY";
                    case KeyAlgorithm.EC -> "EC PRIVATE KEY";
                    case KeyAlgorithm.XDH -> throw new IllegalArgumentException("Cannot use PKCS#1 for X25519 key");
                };
                pemWriter.writeObject((PemObjectGenerator)new PemObject(type, KeyUtils.getPkcs1Bytes(privateKey)));
                pemWriter.flush();
                string = stringWriter.toString();
            }
            return string;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    private static String toPkcs8Pem(PrivateKey privateKey) {
        try (StringWriter stringWriter = new StringWriter();){
            String string;
            try (JcaPEMWriter pemWriter = new JcaPEMWriter((Writer)stringWriter);){
                pemWriter.writeObject((PemObjectGenerator)new PemObject("PRIVATE KEY", privateKey.getEncoded()));
                pemWriter.flush();
                string = stringWriter.toString();
            }
            return string;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private static byte[] getPkcs1Bytes(PrivateKey privateKey) throws IOException {
        byte[] privBytes = privateKey.getEncoded();
        PrivateKeyInfo pkInfo = PrivateKeyInfo.getInstance((Object)privBytes);
        ASN1Encodable encodable = pkInfo.parsePrivateKey();
        ASN1Primitive primitive = encodable.toASN1Primitive();
        return primitive.getEncoded();
    }

    private static KeyFactory createKeyFactory(AlgorithmIdentifier algorithm) throws NoSuchAlgorithmException {
        if (X9ObjectIdentifiers.id_ecPublicKey.equals((ASN1Primitive)algorithm.getAlgorithm())) {
            return KeyUtils.createKeyFactory(KeyAlgorithm.EC);
        }
        if (PKCSObjectIdentifiers.rsaEncryption.equals((ASN1Primitive)algorithm.getAlgorithm())) {
            return KeyUtils.createKeyFactory(KeyAlgorithm.RSA);
        }
        if (EdECObjectIdentifiers.id_X25519.equals((ASN1Primitive)algorithm.getAlgorithm())) {
            return KeyUtils.createKeyFactory(KeyAlgorithm.XDH);
        }
        throw new IllegalArgumentException("Unknown key algorithm: " + String.valueOf(algorithm));
    }

    private static KeyFactory createKeyFactory(KeyAlgorithm algorithm) throws NoSuchAlgorithmException {
        return KeyFactory.getInstance(algorithm.getAlgorithmName(), (Provider)BouncyCastleProviderHolder.getInstance());
    }

    public static XECPublicKey fromRawX25519PublicKey(byte[] rawKeyBytes) {
        try {
            NamedParameterSpec paramSpec = new NamedParameterSpec("X25519");
            KeyFactory keyFactory = KeyFactory.getInstance("XDH");
            byte[] asBigEndian = Arrays.reverse((byte[])rawKeyBytes);
            asBigEndian[0] = (byte)(asBigEndian[0] & 0x7F);
            BigInteger pubU = new BigInteger(asBigEndian);
            return (XECPublicKey)keyFactory.generatePublic(new XECPublicKeySpec(paramSpec, pubU));
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] toRawX25519PublicKeyBytes(XECPublicKey publicKey) {
        return Arrays.reverse((byte[])publicKey.getU().toByteArray());
    }

    public static XECPublicKey fromBase64EncodedX25519PublicKey(String base64pk) {
        byte[] rawKeyBytes = Base64.getUrlDecoder().decode(base64pk);
        return KeyUtils.fromRawX25519PublicKey(rawKeyBytes);
    }

    public static String toBase64EncodedX25519PublicKey(XECPublicKey publicKey) {
        return Base64.getUrlEncoder().withoutPadding().encodeToString(KeyUtils.toRawX25519PublicKeyBytes(publicKey));
    }

    private static void verifyB58InputSmallEnoughToBeX25519Key(String key) {
        if (key.length() > 64) {
            throw new IllegalArgumentException("Input Base58 is too large to represent an X25519 key");
        }
    }

    public static XECPublicKey fromBase58EncodedX25519PublicKey(String base58pk) {
        KeyUtils.verifyB58InputSmallEnoughToBeX25519Key(base58pk);
        byte[] rawKeyBytes = Base58.codec().decode(base58pk);
        return KeyUtils.fromRawX25519PublicKey(rawKeyBytes);
    }

    public static String toBase58EncodedX25519PublicKey(XECPublicKey publicKey) {
        return Base58.codec().encode(KeyUtils.toRawX25519PublicKeyBytes(publicKey));
    }

    public static XECPrivateKey fromRawX25519PrivateKey(byte[] rawScalarBytes) {
        try {
            NamedParameterSpec paramSpec = new NamedParameterSpec("X25519");
            KeyFactory keyFactory = KeyFactory.getInstance("XDH");
            return (XECPrivateKey)keyFactory.generatePrivate(new XECPrivateKeySpec(paramSpec, rawScalarBytes));
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw new RuntimeException(e);
        }
    }

    public static byte[] toRawX25519PrivateKeyBytes(XECPrivateKey privateKey) {
        Optional<byte[]> maybeScalar = privateKey.getScalar();
        if (maybeScalar.isPresent()) {
            return maybeScalar.get();
        }
        throw new IllegalArgumentException("Could not extract scalar representation of X25519 private key. It might be a hardware-protected private key.");
    }

    public static XECPrivateKey fromBase64EncodedX25519PrivateKey(String base64pk) {
        byte[] rawKeyBytes = Base64.getUrlDecoder().decode(base64pk);
        return KeyUtils.fromRawX25519PrivateKey(rawKeyBytes);
    }

    public static String toBase64EncodedX25519PrivateKey(XECPrivateKey privateKey) {
        return Base64.getUrlEncoder().withoutPadding().encodeToString(KeyUtils.toRawX25519PrivateKeyBytes(privateKey));
    }

    public static XECPrivateKey fromBase58EncodedX25519PrivateKey(String base58pk) {
        KeyUtils.verifyB58InputSmallEnoughToBeX25519Key(base58pk);
        byte[] rawKeyBytes = Base58.codec().decode(base58pk);
        return KeyUtils.fromRawX25519PrivateKey(rawKeyBytes);
    }

    public static String toBase58EncodedX25519PrivateKey(XECPrivateKey privateKey) {
        return Base58.codec().encode(KeyUtils.toRawX25519PrivateKeyBytes(privateKey));
    }

    public static KeyPair generateX25519KeyPair() {
        return KeyUtils.generateKeypair(KeyAlgorithm.XDH);
    }

    public static XECPublicKey extractX25519PublicKey(XECPrivateKey privateKey) {
        return (XECPublicKey)KeyUtils.extractPublicKey(privateKey);
    }

    public static byte[] ecdh(XECPrivateKey privateKey, XECPublicKey publicKey) {
        try {
            KeyAgreement keyAgreement = KeyAgreement.getInstance("XDH");
            keyAgreement.init(privateKey);
            keyAgreement.doPhase(publicKey, true);
            byte[] sharedSecret = keyAgreement.generateSecret();
            if (SideChannelSafe.allZeros(sharedSecret)) {
                throw new IllegalArgumentException("Computed shared secret is all zeroes");
            }
            return sharedSecret;
        }
        catch (InvalidKeyException | NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }
}

