/*
 * Decompiled with CFR 0.152.
 */
package com.yahoo.config.application.api;

import java.time.DateTimeException;
import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

public class TimeWindow {
    private final List<DayOfWeek> days;
    private final List<Integer> hours;
    private final ZoneId zone;
    private final LocalDateRange dateRange;

    private TimeWindow(List<DayOfWeek> days, List<Integer> hours, ZoneId zone, LocalDateRange dateRange) {
        this.days = Objects.requireNonNull(days).stream().distinct().sorted().collect(Collectors.toUnmodifiableList());
        this.hours = Objects.requireNonNull(hours).stream().distinct().sorted().collect(Collectors.toUnmodifiableList());
        this.zone = Objects.requireNonNull(zone);
        this.dateRange = Objects.requireNonNull(dateRange);
        if (days.isEmpty()) {
            throw new IllegalArgumentException("At least one day must be specified");
        }
        if (hours.isEmpty()) {
            throw new IllegalArgumentException("At least one hour must be specified");
        }
        for (DayOfWeek day : days) {
            if (dateRange.days().contains(day)) continue;
            throw new IllegalArgumentException("Invalid day: " + dateRange + " does not contain " + day);
        }
    }

    public List<DayOfWeek> days() {
        return this.days;
    }

    public List<Integer> hours() {
        return this.hours;
    }

    public ZoneId zone() {
        return this.zone;
    }

    public LocalDateRange dateRange() {
        return this.dateRange;
    }

    public boolean includes(Instant instant) {
        LocalDateTime dt = LocalDateTime.ofInstant(instant, this.zone);
        return this.days.contains(dt.getDayOfWeek()) && this.hours.contains(dt.getHour()) && this.dateRange.includes(dt.toLocalDate());
    }

    public String toString() {
        return "time window for hour(s) " + this.hours.toString() + " on " + this.days.stream().map(Enum::name).map(String::toLowerCase).toList() + " in time zone " + this.zone + " and " + this.dateRange.toString();
    }

    public static TimeWindow from(String daySpec, String hourSpec, String zoneSpec, String dateStart, String dateEnd) {
        LocalDateRange dateRange = LocalDateRange.from(dateStart, dateEnd);
        List<DayOfWeek> days = daySpec.isEmpty() ? List.copyOf(dateRange.days()) : TimeWindow.parse(daySpec, TimeWindow::parseDays);
        List<Integer> hours = hourSpec.isEmpty() ? IntStream.rangeClosed(0, 23).boxed().toList() : TimeWindow.parse(hourSpec, TimeWindow::parseHours);
        ZoneId zone = TimeWindow.zoneFrom(zoneSpec.isEmpty() ? "UTC" : zoneSpec);
        return new TimeWindow(days, hours, zone, dateRange);
    }

    private static <T> List<T> parse(String spec, BiFunction<String, String, List<T>> valueParser) {
        String[] parts;
        ArrayList values = new ArrayList();
        for (String part : parts = spec.split(",")) {
            if (part.contains("-")) {
                String[] startAndEnd = part.split("-");
                if (startAndEnd.length != 2) {
                    throw new IllegalArgumentException("Invalid range '" + part + "'");
                }
                values.addAll(valueParser.apply(startAndEnd[0], startAndEnd[1]));
                continue;
            }
            values.addAll(valueParser.apply(part, part));
        }
        return Collections.unmodifiableList(values);
    }

    private static List<Integer> parseHours(String startInclusive, String endInclusive) {
        int start = TimeWindow.hourFrom(startInclusive);
        int end = TimeWindow.hourFrom(endInclusive);
        if (end < start) {
            throw new IllegalArgumentException(String.format("Invalid hour range '%s-%s'", startInclusive, endInclusive));
        }
        return IntStream.rangeClosed(start, end).boxed().toList();
    }

    private static List<DayOfWeek> parseDays(String startInclusive, String endInclusive) {
        DayOfWeek start = TimeWindow.dayFrom(startInclusive);
        DayOfWeek end = TimeWindow.dayFrom(endInclusive);
        if (end.getValue() < start.getValue()) {
            throw new IllegalArgumentException(String.format("Invalid day range '%s-%s'", startInclusive, endInclusive));
        }
        return IntStream.rangeClosed(start.getValue(), end.getValue()).boxed().map(DayOfWeek::of).toList();
    }

    private static DayOfWeek dayFrom(String day) {
        return Arrays.stream(DayOfWeek.values()).filter(dayOfWeek -> day.length() >= 3 && dayOfWeek.name().toLowerCase().startsWith(day)).findFirst().orElseThrow(() -> new IllegalArgumentException("Invalid day '" + day + "'"));
    }

    private static int hourFrom(String hour) {
        try {
            return ChronoField.HOUR_OF_DAY.checkValidIntValue(Integer.parseInt(hour));
        }
        catch (NumberFormatException | DateTimeException e) {
            throw new IllegalArgumentException("Invalid hour '" + hour + "'", e);
        }
    }

    private static ZoneId zoneFrom(String zone) {
        try {
            return ZoneId.of(zone);
        }
        catch (DateTimeException e) {
            throw new IllegalArgumentException("Invalid time zone '" + zone + "'", e);
        }
    }

    public static class LocalDateRange {
        private final Optional<LocalDate> start;
        private final Optional<LocalDate> end;

        private LocalDateRange(Optional<LocalDate> start, Optional<LocalDate> end) {
            this.start = Objects.requireNonNull(start);
            this.end = Objects.requireNonNull(end);
            if (start.isPresent() && end.isPresent() && start.get().isAfter(end.get())) {
                throw new IllegalArgumentException("Invalid date range: start date " + start.get() + " is after end date " + end.get());
            }
        }

        public Optional<LocalDate> start() {
            return this.start;
        }

        public Optional<LocalDate> end() {
            return this.end;
        }

        private Set<DayOfWeek> days() {
            if (this.start.isEmpty() || this.end.isEmpty()) {
                return EnumSet.allOf(DayOfWeek.class);
            }
            EnumSet<DayOfWeek> days = EnumSet.noneOf(DayOfWeek.class);
            LocalDate date = this.start.get();
            while (!date.isAfter(this.end.get()) && days.size() < 7) {
                days.add(date.getDayOfWeek());
                date = date.plusDays(1L);
            }
            return days;
        }

        private boolean includes(LocalDate date) {
            if (this.start.isPresent() && date.isBefore(this.start.get())) {
                return false;
            }
            return !this.end.isPresent() || !date.isAfter(this.end.get());
        }

        public String toString() {
            return "date range [" + this.start.map(LocalDate::toString).orElse("any date") + ", " + this.end.map(LocalDate::toString).orElse("any date") + "]";
        }

        private static LocalDateRange from(String start, String end) {
            try {
                return new LocalDateRange(LocalDateRange.optionalDate(start), LocalDateRange.optionalDate(end));
            }
            catch (DateTimeParseException e) {
                throw new IllegalArgumentException("Could not parse date range '" + start + "' and '" + end + "'", e);
            }
        }

        private static Optional<LocalDate> optionalDate(String date) {
            return Optional.of(date).filter(s -> !s.isEmpty()).map(LocalDate::parse);
        }
    }
}

