package com.wiris.plugin.storage;

import java.util.Properties;

import javax.servlet.http.HttpServletRequest;

public interface StorageAndCache {
    /**
     * Initializes the storage and cache system. This method is called before any call to other methods.
     *
     * @param request The servlet request.
     * @param config WIRIS plugin configuration loaded from configuration.ini.
     */
    public void init(HttpServletRequest request, Properties config);

    /**
     * Given a content, computes a digest of it. This digest must be unique in order to use it as identifier of the content.
     * For example, the digest can be the md5 sum of the content.
     *
     * @param content
     * @return A computed digest of the content.
     */
    public String codeDigest(String content);

    /**
     * Given a computed digest, returns the respective content.
     * You might need to store the relation digest content during the {@link #codeDigest}.
     *
     * @param digest A computed digest.
     * @return The content associated to the computed digest. If it is not found, this method should return null.
     */
    public String decodeDigest(String digest);

    /**
     * Given a computed digest, returns the stored data associated with it.
     * This should be a cache system. As a cache there is a contract between the implementation and the caller:
     * <ul>
     *  <li>If the data is not found, the caller is responsible to regenerate and store the data.</li>
     *  <li>The cache is free to remove any data</li>
     * </ul>
     *
     * @param digest A computed digest.
     * @return The data associated with the digest. If it is not found, this method should return null.
     */
    public byte[] retreiveData(String digest);

    /**
     * Associates a data stream with a computed digest.
     *
     * @param digest A computed digest.
     * @param stream The data to be stored.
     */
    public void storeData(String digest, byte[] stream);
}
