/*
 * Decompiled with CFR 0.152.
 */
package com.warrenstrange.googleauth;

import com.warrenstrange.googleauth.GoogleAuthenticatorConfig;
import com.warrenstrange.googleauth.GoogleAuthenticatorException;
import com.warrenstrange.googleauth.GoogleAuthenticatorKey;
import com.warrenstrange.googleauth.ICredentialRepository;
import com.warrenstrange.googleauth.IGoogleAuthenticator;
import com.warrenstrange.googleauth.ReseedingSecureRandom;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.ServiceLoader;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.apache.commons.codec.binary.Base32;
import org.apache.commons.codec.binary.Base64;

public final class GoogleAuthenticator
implements IGoogleAuthenticator {
    private static final Logger LOGGER = Logger.getLogger(GoogleAuthenticator.class.getName());
    private static final int SECRET_BITS = 80;
    private static final int SCRATCH_CODES = 5;
    private static final int SCRATCH_CODE_LENGTH = 8;
    public static final int SCRATCH_CODE_MODULUS = (int)Math.pow(10.0, 8.0);
    private static final int SCRATCH_CODE_INVALID = -1;
    private static final int BYTES_PER_SCRATCH_CODE = 4;
    private static final String RANDOM_NUMBER_ALGORITHM = "SHA1PRNG";
    private static final String RANDOM_NUMBER_ALGORITHM_PROVIDER = "SUN";
    private static final String HMAC_HASH_FUNCTION = "HmacSHA1";
    private final GoogleAuthenticatorConfig config;
    private ReseedingSecureRandom secureRandom = new ReseedingSecureRandom("SHA1PRNG", "SUN");

    public GoogleAuthenticator() {
        this.config = new GoogleAuthenticatorConfig();
    }

    public GoogleAuthenticator(GoogleAuthenticatorConfig config) {
        if (config == null) {
            throw new IllegalArgumentException("Configuration cannot be null.");
        }
        this.config = config;
    }

    int calculateCode(byte[] key, long tm) {
        byte[] data = new byte[8];
        long value = tm;
        int i = 8;
        while (i-- > 0) {
            data[i] = (byte)value;
            value >>>= 8;
        }
        SecretKeySpec signKey = new SecretKeySpec(key, HMAC_HASH_FUNCTION);
        try {
            Mac mac = Mac.getInstance(HMAC_HASH_FUNCTION);
            mac.init(signKey);
            byte[] hash = mac.doFinal(data);
            int offset = hash[hash.length - 1] & 0xF;
            long truncatedHash = 0L;
            for (int i2 = 0; i2 < 4; ++i2) {
                truncatedHash <<= 8;
                truncatedHash |= (long)(hash[offset + i2] & 0xFF);
            }
            truncatedHash &= Integer.MAX_VALUE;
            return (int)(truncatedHash %= (long)this.config.getKeyModulus());
        }
        catch (InvalidKeyException | NoSuchAlgorithmException ex) {
            LOGGER.log(Level.SEVERE, ex.getMessage(), ex);
            throw new GoogleAuthenticatorException("The operation cannot be performed now.");
        }
    }

    private boolean checkCode(String secret, long code, long timestamp, int window) {
        byte[] decodedKey;
        switch (this.config.getKeyRepresentation()) {
            case BASE32: {
                Base32 codec32 = new Base32();
                decodedKey = codec32.decode(secret);
                break;
            }
            case BASE64: {
                Base64 codec64 = new Base64();
                decodedKey = codec64.decode(secret);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown key representation type.");
            }
        }
        long timeWindow = timestamp / this.config.getTimeStepSizeInMillis();
        for (int i = -((window - 1) / 2); i <= window / 2; ++i) {
            long hash = this.calculateCode(decodedKey, timeWindow + (long)i);
            if (hash != code) continue;
            return true;
        }
        return false;
    }

    @Override
    public GoogleAuthenticatorKey createCredentials() {
        byte[] buffer = new byte[30];
        this.secureRandom.nextBytes(buffer);
        byte[] secretKey = Arrays.copyOf(buffer, 10);
        String generatedKey = this.calculateSecretKey(secretKey);
        int validationCode = this.calculateValidationCode(secretKey);
        List<Integer> scratchCodes = this.calculateScratchCodes(buffer);
        return new GoogleAuthenticatorKey(generatedKey, validationCode, scratchCodes);
    }

    @Override
    public GoogleAuthenticatorKey createCredentials(String userName) {
        if (userName == null) {
            throw new IllegalArgumentException("User name cannot be null.");
        }
        GoogleAuthenticatorKey key = this.createCredentials();
        ICredentialRepository repository = this.getValidCredentialRepository();
        repository.saveUserCredentials(userName, key.getKey(), key.getVerificationCode(), key.getScratchCodes());
        return key;
    }

    private List<Integer> calculateScratchCodes(byte[] buffer) {
        ArrayList<Integer> scratchCodes = new ArrayList<Integer>();
        while (scratchCodes.size() < 5) {
            byte[] scratchCodeBuffer = Arrays.copyOfRange(buffer, 10 + 4 * scratchCodes.size(), 10 + 4 * scratchCodes.size() + 4);
            int scratchCode = this.calculateScratchCode(scratchCodeBuffer);
            if (scratchCode != -1) {
                scratchCodes.add(scratchCode);
                continue;
            }
            scratchCodes.add(this.generateScratchCode());
        }
        return scratchCodes;
    }

    private int calculateScratchCode(byte[] scratchCodeBuffer) {
        if (scratchCodeBuffer.length < 4) {
            throw new IllegalArgumentException(String.format("The provided random byte buffer is too small: %d.", scratchCodeBuffer.length));
        }
        int scratchCode = 0;
        for (int i = 0; i < 4; ++i) {
            scratchCode <<= 8;
            scratchCode += scratchCodeBuffer[i];
        }
        if (this.validateScratchCode(scratchCode = (scratchCode & Integer.MAX_VALUE) % SCRATCH_CODE_MODULUS)) {
            return scratchCode;
        }
        return -1;
    }

    boolean validateScratchCode(int scratchCode) {
        return scratchCode >= SCRATCH_CODE_MODULUS / 10;
    }

    private int generateScratchCode() {
        byte[] scratchCodeBuffer;
        int scratchCode;
        do {
            scratchCodeBuffer = new byte[4];
            this.secureRandom.nextBytes(scratchCodeBuffer);
        } while ((scratchCode = this.calculateScratchCode(scratchCodeBuffer)) == -1);
        return scratchCode;
    }

    private int calculateValidationCode(byte[] secretKey) {
        return this.calculateCode(secretKey, 0L);
    }

    private String calculateSecretKey(byte[] secretKey) {
        byte[] encodedKey;
        switch (this.config.getKeyRepresentation()) {
            case BASE32: {
                Base32 codec = new Base32();
                encodedKey = codec.encode(secretKey);
                break;
            }
            case BASE64: {
                Base64 codec64 = new Base64();
                encodedKey = codec64.encode(secretKey);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown key representation type.");
            }
        }
        return new String(encodedKey);
    }

    @Override
    public boolean authorize(String secret, int verificationCode) throws GoogleAuthenticatorException {
        if (secret == null) {
            throw new IllegalArgumentException("Secret cannot be null.");
        }
        if (verificationCode <= 0 || verificationCode >= this.config.getKeyModulus()) {
            return false;
        }
        return this.checkCode(secret, verificationCode, new Date().getTime(), this.config.getWindowSize());
    }

    @Override
    public boolean authorizeUser(String userName, int verificationCode) throws GoogleAuthenticatorException {
        ICredentialRepository repository = this.getValidCredentialRepository();
        return this.authorize(repository.getSecretKey(userName), verificationCode);
    }

    private ICredentialRepository getValidCredentialRepository() {
        ICredentialRepository repository = this.getCredentialRepository();
        if (repository == null) {
            throw new UnsupportedOperationException(String.format("An instance of the %s service must be configured in order to use this feature.", ICredentialRepository.class.getName()));
        }
        return repository;
    }

    private ICredentialRepository getCredentialRepository() {
        ServiceLoader<ICredentialRepository> loader = ServiceLoader.load(ICredentialRepository.class);
        Iterator<ICredentialRepository> iterator = loader.iterator();
        if (iterator.hasNext()) {
            ICredentialRepository repository = iterator.next();
            return repository;
        }
        return null;
    }
}

