/**
 * Copyright (C) 2022 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See <https://vaadin.com/commercial-license-and-service-terms> for the full
 * license.
 */
package com.vaadin.mpr.core.client;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vaadin.client.BrowserInfo;
import com.vaadin.client.ComponentConnector;
import com.vaadin.client.ConnectorHierarchyChangeEvent;
import com.vaadin.client.VTooltip;
import com.vaadin.client.ValueMap;
import com.vaadin.client.communication.MessageHandler;
import com.vaadin.client.ui.AbstractHasComponentsConnector;
import com.vaadin.client.ui.VUI;
import com.vaadin.shared.Connector;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.dom.client.Element;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.AbsolutePanel;
import com.google.gwt.user.client.ui.Panel;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.Widget;

/**
 * Legacy component wrapper connector. Handles server-client communication using
 * Flow communication.
 */
public abstract class AbstractMprUIContentConnector
        extends AbstractHasComponentsConnector {

    private Scheduler.ScheduledCommand pendingResponseQueueCommand;

    @Override
    protected void init() {
        super.init();
        redirectKeyDownEvents();
        handleTooltipHideEvents();
    }

    /**
     * Get the content state component settings.
     *
     * @return component settings map
     */
    protected abstract HashMap<Connector, ComponentSettings> getComponentSettings();

    private native void addResponseHandler(Element element)
    /*-{
        var that = this;
        element.setResponse = element.setResponse || $entry(function (response) {
            that.@AbstractMprUIContentConnector::setResponse(*)(response);
        });
    }-*/;

    /**
     * Handle the response message for this UI content that is received from
     * Flow.
     *
     * @param response
     *            Received response message
     */
    protected void setResponse(String response) {
        // Inspired by XhrResponseHandler.onResponseReceived
        String responseText = "for(;;);[{" + response + "}]";

        ValueMap json = MessageHandler.parseWrappedJson(responseText);
        if (json == null) {
            throw new RuntimeException(
                    "No Json was parsed from the response String.");
        }

        getConnection().getMessageHandler().handleMessage(json);
    }

    @Override
    public SimplePanel getWidget() {
        return (SimplePanel) super.getWidget();
    }

    @Override
    public void updateCaption(ComponentConnector connector) {
        // Ignore child captions
    }

    @Override
    public void onConnectorHierarchyChange(
            ConnectorHierarchyChangeEvent event) {
        Map<Connector, ComponentSettings> settings = getComponentSettings();

        for (ComponentConnector oldChild : event.getOldChildren()) {
            if (!settings.containsKey(oldChild)
                    && oldChild.getWidget().getParent() == null) {
                oldChild.getWidget().removeFromParent();
            }
        }

        List<ComponentConnector> childComponents = getChildComponents();
        for (ComponentConnector currentChild : childComponents) {
            if (currentChild.getWidget().getParent() == null) {
                ComponentSettings componentSettings = settings
                        .get(currentChild);
                assert componentSettings != null;

                attachWidget(currentChild.getWidget(),
                        componentSettings.getAppId(),
                        componentSettings.getNodeId());
            }
        }
    }

    private void redirectKeyDownEvents() {
        // Redirects all keydown events on the page to the UI - this is needed
        // for shortcuts to work
        RootPanel.get().addDomHandler(new KeyDownHandler() {
            @Override
            public void onKeyDown(KeyDownEvent event) {
                VUI vui = getConnection().getUIConnector().getWidget();
                if (vui.actionHandler != null) {
                    vui.actionHandler.handleKeyboardEvent(
                            (Event) event.getNativeEvent().cast());
                }
            }
        }, KeyDownEvent.getType());
        RootPanel.get().sinkEvents(Event.ONKEYDOWN);
    }

    private void handleTooltipHideEvents() {
        // MPR legacy component wrapper doesn't wrap the component with
        // panel/layouts with 'hasTooltip = true', like it's done in the
        // plain FW 7,8 applications - there is always a vertical layout
        // container around the components and it handles mouse move events
        // and hides the tooltip properly.
        // For MPR apps, a tooltip mouse move handler should be added explicitly
        // to ensure a tooltip is being hidden properly.
        getWidget().addDomHandler(new MouseMoveHandler() {
            @Override
            public void onMouseMove(MouseMoveEvent event) {
                VTooltip tooltip = getConnection().getVTooltip();
                handleMouseMoveEvent(event, tooltip);
            }
        }, MouseMoveEvent.getType());
    }

    private native void handleMouseMoveEvent(MouseMoveEvent event, VTooltip tooltip)
    /*-{
        var handler = tooltip.@com.vaadin.client.VTooltip::tooltipEventHandler;
        handler.@com.vaadin.client.VTooltip.TooltipEventHandler::onMouseMove(Lcom/google/gwt/event/dom/client/MouseMoveEvent;)(event);
    }-*/;

    private void attachWidget(final Widget widget, final String appId,
            final int nodeId) {
        Element wrapperElement = getWrapperElement(appId, nodeId);
        if (wrapperElement == null) {
            addDomBindingListener(appId, nodeId, new Runnable() {
                @Override
                public void run() {
                    doAttachWidget(widget, appId, nodeId);
                }
            });
        } else {
            doAttachWidget(widget, appId, nodeId);
        }
        handleResponseMessage();
    }

    private void doAttachWidget(Widget widget, String appId, int nodeId) {
        Element wrapperElement = getWrapperElement(appId, nodeId);
        Panel wrapper = new AbsolutePanel(wrapperElement) {
            {
                onAttach();
            }
        };

        wrapper.add(widget);
        addResizeListener(wrapperElement, BrowserInfo.get().isIE(), appId);
        layout();
    }

    private void handleResponseMessage() {
        /*
         * By the time the first widget is attached based on the initial UIDL,
         * any pending push messages can be processed by the client side. Since
         * there might still be other widgets to be attached, the queue is only
         * purged as deferred so that any pending tasks have been executed.
         */
        if (pendingResponsesExist() && pendingResponseQueueCommand == null) {
            // needs to be anonymous class due to gwt compilation for v7
            pendingResponseQueueCommand = new Scheduler.ScheduledCommand() {
                @Override
                public void execute() {
                    addResponseHandler(getConnection().getUIConnector().getWidget().getElement());
                    pendingResponseQueueCommand = null;
                    processPendingServerResponses();
                }
            };
            Scheduler.get().scheduleDeferred(pendingResponseQueueCommand);
        } else {
            // response handler function is only exposed after it is certain
            // that there are no pending responses
            addResponseHandler(getConnection().getUIConnector().getWidget().getElement());
        }
    }

    /**
     * Adds a resize listener to the given element, so as to trigger
     * re-layouting of this element on resize event.
     * <p>
     * Uses a polyfill for making ResizeObserver work in IE11. Polyfill version:
     * 1.5.1 Repository:
     * https://github.com/que-etc/resize-observer-polyfill/blob/master/dist/ResizeObserver.js
     * <p>
     * The polyfill source code has been changed to use <code>window</code>
     * instead of <code>this</code> as the parameter of the executed function
     * <code>function (global, factory) { ... }</code>, in order to provide
     * visibility of ResizeObserver.
     *
     * @param wrapperElement
     *            element to be re-layouted
     * @param isIE
     *            true if the browser is Internet Explorer 11
     * @param appId
     *            flow application Id
     */
    private native void addResizeListener(Element wrapperElement, boolean isIE,
            String appId)
    /*-{
        var that = this;
        var addResizeListenerHook = function() {
            new $wnd.ResizeObserver(that.@AbstractMprUIContentConnector::layout())
                .observe(wrapperElement);
        }
        if (isIE && !$wnd.ResizeObserver) {
            var polyfillUrl = $wnd.Vaadin.Flow.clients[appId].resolveUri(
                'context://framework/VAADIN/ResizeObserver-1.5.1.min.js');
            var polyfillScript = $doc.createElement('script');
            polyfillScript.async = false;
            polyfillScript.src = polyfillUrl;
            polyfillScript.type = 'text/javascript';
            polyfillScript.onload = addResizeListenerHook;
            $doc.head.appendChild(polyfillScript);
        } else {
            addResizeListenerHook();
        }
    }-*/;

    private native Element getWrapperElement(String appId, int nodeId)
    /*-{
        return $wnd.Vaadin.Flow.clients[appId].getByNodeId(nodeId);
    }-*/;

    private native void addDomBindingListener(String appId, int nodeId,
            Runnable runnable)
    /*-{
         var client = $wnd.Vaadin.Flow.clients[appId];
         if (client.addDomBindingListener){
             client.addDomBindingListener(nodeId , function(){
                 runnable.@java.lang.Runnable::run(*)();
             });
         }
    }-*/;

    private native void layout()
    /*-{
        requestAnimationFrame(function() {
            $wnd.vaadin.forceLayout();
        });
    }-*/;

    /**
     * Processes the pending UIDL responses from server (if any), that come
     * before the AppWidgetset is loaded and initialised.
     * <p>
     * These pending responses may appear in case of Push requests, triggered on
     * the server.
     */
    private void processPendingServerResponses() {
        if (pendingResponsesExist()) {
            Map<Connector, ComponentSettings> componentSettings = getComponentSettings();
            for (ComponentSettings settings : componentSettings.values()) {
                // Iterate through all the child components, having their own
                // Flow UI and appId, and process the responses from them.
                String appId = settings.getAppId();
                processPendingResponsesForApplication(appId);
            }
        }
    }

    private native boolean pendingResponsesExist()
    /*-{
        var responses = $wnd.Vaadin.Flow.pendingResponses;
        return responses && Object.keys(responses).length > 0;
    }-*/;

    private native void processPendingResponsesForApplication(String appId)
    /*-{
        var that = this;
        var responses = $wnd.Vaadin.Flow.pendingResponses;
        var responsesForAppId = responses[appId];
        if (responsesForAppId) {
            for (var i = 0; i < responsesForAppId.length; i++) {
                that.@AbstractMprUIContentConnector::setResponse(*)(responsesForAppId[i]);
            }
            $wnd.Vaadin.Flow.pendingResponses[appId] = [];
        }
    }-*/;
}
