/**
 * Copyright (C) 2020 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Developer License
 * 4.0 (CVDLv4).
 *
 *
 * For the full License, see <https://vaadin.com/license/cvdl-4.0>.
 */
package com.vaadin.mpr.core.client;

import com.google.gwt.dom.client.Style;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Element;

import com.vaadin.client.VLoadingIndicator;

/**
 * Custom MPR loading indicator that links the Legacy Vaadin and Platform Vaadin
 * loading indicators, so that the loading indicator is show as long as one of
 * them is still loading.
 */
public class MprLoadingIndicator extends VLoadingIndicator {

    private boolean shouldBeVisible;
    private Element indicator;

    @Override
    public void show() {
        shouldBeVisible = true;
        super.show();
    }

    @Override
    public void hide() {
        shouldBeVisible = false;
        super.hide();
    }

    @Override
    public boolean isVisible() {
        return shouldBeVisible;
    }

    /**
     * Finds the loading indicator used by Flow, and registers a
     * MutationObserver to it to track style changes. If Flow sets the indicator
     * to be invisible while FW is still loading, the indicator is reset back to
     * visible.
     * 
     * @return the loading indicator used by Flow, or <code>null</code> if it
     *         can not be found.
     */
    private final native Element findFlowLoadingIndicator()
    /*-{
        var divs = $doc.querySelectorAll('body > div.v-loading-indicator');
        if (!divs || divs.length === 0){
            console.warn("Couldn't find the Flow loading indicator. Using an empty one.");
            return null;
        }
        var indicator = divs[0];
        var thisClass = this;
        var observer = new MutationObserver(function (event) {
             var shouldBeVisible = thisClass.@com.vaadin.mpr.core.client.MprLoadingIndicator::shouldBeVisible;
             if (shouldBeVisible && indicator.style.display === 'none'){
                 indicator.style.display = 'block';
             } 
        });
        
        observer.observe(indicator, {
          attributes: true, 
          attributeFilter: ['style'],
          childList: false, 
          characterData: false
        });
        
        return indicator;
    }-*/;

    @Override
    public com.google.gwt.user.client.Element getElement() {
        if (indicator == null) {
            indicator = findFlowLoadingIndicator();
            if (indicator == null) {
                indicator = DOM.createDiv();
                getConnection().getUIConnector().getWidget().getElement()
                        .appendChild(indicator);
                indicator.getStyle().setHeight(0.1, Style.Unit.PX);
            }
        }
        return indicator;
    }
}
