/**
 * Copyright (C) 2018 Vaadin Ltd
 *
 * This program is available under Commercial Vaadin Add-On License 3.0
 * (CVALv3).
 *
 * See the file licensing.txt distributed with this software for more
 * information about licensing.
 *
 * You should have received a copy of the license along with this program.
 * If not, see <http://vaadin.com/license/cval-3>.
 */
package com.vaadin.mpr.core.client;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.gwt.dom.client.Element;
import com.google.gwt.event.dom.client.KeyDownEvent;
import com.google.gwt.event.dom.client.KeyDownHandler;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.ui.AbsolutePanel;
import com.google.gwt.user.client.ui.Panel;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.Widget;
import com.vaadin.client.ComponentConnector;
import com.vaadin.client.ConnectorHierarchyChangeEvent;
import com.vaadin.client.ValueMap;
import com.vaadin.client.communication.MessageHandler;
import com.vaadin.client.ui.AbstractHasComponentsConnector;
import com.vaadin.client.ui.VUI;
import com.vaadin.shared.Connector;

/**
 * Legacy component wrapper connector. Handles server-client communication using
 * Flow communication.
 */
public abstract class AbstractMprUIContentConnector
        extends AbstractHasComponentsConnector {

    @Override
    protected void init() {
        super.init();
        addFunction(getConnection().getUIConnector().getWidget().getElement());

        // Redirects all keydown events on the page to the UI - this is needed
        // for shortcuts to work
        RootPanel.get().addDomHandler(new KeyDownHandler() {
            @Override
            public void onKeyDown(KeyDownEvent event) {
                VUI vui = getConnection().getUIConnector().getWidget();
                if (vui.actionHandler != null) {
                    vui.actionHandler.handleKeyboardEvent(
                            (Event) event.getNativeEvent().cast());
                }
            }
        }, KeyDownEvent.getType());
        RootPanel.get().sinkEvents(Event.ONKEYDOWN);
    }

    /**
     * Get the content state component settings.
     *
     * @return component settings map
     */
    protected abstract HashMap<Connector, ComponentSettings> getComponentSettings();

    private native void addFunction(Element element)
    /*-{
        var that = this;
        element.setResponse = $entry(function (response) {
            that.@AbstractMprUIContentConnector::setResponse(*)(response);
        });
    }-*/;

    /**
     * Handle the response message for this UI content that is received from
     * Flow.
     *
     * @param response
     *         Received response message
     */
    protected void setResponse(String response) {
        // Inspired by XhrResponseHandler.onResponseReceived
        String responseText = "for(;;);[{" + response + "}]";

        ValueMap json = MessageHandler.parseWrappedJson(responseText);
        if (json == null) {
            throw new RuntimeException(
                    "No Json was parsed from the response String.");
        }

        getConnection().getMessageHandler().handleMessage(json);
    }

    @Override
    public SimplePanel getWidget() {
        return (SimplePanel) super.getWidget();
    }

    @Override
    public void updateCaption(ComponentConnector connector) {
        // Ignore child captions
    }

    @Override
    public void onConnectorHierarchyChange(
            ConnectorHierarchyChangeEvent event) {
        Map<Connector, ComponentSettings> settings = getComponentSettings();

        for (ComponentConnector oldChild : event.getOldChildren()) {
            if (!settings.containsKey(oldChild)
                    && oldChild.getWidget().getParent() == null) {
                oldChild.getWidget().removeFromParent();
            }
        }

        List<ComponentConnector> childComponents = getChildComponents();
        for (ComponentConnector currentChild : childComponents) {
            if (currentChild.getWidget().getParent() == null) {
                ComponentSettings componentSettings = settings
                        .get(currentChild);
                assert componentSettings != null;

                attachWidget(currentChild.getWidget(),
                        componentSettings.getAppId(),
                        componentSettings.getNodeId());
            }
        }
    }

    private void attachWidget(Widget widget, String appId, int nodeId) {
        Element wrapperElement = getWrapperElement(appId, nodeId);

        Panel wrapper = new AbsolutePanel(wrapperElement) {
            {
                onAttach();
            }
        };

        wrapper.add(widget);
    }

    private native Element getWrapperElement(String appId, int nodeId)
    /*-{
        return $wnd.Vaadin.Flow.clients[appId].getByNodeId(nodeId);
    }-*/;

}
