/*
 * Copyright 2000-2023 Vaadin Ltd.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.vaadin.flow.server.frontend;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.WebComponentExporter;
import com.vaadin.flow.component.WebComponentExporterFactory;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.internal.StringUtil;
import com.vaadin.flow.internal.UsageStatistics;
import com.vaadin.flow.server.Constants;
import com.vaadin.flow.server.ExecutionFailedException;
import com.vaadin.flow.server.frontend.scanner.ClassFinder;
import com.vaadin.flow.server.frontend.scanner.FrontendDependenciesScanner;
import com.vaadin.flow.server.webcomponent.WebComponentExporterTagExtractor;
import com.vaadin.flow.server.webcomponent.WebComponentExporterUtils;

import com.vaadin.flow.shared.util.SharedUtil;
import com.vaadin.flow.theme.ThemeDefinition;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import elemental.json.JsonType;
import elemental.json.JsonValue;

import static com.vaadin.flow.server.Constants.DEV_BUNDLE_JAR_PATH;

/**
 * Compiles the dev mode bundle if it is out of date.
 * <p>
 * Only used when running in dev mode without a dev server.
 * <p>
 * For internal use only. May be renamed or removed in a future release.
 */
public class TaskRunDevBundleBuild implements FallibleCommand {

    private static final Pattern THEME_PATH_PATTERN = Pattern
            .compile("themes\\/([\\s\\S]+?)\\/theme.json");

    //@formatter:off
    public static final String README =
            "This directory is automatically generated by Vaadin and contains the pre-compiled \n" +
            "frontend files/resources for your project (frontend development bundle).\n\n" +
            "It should be added to Version Control System and committed, so that other developers\n" +
            "do not have to compile it again.\n\n" +
            "Frontend development bundle is automatically updated when needed:\n" +
            "- an npm/pnpm package is added with @NpmPackage or directly into package.json\n" +
            "- CSS, JavaScript or TypeScript files are added with @CssImport, @JsModule or @JavaScript\n" +
            "- Vaadin add-on with front-end customizations is added\n" +
            "- Custom theme imports/assets added into 'theme.json' file\n" +
            "- Exported web component is added.\n\n" +
            "If your project development needs a hot deployment of the frontend changes, \n" +
            "you can switch Flow to use Vite development server (default in Vaadin 23.3 and earlier versions):\n" +
            "- set `vaadin.frontend.hotdeploy=true` in `application.properties`\n" +
            "- configure `vaadin-maven-plugin`:\n" +
            "```\n" +
            "   <configuration>\n" +
            "       <frontendHotdeploy>true</frontendHotdeploy>\n" +
            "   </configuration>\n" +
            "```\n" +
            "- configure `jetty-maven-plugin`:\n" +
            "```\n" +
            "   <configuration>\n" +
            "       <systemProperties>\n" +
            "           <vaadin.frontend.hotdeploy>true</vaadin.frontend.hotdeploy>\n" +
            "       </systemProperties>\n" +
            "   </configuration>\n" +
            "```\n\n" +
            "Read more [about Vaadin development mode](https://vaadin.com/docs/next/configuration/development-mode/#pre-compiled-front-end-bundle-for-faster-start-up).";
    //@formatter:on

    public static final String README_NOT_CREATED = "Failed to create a README file in "
            + Constants.DEV_BUNDLE_LOCATION;

    private final Options options;

    /**
     * Create an instance of the command.
     *
     * @param options
     *            the task options
     */
    TaskRunDevBundleBuild(Options options) {
        this.options = options;
    }

    @Override
    public void execute() throws ExecutionFailedException {
        getLogger().info(
                "Creating a new development mode bundle. This can take a while but will only run when the project setup is changed, addons are added or frontend files are modified");

        runFrontendBuildTool("Vite", "vite/bin/vite.js", Collections.emptyMap(),
                "build");

        addReadme();
    }

    public static boolean needsBuild(Options options,
            FrontendDependenciesScanner frontendDependencies,
            ClassFinder finder) {
        getLogger()
                .info("Checking if a development mode bundle build is needed");

        try {
            boolean needsBuild = needsBuildInternal(options,
                    frontendDependencies, finder);
            if (needsBuild) {
                getLogger().info("A development mode bundle build is needed");
            } else {
                getLogger()
                        .info("A development mode bundle build is not needed");
            }
            return needsBuild;
        } catch (Exception e) {
            getLogger().error(
                    "Error when checking if a development bundle build is needed",
                    e);
            return true;
        }
    }

    protected static boolean needsBuildInternal(Options options,
            FrontendDependenciesScanner frontendDependencies,
            ClassFinder finder) throws IOException {
        File npmFolder = options.getNpmFolder();

        if (!FrontendUtils.getDevBundleFolder(npmFolder).exists()
                && !hasJarBundle()) {
            getLogger().info("No dev-bundle found.");
            return true;
        }

        String statsJsonContent = FrontendUtils.findBundleStatsJson(npmFolder);
        if (statsJsonContent == null) {
            // without stats.json in bundle we can not say if it is up to date
            getLogger().info("No dev-bundle stats.json found for validation.");
            return true;
        }

        JsonObject packageJson = getPackageJson(options, frontendDependencies,
                finder);
        JsonObject statsJson = Json.parse(statsJsonContent);

        // Get scanned @NpmPackage annotations
        final Map<String, String> npmPackages = frontendDependencies
                .getPackages();

        if (!hashAndBundleModulesEqual(statsJson, packageJson, npmPackages)) {
            UsageStatistics.markAsUsed("flow/rebundle-reason-missing-package",
                    null);
            // Hash in the project doesn't match the bundle hash or NpmPackages
            // are found missing in bundle.
            return true;
        }
        if (!frontendImportsFound(statsJson, options, finder,
                frontendDependencies)) {
            UsageStatistics.markAsUsed(
                    "flow/rebundle-reason-missing-frontend-import", null);
            return true;
        }

        if (themeConfigurationChanged(options, statsJson,
                frontendDependencies)) {
            UsageStatistics.markAsUsed(
                    "flow/rebundle-reason-changed-theme-config", null);
            return true;
        }

        if (exportedWebComponents(statsJson, finder)) {
            UsageStatistics.markAsUsed(
                    "flow/rebundle-reason-added-exported-component", null);
            return true;
        }

        return false;
    }

    private static boolean exportedWebComponents(JsonObject statsJson,
            ClassFinder finder) {
        try {
            Set<Class<?>> exporterRelatedClasses = new HashSet<>();
            finder.getSubTypesOf(WebComponentExporter.class.getName())
                    .forEach(exporterRelatedClasses::add);
            finder.getSubTypesOf(WebComponentExporterFactory.class.getName())
                    .forEach(exporterRelatedClasses::add);

            Set<String> webComponents = WebComponentExporterUtils
                    .getFactories(exporterRelatedClasses).stream()
                    .map(TaskRunDevBundleBuild::getTag)
                    .collect(Collectors.toSet());

            JsonArray webComponentsInStats = statsJson
                    .getArray("webComponents");

            if (webComponentsInStats == null) {
                if (!webComponents.isEmpty()) {
                    getLogger().info(
                            "Found embedded web components not yet included "
                                    + "into the dev bundle: {}",
                            String.join(", ", webComponents));
                    return true;
                }
                return false;
            } else {
                for (int index = 0; index < webComponentsInStats
                        .length(); index++) {
                    String webComponentInStats = webComponentsInStats
                            .getString(index);
                    webComponents.remove(webComponentInStats);
                }
            }

            if (!webComponents.isEmpty()) {
                getLogger().info(
                        "Found newly added embedded web components not "
                                + "yet included into the dev bundle: {}",
                        String.join(", ", webComponents));
                return true;
            }

            return false;

        } catch (ClassNotFoundException e) {
            getLogger()
                    .error("Unable to locate embedded web component classes.");
            return false;
        }
    }

    private static boolean themeConfigurationChanged(Options options,
            JsonObject statsJson,
            FrontendDependenciesScanner frontendDependencies)
            throws IOException {
        Map<String, JsonObject> themeJsonContents = new HashMap<>();

        if (options.getJarFiles() != null) {
            options.getJarFiles().stream().filter(File::exists)
                    .filter(file -> !file.isDirectory())
                    .forEach(jarFile -> getPackagedThemeJsonContents(jarFile,
                            themeJsonContents));
        }

        ThemeDefinition themeDefinition = frontendDependencies
                .getThemeDefinition();
        Optional<JsonObject> projectThemeJson = FrontendUtils
                .getThemeJsonInFrontend(options, themeDefinition);

        JsonObject contentsInStats = statsJson.getObject("themeJsonContents");
        if (contentsInStats == null && (!themeJsonContents.isEmpty()
                || projectThemeJson.isPresent())) {
            getLogger().info(
                    "Found newly added theme configurations in 'theme.json'.");
            return true;
        }

        if (projectThemeJson.isPresent()) {
            String projectThemeName = themeDefinition.getName();
            String key;
            if (contentsInStats.hasKey(projectThemeName)) {
                key = projectThemeName;
            } else if (contentsInStats.hasKey(Constants.DEV_BUNDLE_NAME)) {
                key = Constants.DEV_BUNDLE_NAME;
            } else {
                getLogger().info(
                        "Found newly added configuration for project theme '{}' in 'theme.json'.",
                        projectThemeName);
                return true;
            }

            List<String> missedKeys = new ArrayList<>();
            JsonObject themeJsonContent = Json
                    .parse(contentsInStats.getString(key));
            if (!objectIncludesEntry(themeJsonContent, projectThemeJson.get(),
                    missedKeys)) {
                getLogger().info(
                        "Project custom theme '{}' has imports/assets in 'theme.json' not present in the bundle",
                        projectThemeName);
                logMissedEntries(missedKeys);
                return true;
            }

            Optional<String> parentThemeInFrontend = FrontendUtils
                    .getParentThemeNameInFrontend(
                            options.getFrontendDirectory(),
                            projectThemeJson.get());
            if (parentThemeInFrontend.isPresent()) {
                String parentThemeName = parentThemeInFrontend.get();
                Optional<JsonObject> themeJson = FrontendUtils
                        .getThemeJsonInFrontend(options.getFrontendDirectory(),
                                parentThemeName);
                themeJson.ifPresent(jsonObject -> themeJsonContents
                        .put(parentThemeName, jsonObject));
            }
        }

        for (Map.Entry<String, JsonObject> themeContent : themeJsonContents
                .entrySet()) {
            if (!contentsInStats.hasKey(themeContent.getKey())) {
                getLogger().info(
                        "Found new configuration for theme '{}' in 'theme.json'.",
                        themeContent.getKey());
                return true;
            } else {
                List<String> missedKeys = new ArrayList<>();
                JsonObject content = Json.parse(
                        contentsInStats.getString(themeContent.getKey()));
                if (!objectIncludesEntry(content, themeContent.getValue(),
                        missedKeys)) {
                    getLogger().info(
                            "Parent theme '{}' has imports/assets in 'theme.json' not present in the bundle",
                            themeContent.getKey());
                    logMissedEntries(missedKeys);
                    return true;
                }
            }
        }

        return false;
    }

    private static boolean objectIncludesEntry(JsonValue jsonFromBundle,
            JsonValue projectJson, Collection<String> missedKeys) {
        JsonType bundleJsonType = jsonFromBundle.getType();
        JsonType projectJsonObjectTypeType = projectJson.getType();
        assert bundleJsonType.equals(projectJsonObjectTypeType);

        if (bundleJsonType == JsonType.NULL) {
            return true;
        } else if (bundleJsonType == JsonType.BOOLEAN) {
            return JsonUtils.booleanEqual(jsonFromBundle, projectJson);
        } else if (bundleJsonType == JsonType.NUMBER) {
            return JsonUtils.numbersEqual(jsonFromBundle, projectJson);
        } else if (bundleJsonType == JsonType.STRING) {
            return JsonUtils.stringEqual(jsonFromBundle, projectJson);
        } else if (bundleJsonType == JsonType.ARRAY) {
            JsonArray jsonArrayFromBundle = (JsonArray) jsonFromBundle;
            JsonArray projectJsonArray = (JsonArray) projectJson;
            return compareArrays(missedKeys, jsonArrayFromBundle,
                    projectJsonArray);
        } else if (bundleJsonType == JsonType.OBJECT) {
            JsonObject jsonObjectFromBundle = (JsonObject) jsonFromBundle;
            JsonObject projectJsonObject = (JsonObject) projectJson;
            return compareObjects(missedKeys, jsonObjectFromBundle,
                    projectJsonObject);
        } else {
            throw new IllegalArgumentException(
                    "Unsupported JsonType: " + bundleJsonType);
        }
    }

    private static boolean frontendImportsFound(JsonObject statsJson,
            Options options, ClassFinder finder,
            FrontendDependenciesScanner frontendDependencies)
            throws IOException {

        // Validate frontend requirements in flow-generated-imports.js
        final GenerateMainImports generateMainImports = new GenerateMainImports(
                finder, frontendDependencies, options, statsJson);
        generateMainImports.run();
        final List<String> imports = generateMainImports.getLines().stream()
                .filter(line -> line.startsWith("import"))
                .map(line -> line.substring(line.indexOf('\'') + 1,
                        line.lastIndexOf('\'')))
                .map(importString -> importString.contains("?")
                        ? importString.substring(0,
                                importString.lastIndexOf("?"))
                        : importString)
                .collect(Collectors.toList());
        JsonArray statsBundle = statsJson.hasKey("bundleImports")
                ? statsJson.getArray("bundleImports")
                : Json.createArray();
        final List<String> missingFromBundle = imports.stream().filter(
                importString -> !arrayContainsString(statsBundle, importString))
                .collect(Collectors.toList());

        if (!missingFromBundle.isEmpty()) {
            for (String dependency : missingFromBundle) {
                getLogger().info("Frontend import " + dependency
                        + " is missing from the bundle");
            }
            return false;
        }

        String resourcePath = "generated/jar-resources/";
        final List<String> jarImports = imports.stream()
                .filter(importString -> importString.contains(resourcePath))
                .map(importString -> importString
                        .substring(importString.indexOf(resourcePath)
                                + resourcePath.length()))
                .collect(Collectors.toList());

        final List<String> projectImports = imports.stream()
                .filter(importString -> importString
                        .startsWith(FrontendUtils.FRONTEND_FOLDER_ALIAS)
                        && !importString.contains(resourcePath))
                .map(importString -> importString.substring(
                        FrontendUtils.FRONTEND_FOLDER_ALIAS.length()))
                .collect(Collectors.toList());

        final JsonObject frontendHashes = statsJson.getObject("frontendHashes");
        List<String> faultyContent = new ArrayList<>();

        for (String jarImport : jarImports) {
            final String jarResourceString = FrontendUtils
                    .getJarResourceString(jarImport);
            if (jarResourceString == null) {
                getLogger().info("No file found for '{}'", jarImport);
                return false;
            }
            compareFrontendHashes(frontendHashes, faultyContent, jarImport,
                    jarResourceString);
        }

        for (String projectImport : projectImports) {
            File frontendFile = new File(options.getFrontendDirectory(),
                    projectImport);
            if (!frontendFile.exists()) {
                getLogger().info("No file found for '{}'", projectImport);
                return false;
            }
            String frontendFileContent = FileUtils
                    .readFileToString(frontendFile, StandardCharsets.UTF_8);
            compareFrontendHashes(frontendHashes, faultyContent, projectImport,
                    frontendFileContent);
        }

        if (!faultyContent.isEmpty()) {
            logChangedFiles(faultyContent,
                    "Detected changed content for frontend files:");
            return false;
        }

        if (indexFileAddedOrDeleted(options, frontendHashes)) {
            return false;
        }

        Map<String, String> remainingImports = getRemainingImports(jarImports,
                projectImports, frontendHashes);

        if (importedFrontendFilesChanged(options.getFrontendDirectory(),
                remainingImports)) {
            return false;
        }

        return true;
    }

    private static boolean indexFileAddedOrDeleted(Options options,
            JsonObject frontendHashes) {
        Collection<String> indexFiles = Arrays.asList(FrontendUtils.INDEX_TS,
                FrontendUtils.INDEX_JS, FrontendUtils.INDEX_TSX);
        for (String indexFile : indexFiles) {
            File file = new File(options.getFrontendDirectory(), indexFile);
            if (file.exists() && !frontendHashes.hasKey(indexFile)) {
                getLogger().info("Detected added {} file", indexFile);
                return true;
            } else if (!file.exists() && frontendHashes.hasKey(indexFile)) {
                getLogger().info("Detected deleted {} file", indexFile);
                return true;
            }
        }
        return false;
    }

    private static Map<String, String> getRemainingImports(
            List<String> jarImports, List<String> projectImports,
            JsonObject frontendHashes) {
        Map<String, String> remainingImportEntries = new HashMap<>();
        List<String> remainingKeys = new ArrayList<>(
                Arrays.asList(frontendHashes.keys()));

        remainingKeys.removeAll(jarImports);
        remainingKeys.removeAll(projectImports);

        if (!remainingKeys.isEmpty()) {
            for (String key : remainingKeys) {
                remainingImportEntries.put(key, frontendHashes.getString(key));
            }
            return remainingImportEntries;
        }

        return Collections.emptyMap();
    }

    /**
     * Checks for possible changes in TS/JS files in frontend folder which are
     * referenced from other TS/JS files with 'import {foo} from 'Frontend/bar'.
     *
     * @param frontendDirectory
     *            folder with frontend resources in the project
     * @param remainingImports
     *            frontend resource imports listed in the bundle, but not found
     *            with class scanner.
     * @return true if changes detected, false otherwise
     * @throws IOException
     *             if exception is thrown while reading files content
     */
    private static boolean importedFrontendFilesChanged(File frontendDirectory,
            Map<String, String> remainingImports) throws IOException {
        if (!remainingImports.isEmpty()) {
            List<String> changed = new ArrayList<>();
            for (Map.Entry<String, String> importEntry : remainingImports
                    .entrySet()) {
                String filePath = importEntry.getKey();
                String expectedHash = importEntry.getValue();
                File frontendFile = new File(frontendDirectory, filePath);
                if (frontendFile.exists()) {
                    final String hash = calculateHash(
                            FileUtils.readFileToString(frontendFile,
                                    StandardCharsets.UTF_8));
                    if (!expectedHash.equals(hash)) {
                        changed.add(filePath);
                    }
                }
            }
            if (!changed.isEmpty()) {
                logChangedFiles(changed, "Detected changed frontend files:");
                return true;
            }
        }
        return false;
    }

    private static void logChangedFiles(List<String> frontendFiles,
            String message) {
        if (message == null || message.isEmpty()) {
            throw new IllegalArgumentException(
                    "Changed files message cannot be empty");
        }
        if (message.contains("{}")) {
            throw new IllegalArgumentException(
                    "Changed files message shouldn't include '{}' placeholder");
        }
        message += "\n{}";
        StringBuilder handledFiles = new StringBuilder();
        for (String file : frontendFiles) {
            handledFiles.append(" - ").append(file).append("\n");
        }
        getLogger().info(message, handledFiles);
    }

    private static void logMissedEntries(List<String> missedKeys) {
        Collections.reverse(missedKeys);
        logChangedFiles(missedKeys, "Detected missed entries:");
    }

    private static void compareFrontendHashes(JsonObject frontendHashes,
            List<String> faultyContent, String frontendFilePath,
            String frontendFileContent) {
        final String contentHash = calculateHash(frontendFileContent);
        if (frontendHashes.hasKey(frontendFilePath) && !frontendHashes
                .getString(frontendFilePath).equals(contentHash)) {
            faultyContent.add(frontendFilePath);
        } else if (!frontendHashes.hasKey(frontendFilePath)) {
            getLogger().info("No hash info for '{}'", frontendFilePath);
            faultyContent.add(frontendFilePath);
        }
    }

    private static String calculateHash(String fileContent) {
        String content = fileContent.replaceAll("\\r\\n", "\n");
        return StringUtil.getHash(content, StandardCharsets.UTF_8);
    }

    private static boolean arrayContainsString(JsonArray array, String string) {
        string = string.replace("Frontend/", "./");
        for (int i = 0; i < array.length(); i++) {
            if (string.equals(array.getString(i).replace("Frontend/", "./"))) {
                return true;
            }
        }
        return false;
    }

    private static boolean hasJarBundle() {
        final URL resource = TaskRunDevBundleBuild.class.getClassLoader()
                .getResource(DEV_BUNDLE_JAR_PATH + "config/stats.json");
        return resource != null;
    }

    /**
     * Verify that package hash versions are equal and that all project
     * npmPackages are in bundle.
     *
     *
     * @param statsJson
     *            devBundle statsJson
     * @param packageJson
     *            packageJson
     * @param npmPackages
     *            npm packages map
     * @return {@code true} if up to date
     */
    private static boolean hashAndBundleModulesEqual(JsonObject statsJson,
            JsonObject packageJson, Map<String, String> npmPackages) {

        String packageJsonHash = getPackageJsonHash(packageJson);
        String bundlePackageJsonHash = getStatsHash(statsJson);

        if (packageJsonHash == null || packageJsonHash.isEmpty()) {
            getLogger().error(
                    "No hash found for 'package.json' even though one should always be generated!");
            return false;
        }

        JsonObject bundleModules = statsJson
                .getObject("packageJsonDependencies");

        if (bundleModules == null) {
            getLogger().error(
                    "Dev bundle did not contain package json dependencies to validate.\n"
                            + "Rebuild of bundle needed.");
            return false;
        }

        // Check that bundle modules contains all package dependencies
        if (packageJsonHash.equals(bundlePackageJsonHash)) {
            if (!dependenciesContainsAllPackages(npmPackages, bundleModules)) {
                return false;
            }
        }

        JsonObject dependencies = packageJson.getObject("dependencies");

        List<String> dependenciesList = Arrays.stream(dependencies.keys())
                // skip checking flow-frontend as it was used in previous
                // versions as an alias for ./target/flow-frontend
                .filter(pkg -> !"@vaadin/flow-frontend".equals(pkg))
                .collect(Collectors.toList());

        List<String> missingFromBundle = dependenciesList.stream()
                .filter(pkg -> !bundleModules.hasKey(pkg))
                .collect(Collectors.toList());

        if (!missingFromBundle.isEmpty()) {
            for (String dependency : missingFromBundle) {
                getLogger().info("Dependency " + dependency
                        + " is missing from the bundle");
            }
            return false;
        }

        // We know here that all dependencies exist
        missingFromBundle = dependenciesList.stream()
                .filter(pkg -> !versionAccepted(dependencies.getString(pkg),
                        bundleModules.getString(pkg)))
                .collect(Collectors.toList());

        if (!missingFromBundle.isEmpty()) {
            for (String pkg : missingFromBundle) {
                getLogger().info(
                        "Dependency {}:{} has the wrong version {} in the bundle",
                        pkg, dependencies.getString(pkg),
                        bundleModules.getString(pkg));
            }
            return false;
        }

        return true;
    }

    private static boolean versionAccepted(String expected, String actual) {
        FrontendVersion expectedVersion = new FrontendVersion(expected);
        FrontendVersion actualVersion = new FrontendVersion(actual);

        if (expected.startsWith("~")) {
            boolean correctRange = expectedVersion
                    .getMajorVersion() == actualVersion.getMajorVersion()
                    && expectedVersion.getMinorVersion() == actualVersion
                            .getMinorVersion();
            // Installed version may be newer than the package version.
            return (expectedVersion.isEqualTo(actualVersion)
                    || expectedVersion.isOlderThan(actualVersion))
                    && correctRange;
        } else if (expected.startsWith("^")) {
            boolean correctRange = expectedVersion
                    .getMajorVersion() == actualVersion.getMajorVersion();
            // Installed version may be newer than the package version.
            return (expectedVersion.isEqualTo(actualVersion)
                    || expectedVersion.isOlderThan(actualVersion))
                    && correctRange;
        }
        return expectedVersion.isEqualTo(actualVersion);
    }

    /**
     * Get the package.json file from disk if available else generate in memory.
     * <p>
     * For the loaded file update versions as per in memory to get correct
     * application versions.
     *
     * @param options
     *            the task options
     * @param frontendDependencies
     *            frontend dependency scanner
     * @param finder
     *            classfinder
     * @return package.json content as JsonObject
     */
    private static JsonObject getPackageJson(Options options,
            FrontendDependenciesScanner frontendDependencies,
            ClassFinder finder) {
        File packageJsonFile = new File(options.getNpmFolder(), "package.json");

        if (packageJsonFile.exists()) {
            try {
                final JsonObject packageJson = Json
                        .parse(FileUtils.readFileToString(packageJsonFile,
                                StandardCharsets.UTF_8));
                cleanOldPlatformDependencies(packageJson);
                return getDefaultPackageJson(options, frontendDependencies,
                        finder, packageJson);
            } catch (IOException e) {
                getLogger().warn("Failed to read package.json", e);
            }
        } else {
            return getDefaultPackageJson(options, frontendDependencies, finder,
                    null);
        }
        return null;
    }

    protected static JsonObject getDefaultPackageJson(Options options,
            FrontendDependenciesScanner frontendDependencies,
            ClassFinder finder, JsonObject packageJson) {
        NodeUpdater nodeUpdater = new NodeUpdater(finder, frontendDependencies,
                options) {
            @Override
            public void execute() {
            }
        };
        try {
            if (packageJson == null) {
                packageJson = nodeUpdater.getPackageJson();
            }
            nodeUpdater.addVaadinDefaultsToJson(packageJson);
            nodeUpdater.updateDefaultDependencies(packageJson);

            final Map<String, String> applicationDependencies = frontendDependencies
                    .getPackages();

            // Add application dependencies
            for (Map.Entry<String, String> dep : applicationDependencies
                    .entrySet()) {
                nodeUpdater.addDependency(packageJson, NodeUpdater.DEPENDENCIES,
                        dep.getKey(), dep.getValue());
            }

            final String hash = TaskUpdatePackages
                    .generatePackageJsonHash(packageJson);
            packageJson.getObject(NodeUpdater.VAADIN_DEP_KEY)
                    .put(NodeUpdater.HASH_KEY, hash);

            final JsonObject platformPinnedDependencies = nodeUpdater
                    .getPlatformPinnedDependencies();
            for (String key : platformPinnedDependencies.keys()) {
                // need to double check that not overriding a scanned
                // dependency since add-ons should be able to downgrade
                // version through exclusion
                if (!applicationDependencies.containsKey(key)) {
                    TaskUpdatePackages.pinPlatformDependency(packageJson,
                            platformPinnedDependencies, key);
                }
            }

            return packageJson;
        } catch (IOException e) {
            getLogger().warn("Failed to generate package.json", e);
        }
        return null;
    }

    private static String getStatsHash(JsonObject statsJson) {
        if (statsJson.hasKey("packageJsonHash")) {
            return statsJson.getString("packageJsonHash");
        }

        return null;
    }

    private static String getPackageJsonHash(JsonObject packageJson) {
        if (packageJson != null && packageJson.hasKey("vaadin")
                && packageJson.getObject("vaadin").hasKey("hash")) {
            return packageJson.getObject("vaadin").getString("hash");
        }

        return null;
    }

    /**
     * Check that all npmPackages are in the given dependency object.
     *
     * @param npmPackages
     *            {@code @NpmPackage} key-value map
     * @param dependencies
     *            json object containing dependencies to check against
     * @return {@code false} if all packages are found
     */
    private static boolean dependenciesContainsAllPackages(
            Map<String, String> npmPackages, JsonObject dependencies) {
        final List<String> collect = npmPackages.keySet().stream()
                .filter(pkg -> !(dependencies.hasKey(pkg) && versionAccepted(
                        dependencies.getString(pkg), npmPackages.get(pkg))))
                .collect(Collectors.toList());
        if (!collect.isEmpty()) {
            collect.forEach(dependency -> getLogger().info("Dependency "
                    + dependency + " is missing from the bundle"));
            return false;
        }
        return true;
    }

    private static void getPackagedThemeJsonContents(File jarFileToLookup,
            Map<String, JsonObject> packagedThemeHashes) {
        JarContentsManager jarContentsManager = new JarContentsManager();
        if (jarContentsManager.containsPath(jarFileToLookup,
                Constants.RESOURCES_THEME_JAR_DEFAULT)) {
            List<String> themeJsons = jarContentsManager.findFiles(
                    jarFileToLookup, Constants.RESOURCES_THEME_JAR_DEFAULT,
                    "theme.json");
            for (String themeJson : themeJsons) {
                byte[] byteContent = jarContentsManager
                        .getFileContents(jarFileToLookup, themeJson);
                String content = IOUtils.toString(byteContent, "UTF-8");
                content = content.replaceAll("\\r\\n", "\n");

                Matcher matcher = THEME_PATH_PATTERN.matcher(themeJson);
                if (!matcher.find()) {
                    throw new IllegalStateException(
                            "Packaged theme folders structure is incorrect, should have META-INF/resources/themes/[theme-name]/");
                }
                String themeName = matcher.group(1);
                JsonObject jsonContent = Json.parse(content);
                packagedThemeHashes.put(themeName, jsonContent);
            }
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(TaskRunDevBundleBuild.class);
    }

    private void runFrontendBuildTool(String toolName, String executable,
            Map<String, String> environment, String... params)
            throws ExecutionFailedException {
        Logger logger = getLogger();

        FrontendToolsSettings settings = new FrontendToolsSettings(
                options.getNpmFolder().getAbsolutePath(),
                () -> FrontendUtils.getVaadinHomeDirectory().getAbsolutePath());
        settings.setNodeDownloadRoot(options.getNodeDownloadRoot());
        settings.setForceAlternativeNode(options.isRequireHomeNodeExec());
        settings.setUseGlobalPnpm(options.isUseGlobalPnpm());
        settings.setAutoUpdate(options.isNodeAutoUpdate());
        settings.setNodeVersion(options.getNodeVersion());
        FrontendTools frontendTools = new FrontendTools(settings);

        File buildExecutable = new File(options.getNpmFolder(),
                "node_modules/" + executable);
        if (!buildExecutable.isFile()) {
            throw new IllegalStateException(String.format(
                    "Unable to locate %s executable by path '%s'. Double"
                            + " check that the plugin is executed correctly",
                    toolName, buildExecutable.getAbsolutePath()));
        }

        String nodePath;
        if (options.isRequireHomeNodeExec()) {
            nodePath = frontendTools.forceAlternativeNodeExecutable();
        } else {
            nodePath = frontendTools.getNodeExecutable();
        }

        List<String> command = new ArrayList<>();
        command.add(nodePath);
        command.add(buildExecutable.getAbsolutePath());
        command.addAll(Arrays.asList(params));

        String commandString = command.stream()
                .collect(Collectors.joining(" "));

        ProcessBuilder builder = FrontendUtils.createProcessBuilder(command);
        builder.environment().put("devBundle", "true");
        builder.environment().put("NODE_ENV", "development");

        Process process = null;
        try {
            builder.directory(options.getNpmFolder());
            builder.redirectInput(ProcessBuilder.Redirect.PIPE);
            builder.redirectErrorStream(true);

            process = builder.start();

            // This will allow to destroy the process which does IO regardless
            // whether it's executed in the same thread or another (may be
            // daemon) thread
            Runtime.getRuntime()
                    .addShutdownHook(new Thread(process::destroyForcibly));

            logger.debug("Output of `{}`:", commandString);
            StringBuilder toolOutput = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(
                    new InputStreamReader(process.getInputStream(),
                            StandardCharsets.UTF_8))) {
                String stdoutLine;
                while ((stdoutLine = reader.readLine()) != null) {
                    logger.debug(stdoutLine);
                    toolOutput.append(stdoutLine)
                            .append(System.lineSeparator());
                }
            }

            int errorCode = process.waitFor();

            if (errorCode != 0) {
                logger.error("Command `{}` failed:\n{}", commandString,
                        toolOutput);
                throw new ExecutionFailedException(
                        SharedUtil.capitalize(toolName)
                                + " build exited with a non zero status");
            } else {
                logger.info("Development frontend bundle built");
            }
        } catch (InterruptedException | IOException e) {
            logger.error("Error when running `{}`", commandString, e);
            if (e instanceof InterruptedException) {
                // Restore interrupted state
                Thread.currentThread().interrupt();
            }
            throw new ExecutionFailedException(
                    "Command '" + commandString + "' failed to finish", e);
        } finally {
            if (process != null) {
                process.destroyForcibly();
            }
        }
    }

    /**
     * Removes Vaadin managed dependencies found under "vaadin:dependencies"
     * from "dependencies" block if the package name and version match. Needed
     * for cases when Vaadin provided dependencies are not a part of the
     * platform anymore, so these dependencies are not treated as a developer
     * provided dependencies and, thus, a new bundle generation is not triggered
     * erroneously.
     * <p>
     * Examples: polymer iron-list dependencies or Vaadin web components with a
     * deprecated "vaadin-" prefix that are not a part of platform 24.0 anymore.
     *
     * @param packageJson
     *            content of the package.json content red from a file
     */
    private static void cleanOldPlatformDependencies(JsonObject packageJson) {
        if (packageJson == null
                || !hasFrameworkDependencyObjects(packageJson)) {
            return;
        }

        JsonObject dependencies = packageJson
                .getObject(NodeUpdater.DEPENDENCIES);
        JsonObject vaadinDependencies = packageJson
                .getObject(NodeUpdater.VAADIN_DEP_KEY)
                .getObject(NodeUpdater.DEPENDENCIES);

        for (String vaadinDependency : vaadinDependencies.keys()) {
            String version = vaadinDependencies.getString(vaadinDependency);
            if (dependencies.hasKey(vaadinDependency) && version
                    .equals(dependencies.getString(vaadinDependency))) {
                dependencies.remove(vaadinDependency);
                getLogger().debug(
                        "Old Vaadin provided dependency '{}':'{}' has been removed from package.json",
                        vaadinDependency, version);
            }
        }
    }

    private static boolean hasFrameworkDependencyObjects(
            JsonObject packageJson) {
        return packageJson.hasKey(NodeUpdater.VAADIN_DEP_KEY)
                && packageJson.getObject(NodeUpdater.VAADIN_DEP_KEY)
                        .hasKey(NodeUpdater.DEPENDENCIES)
                && packageJson.hasKey(NodeUpdater.DEPENDENCIES);
    }

    private static String getTag(
            WebComponentExporterFactory<? extends Component> factory) {
        WebComponentExporterTagExtractor exporterTagExtractor = new WebComponentExporterTagExtractor();
        return exporterTagExtractor.apply(factory);
    }

    private static boolean compareObjects(Collection<String> missedKeys,
            JsonObject jsonObjectFromBundle, JsonObject projectJsonObject) {
        boolean allEntriesFound = true;

        for (String projectEntryKey : projectJsonObject.keys()) {
            JsonValue projectEntry = projectJsonObject.get(projectEntryKey);
            // ignore parent theme, because having a parent theme doesn't
            // need a new bundle per se
            if (projectEntry.getType() == JsonType.STRING
                    && "parent".equals(projectEntryKey)) {
                continue;
            }
            boolean entryFound = false;
            for (String bundleEntryKey : jsonObjectFromBundle.keys()) {
                JsonValue bundleEntry = jsonObjectFromBundle
                        .get(bundleEntryKey);
                if (bundleEntry.getType() == projectEntry.getType()
                        && objectIncludesEntry(bundleEntry, projectEntry,
                                missedKeys)) {
                    entryFound = true;
                    break;
                }
            }
            if (!entryFound) {
                missedKeys.add(projectEntryKey);
            }
            allEntriesFound = allEntriesFound && entryFound;
        }
        return allEntriesFound;
    }

    private static boolean compareArrays(Collection<String> missedKeys,
            JsonArray jsonArrayFromBundle, JsonArray projectJsonArray) {
        boolean allEntriesFound = true;

        for (int projectArrayIndex = 0; projectArrayIndex < projectJsonArray
                .length(); projectArrayIndex++) {
            JsonValue projectArrayEntry = projectJsonArray
                    .get(projectArrayIndex);
            boolean entryFound = false;
            for (int bundleArrayIndex = 0; bundleArrayIndex < jsonArrayFromBundle
                    .length(); bundleArrayIndex++) {
                JsonValue bundleArrayEntry = jsonArrayFromBundle
                        .get(bundleArrayIndex);
                if (bundleArrayEntry.getType() == projectArrayEntry.getType()
                        && objectIncludesEntry(bundleArrayEntry,
                                projectArrayEntry, missedKeys)) {
                    entryFound = true;
                    break;
                }
            }
            if (!entryFound) {
                missedKeys.add(projectArrayEntry.toJson());
            }
            allEntriesFound = allEntriesFound && entryFound;
        }
        return allEntriesFound;
    }

    private void addReadme() {
        File devBundleFolder = new File(options.getNpmFolder(),
                Constants.DEV_BUNDLE_LOCATION);
        assert devBundleFolder.exists();

        try {
            File readme = new File(devBundleFolder, "README.md");
            boolean created = readme.createNewFile();
            if (created) {
                FileUtils.writeStringToFile(readme, README,
                        StandardCharsets.UTF_8);
            } else {
                getLogger().warn(README_NOT_CREATED);
            }
        } catch (Exception e) {
            getLogger().error(README_NOT_CREATED, e);
        }
    }
}
