/**
 * Copyright (C) 2024 Vaadin Ltd
 *
 * This program is available under Vaadin Commercial License and Service Terms.
 *
 * See {@literal <https://vaadin.com/commercial-license-and-service-terms>}  for the full
 * license.
 */
package com.vaadin.flow.internal.nodefeature;

import com.vaadin.flow.component.page.LoadingIndicatorConfiguration;
import com.vaadin.flow.internal.StateNode;

/**
 * Map for storing configuration for the loading indicator.
 * <p>
 * For internal use only. May be renamed or removed in a future release.
 *
 * @author Vaadin Ltd
 * @since 1.0
 */
public class LoadingIndicatorConfigurationMap extends NodeMap
        implements LoadingIndicatorConfiguration {
    public static final String FIRST_DELAY_KEY = "first";
    public static final int FIRST_DELAY_DEFAULT = 450;
    public static final String SECOND_DELAY_KEY = "second";
    public static final int SECOND_DELAY_DEFAULT = 1500;
    public static final String THIRD_DELAY_KEY = "third";
    public static final int THIRD_DELAY_DEFAULT = 5000;
    public static final String DEFAULT_THEME_APPLIED_KEY = "theme";
    public static final boolean DEFAULT_THEME_APPLIED_DEFAULT = true;

    /**
     * Creates a new map for the given node.
     *
     * @param node
     *            the node that the map belongs to
     */
    public LoadingIndicatorConfigurationMap(StateNode node) {
        super(node);
    }

    @Override
    public void setFirstDelay(int firstDelay) {
        put(FIRST_DELAY_KEY, firstDelay);
    }

    @Override
    public int getFirstDelay() {
        return getOrDefault(FIRST_DELAY_KEY, FIRST_DELAY_DEFAULT);
    }

    @Override
    public void setSecondDelay(int secondDelay) {
        put(SECOND_DELAY_KEY, secondDelay);
    }

    @Override
    public int getSecondDelay() {
        return getOrDefault(SECOND_DELAY_KEY, SECOND_DELAY_DEFAULT);
    }

    @Override
    public void setThirdDelay(int thirdDelay) {
        put(THIRD_DELAY_KEY, thirdDelay);
    }

    @Override
    public int getThirdDelay() {
        return getOrDefault(THIRD_DELAY_KEY, THIRD_DELAY_DEFAULT);
    }

    @Override
    public boolean isApplyDefaultTheme() {
        return getOrDefault(DEFAULT_THEME_APPLIED_KEY,
                DEFAULT_THEME_APPLIED_DEFAULT);
    }

    @Override
    public void setApplyDefaultTheme(boolean applyDefaultTheme) {
        put(DEFAULT_THEME_APPLIED_KEY, applyDefaultTheme);
    }
}
