package com.vaadin.copilot.plugins.propertypanel;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.vaadin.copilot.JavaReflectionUtil;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.internal.BeanUtil;

/**
 * Collects and caches default values of a component by instantiating a new
 * instance from {@link Class} and by calling getter methods
 */
public class FlowComponentDefaultValueCache {
    private static FlowComponentDefaultValueCache instance;

    public static FlowComponentDefaultValueCache getInstance() {
        if (instance == null) {
            instance = new FlowComponentDefaultValueCache();
        }
        return instance;
    }

    private FlowComponentDefaultValueCache() {
    }

    private final Map<String, Map<String, Object>> cache = new HashMap<>();

    /**
     * Retrieves the default value of a specified property for a given component.
     * <p>
     * If the default values for the component's class have not been cached yet,
     * this method invokes {@code collectDefaultProperties} to compute and cache
     * them. Then, it attempts to retrieve the default value for the specified
     * property name from the cache.
     * </p>
     *
     * @param component
     *            the UI component whose property's default value is being queried
     * @param propertyName
     *            the name of the property to look up
     * @return an {@link Optional} containing the default value of the property if
     *         available, or an empty {@link Optional} if the property is not found
     * @throws IntrospectionException
     *             if an error occurs while introspecting the component to collect
     *             its default properties
     */
    public Optional<Object> getDefaultPropertyValue(Component component, String propertyName)
            throws IntrospectionException {
        String qualifiedClassName = component.getClass().getName();
        if (!cache.containsKey(qualifiedClassName)) {
            Map<String, Object> defaultProperties = collectDefaultProperties(component);
            cache.put(qualifiedClassName, defaultProperties);
        }
        Map<String, Object> defaultProperties = cache.get(qualifiedClassName);
        return Optional.ofNullable(defaultProperties.get(propertyName));
    }

    private static Map<String, Object> collectDefaultProperties(Component component) throws IntrospectionException {
        Class<? extends Component> clazz = component.getClass();
        Component initialInstance;
        try {
            initialInstance = JavaReflectionUtil.instantiateClass(clazz);
        } catch (Exception e) {
            throw new DefaultValueNotAccessibleException(e.getMessage());
        }

        List<PropertyDescriptor> beanPropertyDescriptors = BeanUtil.getBeanPropertyDescriptors(clazz);
        Map<String, Object> defaultProperties = new HashMap<>();
        for (PropertyDescriptor beanPropertyDescriptor : beanPropertyDescriptors) {
            Method readMethod = beanPropertyDescriptor.getReadMethod();
            Optional<Object> methodInvocationResult = JavaReflectionUtil.getMethodInvocationResult(initialInstance,
                    readMethod);
            methodInvocationResult.ifPresent(o -> defaultProperties.put(beanPropertyDescriptor.getName(), o));
        }
        return defaultProperties;
    }
}
