package com.vaadin.copilot;

import java.io.File;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentInfoFinder;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaFileSourceProvider;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

record FlowComponentReference(int nodeId, int uiId) {
};

public class LitTemplateHandler extends CopilotCommand {
    private ComponentSourceFinder sourceFinder;

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        if (command.equals("get-lit-template-name")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put("reqId", data.get("reqId").asText());
            try {
                this.sourceFinder = new ComponentSourceFinder(getVaadinSession());
                ComponentTypeAndSourceLocation litTemplate = sourceFinder
                        .findTypeAndSourceLocation(data.get("litTemplate"));
                var jsModule = litTemplate.component().getClass().getAnnotation(JsModule.class);
                var litTemplateFile = new File(projectFileManager.getFrontendFolder(),
                        jsModule.value().replace("/", File.separator));
                responseData.put("litTemplateFilename", litTemplateFile.getCanonicalPath());
                devToolsInterface.send(command + "-resp", responseData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                        "Unable to get lit template name", e);
            }
        } else if (command.equals("import-lit-template")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put("reqId", data.get("reqId").asText());
            try {
                var componentDefinitions = JavaComponent.componentsFromJson(data.withArray("componentDefinitions"));
                ComponentTypeAndSourceLocation litTemplate = sourceFinder
                        .findTypeAndSourceLocation(data.get("litTemplate"));
                JavaFileSourceProvider javaFileSourceProvider = new JavaFileSourceProvider(true);
                ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(javaFileSourceProvider, litTemplate);
                ComponentInfo litTemplateInfo = componentInfoFinder.find();
                var customComponentInfo = litTemplateInfo.customComponentInfo().orElseThrow(
                        () -> new IllegalArgumentException("The given lit template was not found in the project"));
                File litTemplateClass = projectFileManager.getFileForClass(customComponentInfo.componentClass());
                var litTemplateSource = javaFileSourceProvider.getJavaSource(litTemplateClass);
                JavaRewriter javaRewriter = new JavaRewriter();
                javaRewriter.importLitTemplate(litTemplateSource, componentDefinitions);
                String result = litTemplateSource.getResult();
                ProjectFileManager.get().writeFile(litTemplateSource.getFile(), "Convert LitTemplate to Java", result);

                var litTemplateFile = new File(data.get("litTemplateFilename").asText());
                ProjectFileManager.get().deleteFile(litTemplateFile);
                devToolsInterface.send(command + "-resp", responseData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                        "Unable to get lit template name", e);
            }
        }
        return false;
    }
}
